#include "check_device_signature.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/signer.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/check_device_signature_result.h>

#include <passport/infra/libs/cpp/utils/crypto/rsa.h>

namespace NPassport::NBb {
    TCheckDeviceSignatureProcessor::TCheckDeviceSignatureProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_OK = "OK";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_NONCE_EXPIRED = "NONCE.EXPIRED";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_NONCE_NO_KEY = "NONCE.NO_KEY";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_NONCE_INVALID = "NONCE.INVALID";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_DEVICE_ID_INVALID = "DEVICE_ID.INVALID";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_PUBLIC_KEY_UNSUPPORTED_VERSION = "PUBLIC_KEY.UNSUPPORTED_VERSION";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_PUBLIC_KEY_NOT_FOUND = "PUBLIC_KEY.NOT_FOUND";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_PUBLIC_KEY_INVALID = "PUBLIC_KEY.INVALID";
    const TString TCheckDeviceSignatureProcessor::CDS_STATUS_SIGNATURE_INVALID = "SIGNATURE.INVALID";

    static const TString NONCE_IS_INVALID = "Nonce is invalid";

    std::unique_ptr<TCheckDeviceSignatureResult> TCheckDeviceSignatureProcessor::CdsCheckNonce(
        const TSigner& signer,
        const TString& nonce,
        const TString& nonceSignSpace,
        const TString& deviceId,
        time_t now) {
        TString details;

        TSigner::TCheckResult result = signer.CheckSign(nonce, nonceSignSpace, now);
        switch (result.Status) {
            case TSigner::EStatus::Expired:
                details = NUtils::CreateStr("Nonce expires at ", result.Expires, ". now is ", now);
                TLog::Debug() << "checkDeviceSignature: " << details;
                return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_NONCE_EXPIRED, details);
            case TSigner::EStatus::NoKey:
                details = NUtils::CreateStr("Nonce is not expired, but key_id==", result.KeyId, " was not found");
                TLog::Error() << "checkDeviceSignature: " << details;
                return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_NONCE_NO_KEY, details);
            case TSigner::EStatus::Invalid:
                TLog::Debug() << "checkDeviceSignature: " << NONCE_IS_INVALID;
                return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_NONCE_INVALID, NONCE_IS_INVALID);
            case TSigner::EStatus::Ok:
                break;
        }

        if (deviceId != result.Value) {
            details = NUtils::CreateStr("Device in nonce: '", result.Value, "' but in request: '", deviceId, "'");
            TLog::Debug() << "checkDeviceSignature: " << details;
            return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_DEVICE_ID_INVALID, details);
        }

        return nullptr;
    }

    TGrantsChecker TCheckDeviceSignatureProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::CheckDeviceSignature);

        checker.CheckArg(TStrings::NONCE_SIGN_SPACE, [&](const TString& value) {
            return consumer.IsCheckDeviceSignatureAllowed(value);
        });

        checker.CheckNotEmptyArgAllowed(TStrings::PUBLIC_KEY, TBlackboxFlags::CheckDeviceSignatureWithPublicKey);
        checker.CheckNotEmptyArgAllowed(TStrings::TIMESTAMP, TBlackboxFlags::CheckDeviceSignatureWithDebugMode);

        return checker;
    }

    static const TString INVALID_BASE64 = "invalid base64 in signature";

    std::unique_ptr<TCheckDeviceSignatureResult> TCheckDeviceSignatureProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& nonce = TUtils::GetCheckedArg(Request_, TStrings::NONCE);
        const TString& nonceSignSpace = TUtils::GetCheckedArg(Request_, TStrings::NONCE_SIGN_SPACE);
        const TString& deviceId = TUtils::GetCheckedArg(Request_, TStrings::DEVICE_ID);
        const TString& signature = TUtils::GetCheckedArg(Request_, TStrings::SIGNATURE);
        const TString& timestamp = TUtils::GetUIntArg(Request_, TStrings::TIMESTAMP);
        const TString& maxAge = TUtils::GetUIntArg(Request_, TStrings::MAX_AGE);
        TString publicKey = Request_.GetArg(TStrings::PUBLIC_KEY);
        TString version = Request_.GetArg(TStrings::VERSION);

        if (publicKey.empty() != version.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "'" << TStrings::PUBLIC_KEY << "' and '" << TStrings::VERSION << "'"
                << " should be both empty OR both non-empty";
        }

        const TString signatureDecoded = NUtils::Base64ToBin(signature);
        if (signatureDecoded.empty()) {
            TLog::Debug() << "checkDeviceSignature: " << INVALID_BASE64;
            return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_SIGNATURE_INVALID, INVALID_BASE64);
        }

        // check nonce
        {
            time_t now = timestamp.empty() ? time(nullptr) : TUtils::ToTime(timestamp);
            if (maxAge) {
                const ui64 age = IntFromString<ui64, 10>(maxAge);
                if (Blackbox_.DeviceSignatureMaxAge() < age) {
                    throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                        << "Arg '" << TStrings::MAX_AGE << "' has too large value: "
                        << "max allowed: " << Blackbox_.DeviceSignatureMaxAge()
                        << ": " << InvalidValue(age);
                }
                now -= age;
            }
            std::unique_ptr<TCheckDeviceSignatureResult> res = CdsCheckNonce(Blackbox_.Signer(), nonce, nonceSignSpace, deviceId, now);
            if (res) {
                return res;
            }
        }

        // get publicKey
        if (publicKey.empty()) {
            TString query;

            try {
                NDbPool::TBlockingHandle handle(Blackbox_.CentralDb());

                query = NUtils::CreateStr(
                    "SELECT public_key,version FROM device_public_key WHERE device_id='",
                    handle.EscapeQueryParam(deviceId),
                    "'");

                if (!handle.Query(query)->Fetch(publicKey, version)) {
                    const TString details = NUtils::CreateStr("Public key is absent for device_id=", deviceId);
                    TLog::Debug() << "checkDeviceSignature: " << details;
                    return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_PUBLIC_KEY_NOT_FOUND, details);
                }
            } catch (const NDbPool::TException& e) {
                TLog::Debug() << "Failed to get public_key for device_id: <" << query << ">. " << e.what();
                throw TDbpoolError("Failed to get public_key for device_id", e.what());
            }
        }

        // check version
        if ("1" != version) {
            const TString details = NUtils::CreateStr("Supported versions: 1. got: ", version);
            TLog::Debug() << "checkDeviceSignature: " << details;
            return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_PUBLIC_KEY_UNSUPPORTED_VERSION, details);
        }

        // parse public key
        NUtils::TRsaPublicEvp rsa;
        try {
            rsa = NUtils::TRsaPublicEvp::FromPem(publicKey);
        } catch (const std::exception& e) {
            const TString details = NUtils::CreateStr("Public key is invalid: ", e.what());
            TLog::Debug() << "checkDeviceSignature: " << details;
            return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_PUBLIC_KEY_INVALID, details);
        }

        // check signature
        const NUtils::TRsaPublicEvp::TResult check = rsa.VerifyWithSha256(nonce, signatureDecoded);
        if (!check.IsSuccess) {
            TLog::Debug() << "checkDeviceSignature: signature check failed: " << check.Details;
            return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_SIGNATURE_INVALID, TString(check.Details));
        }

        return std::make_unique<TCheckDeviceSignatureResult>(CDS_STATUS_OK, TString());
    }
}
