#include "check_has_plus.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/check_has_plus_result.h>

namespace NPassport::NBb {
    TCheckHasPlusProcessor::TCheckHasPlusProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TCheckHasPlusProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::CheckHasPlus);

        return checker;
    }

    static const TString CHECK_HAS_PLUS_QUERY =
        "SELECT pb.uid,at.type,at.value"
        " FROM phone_bindings as pb, attributes as at"
        " WHERE unix_timestamp(pb.bound)>0 AND pb.flags=0 AND pb.uid=at.uid AND at.type IN(3,160) AND pb.number=";
    static const TString CHECK_HAS_PLUS_QUERY_END = " LIMIT 100";

    std::unique_ptr<TCheckHasPlusResult> TCheckHasPlusProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& number = TUtils::GetCheckedArg(Request_, TStrings::PHONE_NUMBER);
        if (!NUtils::DigitsOnly(number)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid " << TStrings::PHONE_NUMBER << " value: " << InvalidValue(number);
        }

        TString query = NUtils::CreateStr(
            CHECK_HAS_PLUS_QUERY,
            number,
            CHECK_HAS_PLUS_QUERY_END);

        std::vector<NDbPool::TNonBlockingHandle> handles;
        for (NDbPool::TDbPool& db : Blackbox_.ShardsMap().GetShards()) {
            NDbPool::TNonBlockingHandle sqlh(db);
            sqlh.SendQuery(query);
            handles.push_back(std::move(sqlh));
        }

        struct TPossiblePluser {
            bool Disabled = false;
            bool HasPlus = false;
        };
        std::unordered_map<TString, TPossiblePluser> plusers;

        for (auto& sqlh : handles) {
            std::unique_ptr<NDbPool::TResult> res = TUtils::WaitResult(sqlh, "dbpool exception in CheckHasPlus");

            plusers.clear();
            plusers.reserve(2 * res->size());

            for (const NDbPool::TRow& row : res->Table()) {
                const TString& uid = row[0].AsString();
                int type = row[1].AsInt();
                bool attrValue = TDbValue(row[2].AsString()).AsBoolean();

                auto it = plusers.insert({uid, TPossiblePluser()}).first;
                switch (type) {
                    case 3:
                        it->second.Disabled = attrValue;
                        break;
                    case 160:
                        it->second.HasPlus = attrValue;
                        break;
                    default:
                        break;
                }
            }

            for (const auto& [uid, pluser] : plusers) {
                if (!pluser.Disabled && pluser.HasPlus) {
                    return std::make_unique<TCheckHasPlusResult>(true, uid);
                }
            }

            sqlh = NDbPool::TNonBlockingHandle();
        }

        return std::make_unique<TCheckHasPlusResult>(false, TStrings::EMPTY);
    }
}
