#include "check_rfc_totp.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/check_rfc_totp_result.h>
#include <passport/infra/daemons/blackbox/src/totp/checker.h>

#include <passport/infra/libs/cpp/utils/string/format.h>

namespace NPassport::NBb {
    TCheckRfcTotpProcessor::TCheckRfcTotpProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TCheckRfcTotpProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::CheckRfcTotp);

        return checker;
    }

    static const TString CHECK_RFC_TOTP_QUERY_ =
        "SELECT type,value FROM attributes WHERE type IN (155,156) AND uid=";
    static const TString SECRET_NOT_FOUND = "SECRET_NOT_FOUND";
    static const TString VALID_ = "VALID";
    static const TString INVALID_ = "INVALID";
    static const TString ALREADY_USED_ = "ALREADY_USED";

    std::unique_ptr<TCheckRfcTotpResult> TCheckRfcTotpProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        if (!Blackbox_.TotpServiceRfc()) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "Rfc TOTP checker is not initialized";
        }

        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID);
        const TString totp = NUtils::RemoveSpaces(Request_.GetArg(TStrings::TOTP));

        if (totp.empty() || !NUtils::DigitsOnly(totp) || totp.size() != NTotp::TServiceRfc::TOTP_LENGTH) {
            TLog::Debug("CheckRfcTotp: totp looks like invalid: '%s'", totp.c_str());
            return std::make_unique<TCheckRfcTotpResult>(INVALID_);
        }

        std::unique_ptr<NDbPool::TResult> dbResult;
        {
            NDbPool::TBlockingHandle h(Blackbox_.ShardsMap().GetPool(uid));
            dbResult = h.Query(CHECK_RFC_TOTP_QUERY_ + uid);
        }

        TString secret;
        TString lastCheckTime = "0";
        for (const NDbPool::TRow& row : dbResult->Table()) {
            if (row[0].AsString() == TAttr::ACCOUNT_RFC_TOTP_SECRET) {
                secret = row[1].AsString();
            } else if (row[0].AsString() == TAttr::ACCOUNT_RFC_TOTP_CHECK_TIME) {
                lastCheckTime = row[1].AsString();
                if (!NUtils::DigitsOnly(lastCheckTime)) {
                    throw yexception() << "Attribute " << TAttr::ACCOUNT_RFC_TOTP_CHECK_TIME
                                       << " should be integer for uid " << uid;
                }
            }
        }

        if (secret.empty()) {
            return std::make_unique<TCheckRfcTotpResult>(SECRET_NOT_FOUND);
        }
        if (!secret.StartsWith("1:") ||
            (secret = NUtils::Base64ToBin(TStringBuf(secret).Skip(2))).empty()) {
            throw yexception() << "Attribute " << TAttr::ACCOUNT_RFC_TOTP_SECRET
                               << " is malformed for uid " << uid;
        }

        NTotp::TTotpResult res = Blackbox_.TotpServiceRfc()->Check(secret, totp, TUtils::ToTime(lastCheckTime));

        switch (res.Code()) {
            case NTotp::TTotpResult::Valid:
                return std::make_unique<TCheckRfcTotpResult>(VALID_, res.TotpTime());
            case NTotp::TTotpResult::Invalid:
                return std::make_unique<TCheckRfcTotpResult>(INVALID_);
            case NTotp::TTotpResult::AlreadyUsed:
                return std::make_unique<TCheckRfcTotpResult>(ALREADY_USED_);
        }
    }
}
