#include "create_oauth_token.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/oauth/config.h>
#include <passport/infra/daemons/blackbox/src/output/create_oauth_token_result.h>

#include <passport/infra/libs/cpp/auth_core/oauth_token.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TCreateOAuthTokenProcessor::TCreateOAuthTokenProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TCreateOAuthTokenProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::CreateOAuthToken);

        return checker;
    }

    std::unique_ptr<TCreateOAuthTokenResult> TCreateOAuthTokenProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        // mandatory arguments
        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);
        const TString& clientId = TUtils::GetUIntArg(Request_, TStrings::CLIENT_ID, true);
        const TString& expireTime = TUtils::GetUIntArg(Request_, TStrings::EXPIRE_TIME, true);
        const TString& scopesId = Request_.GetArg(TStrings::SCOPES);

        // optional arguments
        const TString& createTime = TUtils::GetUIntArg(Request_, TStrings::CREATE_TIME);
        // get token_id if present, and get authid_time from token_id
        const TString& tokenId = Request_.GetArg(TStrings::TOKEN_ID);
        const TString& xtokenId = TUtils::GetUIntArg(Request_, TStrings::XTOKEN_ID);
        const TString& xtokenShard = Request_.GetArg(TStrings::XTOKEN_SHARD);

        // if no create_time or tokenid_time given, use current time
        long long nowMs = TInstant::Now().MilliSeconds();
        time_t ctime = createTime.empty() ? nowMs / 1000 : TUtils::ToUInt(createTime, TStrings::CREATE_TIME);

        TString tokenIdTime;
        if (!tokenId.empty()) {
            // we have a convention that for stateless tokens token_id is 'create_time_ms + uid'
            // so when given custom token_id we need to ensure that
            // 1) it ends with uid, 2) remaining part is valid timestamp

            if (!tokenId.EndsWith(uid)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "token_id does not end with given uid: " << InvalidValue(tokenId);
            }
            tokenIdTime = tokenId.substr(0, tokenId.size() - uid.size());
            if (!NAuth::TOAuthTokenParser::CheckTokenIdTime(tokenIdTime)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "invalid token_id timestamp: " << InvalidValue(tokenIdTime);
            }
        } else {
            tokenIdTime = IntToString<10>(nowMs);
        }

        NAuth::TOAuthToken token = NAuth::TOAuthTokenParser::CreateToken(
            uid,
            ctime,
            TUtils::ToUInt(expireTime, TStrings::EXPIRE_TIME),
            clientId,
            tokenIdTime);

        NAuth::TOAuthToken::TScopeIds scopes;
        for (const TString& s : NUtils::ToVector(scopesId, ',', 8)) {
            unsigned scopeId = TUtils::ToUInt(s, TStrings::SCOPES);
            scopes.insert(scopeId);
        }
        token.SetScopes(scopes);

        token.SetXtokenId(xtokenId);

        if (!xtokenId.empty() && xtokenShard.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Missing xtoken_shard number for xtoken_id";
        }

        if (!xtokenShard.empty()) {
            unsigned shardNo = TUtils::ToUInt(xtokenShard, TStrings::XTOKEN_SHARD);
            if (shardNo < 1) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "invalid xtoken_shard value: " << InvalidValue(shardNo);
            }
            token.SetXtokenShard(shardNo - 1); // we have 0-based index but parameter is 1-based
        }

        // add optional params
        token.SetDeviceId(Request_.GetArg(TStrings::DEVICE_ID));
        token.SetMeta(Request_.GetArg(TStrings::META));
        token.SetPaymentAuthContextId(Request_.GetArg(TStrings::PAYMENT_AUTH_CONTEXT_ID));
        token.SetPaymentAuthScopeAddendum(Request_.GetArg(TStrings::PAYMENT_AUTH_SCOPE_ADDENDUM));
        const TString& loginId = Request_.GetArg(TStrings::LOGIN_ID);
        if (!loginId.empty()) {
            if (uid == TStrings::ZERO) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "login_id is not allowed for tokens without uid: " << InvalidValue(uid);
            }
            token.SetLoginId(loginId);
        }

        TString errMsg;
        TString tokenStr = Blackbox_.OauthParser().MakeTokenStr(token, errMsg);

        if (tokenStr.empty()) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Couldn't make token string:" << errMsg;
        }

        return std::make_unique<TCreateOAuthTokenResult>(
            tokenStr,
            token.TokenId());
    }
}
