#include "create_pwd_hash.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/password_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/typed_value_result.h>

namespace NPassport::NBb {
    TCreatePwdHashProcessor::TCreatePwdHashProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TCreatePwdHashProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::CreatePwdHash);

        return checker;
    }

    std::unique_ptr<TTypedValueResult> TCreatePwdHashProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& strversion = TUtils::GetUIntArg(Request_, TStrings::API_VERSION);

        // first, check if plaintext password given
        const TString& password = Request_.GetArg(TStrings::PASSWORD);
        if (!password.empty()) {
            if (password.size() > 255) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "too long password.";
            }
            const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID);
            unsigned version = strversion.empty() ? 6 : IntFromString<ui32, 10>(strversion);

            return std::make_unique<TTypedValueResult>(TOutTokens::HASH, Blackbox_.PasswordChecker().MakeHash(password, uid, version));
        }

        // if no plaintext password, try to use md5crypt hash
        const TString& md5crypt = Request_.GetArg(TStrings::MD5CRYPT);
        if (!md5crypt.empty()) {
            if (md5crypt.size() < 25 || !md5crypt.StartsWith("$1$")) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "invalid md5crypt hash format: " << md5crypt;
            }
            const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);
            if (!strversion.empty() && strversion != "6") {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "can not convert md5crypt hash to version " << strversion;
            }

            return std::make_unique<TTypedValueResult>(TOutTokens::HASH, Blackbox_.PasswordChecker().ConvertMD5ToArgon(md5crypt, uid));
        }

        // if no plaintext and no md5crypt, try rawmd5
        const TString& rawMd5 = Request_.GetArg(TStrings::RAWMD5);
        if (!rawMd5.empty()) {
            if (rawMd5.size() != 32) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "invalid rawmd5 hash format: " << rawMd5;
            }
            const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);
            if (!strversion.empty() && strversion != "7") {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "can not convert rawmd5 hash to version " << strversion;
            }

            return std::make_unique<TTypedValueResult>(TOutTokens::HASH, Blackbox_.PasswordChecker().ConvertRawMD5ToArgon(rawMd5, uid));
        }

        throw TBlackboxError(TBlackboxError::EType::InvalidParams)
            << "No password, md5crypt or rawmd5 argument found in request, please specify one.";
    }
}
