#include "createsession.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/strong_pwd_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/misc/session_utils.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/authid_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/create_session_result.h>
#include <passport/infra/daemons/blackbox/src/output/new_session_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/sessguard_chunk.h>

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>

namespace NPassport::NBb {
    TCreateSessionProcessor::TCreateSessionProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : TSessionProcessorBase(impl, request)
    {
    }

    TGrantsChecker TCreateSessionProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckNotEmptyArgAllowed(TStrings::GUARD_HOSTS, TBlackboxFlags::CreateGuard);

        // CreateStressSession mode
        bool isStress = TUtils::GetBoolArg(Request_, TStrings::IS_STRESS) ||
                        Blackbox_.IsStressSessionAllowed();
        if (isStress && consumer.IsAllowed(TBlackboxFlags::CreateStressSession)) {
            return checker; // creating stress session, grants OK
        }

        // default CreateSession mode
        checker.CheckMethodAllowed(TBlackboxMethods::CreateSession);

        return checker;
    }

    std::unique_ptr<TCreateSessionResult> TCreateSessionProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        TUtils::CheckUserIpArg(Request_);

        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);
        UserIp_ = TUtils::GetUserIpArg(Request_);

        // check request arguments, override default values
        int version = 3;
        if (Request_.HasArg(TStrings::API_VERSION)) {
            version = TUtils::ToUInt(Request_.GetArg(TStrings::API_VERSION), TStrings::API_VERSION);
            if (version != 3) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Invalid sessionid version, only ver=3 is supported: " << InvalidValue(version);
            }
        }

        TString ttl = TStrings::ZERO; // default: session
        if (Request_.HasArg(TStrings::TTL)) {
            ttl = TUtils::GetUIntArg(Request_, TStrings::TTL);
            ui64 num = IntFromString<ui64, 10>(ttl);
            if (num > 5) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Ttl has illegal enum value: " << InvalidValue(ttl);
            }
        }

        TString lang = TStrings::ONE; // default: ru
        if (Request_.HasArg(TStrings::LANG)) {
            lang = TUtils::GetUIntArg(Request_, TStrings::LANG);
        }

        const bool isLite = TUtils::GetBoolArg(Request_, TStrings::IS_LITE);
        const bool havePassword = TUtils::GetBoolArg(Request_, TStrings::HAVE_PASSWORD);
        const TString socialId = TUtils::GetUIntArg(Request_, TStrings::SOCIAL_ID);
        const bool isBetatester = TUtils::GetBoolArg(Request_, TStrings::IS_BETATESTER);
        const bool isScholar = TUtils::GetBoolArg(Request_, TStrings::IS_SCHOLAR);
        const bool isYastaff = TUtils::GetBoolArg(Request_, TStrings::IS_YASTAFF);

        const bool isStress = Blackbox_.IsStressSessionAllowed()
                                  ? true // create only stress cookies on stress hosts
                                  : TUtils::GetBoolArg(Request_, TStrings::IS_STRESS);

        const TString keyspace = GetKeyspaceFromArg();

        const TString loginId = Request_.GetArg(TStrings::LOGIN_ID);

        TString createTime = TUtils::GetUIntArg(Request_, TStrings::CREATE_TIME);
        TString authTime = TUtils::GetUIntArg(Request_, TStrings::AUTH_TIME);

        // NOTE: create_time in seconds, auth_time in milliseconds!
        // if no auth_time - take create_time*1000 if present, or current time in ms
        // if no create_time, take current time (in sec)
        long long nowMs = TInstant::Now().MilliSeconds();
        long long nowS = nowMs / 1000;

        if (authTime.empty()) {
            if (createTime.empty()) {
                authTime = IntToString<10>(nowMs);
                createTime = IntToString<10>(nowS);
            } else {
                authTime.assign(createTime).append("000");
            }
        }

        if (createTime.empty()) {
            createTime = IntToString<10>(nowS);
        }

        if (createTime.size() < 10) { // invalid timestamp
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid create_time value: " << InvalidValue(createTime);
        }
        if (authTime.size() < 13) { // invalid milliseconds timestamp
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid auth_time value: " << InvalidValue(authTime);
        }

        TString hostId = Request_.GetArg(TStrings::HOST_ID);
        if (!NUtils::HexDigitsOnly(hostId)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Invalid host_id value: " << InvalidValue(hostId);
        }

        if (hostId.empty()) {
            // we decided that it is safe to take blackbox host_id if no other value given
            hostId = Blackbox_.HostId();
        }

        // Start constructing the cookie
        std::unique_ptr<TCreateSessionResult> result = std::make_unique<TCreateSessionResult>();

        TString authId;
        std::unique_ptr<TAuthIdChunk> authidChunk;
        { // build authid chunk
            NUtils::TIpAddr ip;
            if (!ip.Parse(UserIp_)) {
                // should not get here, since user_ip_ is sanitized above
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Invalid userip value: " << InvalidValue(UserIp_);
            }

            authId = NUtils::CreateStr(authTime, ':', ip.ToBase64String(), ':', hostId);

            authidChunk = std::make_unique<TAuthIdChunk>(authId, authTime, UserIp_, hostId);
        }

        time_t authidTimestamp = TUtils::ToTime(authidChunk->Timestamp) / 1000;

        long passwordCheckDelta = -1; // default: unknown
        if (Request_.HasArg(TStrings::PASSWORD_CHECK_TIME)) {
            time_t pwdCheckTime = TUtils::ToTime(TUtils::GetUIntArg(Request_, TStrings::PASSWORD_CHECK_TIME));
            time_t delta = pwdCheckTime - authidTimestamp;
            // if password check time < authid_time write 0, because password WAS checked
            // this can happen if second value flipped during creating the session
            passwordCheckDelta = delta < 0 ? 0 : delta;
        }

        NAuth::TSession sess = Blackbox_.SessionParser().Create(uid, ttl, keyspace);

        sess.SetVersion(version);
        sess.SetTime(createTime);
        sess.SetHavePassword(havePassword);
        sess.SetPasswordCheckDelta(passwordCheckDelta);
        sess.SetLang(lang);
        sess.SetAuthId(authId);

        time_t logindelta = TUtils::ToUInt(createTime, TStrings::CREATE_TIME) - authidTimestamp;
        if (logindelta > 0) {
            sess.SetLoginDelta(logindelta);
        }

        if (isLite) {
            sess.TurnLite();
        }
        sess.SetBetatester(isBetatester);
        sess.SetScholar(isScholar);
        sess.SetStaff(isYastaff);
        sess.SetStress(isStress);
        sess.SetSocialId(socialId);
        sess.SetLoginId(loginId);

        // set up internal/external flags for yateam
        if (Request_.HasArg(TStrings::YATEAM_AUTH)) {
            bool yateamAuth = TUtils::GetBoolArg(Request_, TStrings::YATEAM_AUTH);
            bool yandexIp = Blackbox_.CheckYandexIp(UserIp_, Request_, uid) == ENetworkKind::Internal;

            if (yateamAuth) {
                if (!yandexIp) { // don't allow internal auth from external IPs
                    throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                        << "Yateam auth requested but user_ip is not Yandex: " << InvalidValue(UserIp_);
                }
                sess.SetInternalAuth(true);
            } else {
                sess.SetExternalAuth(true);

                NUtils::TIpAddr ip;
                ip.Parse(UserIp_);
                sess.SetExtUserIp(ip.ToBase64String());

                // we need to cut off milliseconds from original auth_time
                sess.SetExtAuthTs(authTime.substr(0, authTime.size() - 3));
                sess.SetExtUpdateTs(createTime);
            }
        }

        TString domain;
        domain.reserve(keyspace.size() + 1);

        if (keyspace.find('_') != TString::npos) {
            domain.push_back('.');
        }
        domain.append(keyspace);
        NAuth::TSessionUtils::KeyspaceToDomain(domain);

        result->NewSession = std::make_unique<TNewSessionChunk>(sess, domain, false);

        std::vector<TStringBuf> guardHosts = GetGuardHostsArg();
        if (!guardHosts.empty()) {
            result->NewSessguards = GetSessguardCookies(guardHosts, authId, domain, NUtils::ToUInt(createTime, TStrings::CREATE_TIME));
        }

        result->AuthId = std::move(authidChunk);
        result->DefaultUid = sess.Uid();
        if (TUtils::GetBoolArg(Request_, TStrings::GET_LOGIN_ID)) {
            result->LoginId = TSessionUtils::GetLoginId(sess);
        }

        return result;
    }

    TString TCreateSessionProcessor::GetKeyspaceFromArg() const {
        TString arg = TUtils::GetCheckedArg(Request_, TStrings::KEYSPACE);
        NAuth::TSessionUtils::DomainToKeyspace(arg);

        NAuth::TKeyRing* ring = Blackbox_.SessionSigner().GetRingByName(arg);
        if (!ring) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Invalid keyspace value: " << InvalidValue(arg);
        }

        return ring->KSpace();
    }
}
