#include "decrease_sessionid_lifetime.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/account_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/strong_pwd_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/session_utils.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/authid_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/decrease_sessionid_lifetime_result.h>

namespace NPassport::NBb {
    TDecreaseSessionidLifetimeProcessor::TDecreaseSessionidLifetimeProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : TSessionProcessorBase(impl, request)
    {
    }

    TGrantsChecker TDecreaseSessionidLifetimeProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        // This method does not require grants - it is by design: PASSP-33401
        // It requires ServiceTicket - this check was done earlier

        return checker;
    }

    std::unique_ptr<TDecreaseSessionidLifetimeResult> TDecreaseSessionidLifetimeProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        GetSessionidFromArgs();
        Host_ = TUtils::GetCheckedArg(Request_, TStrings::HOST);
        UserIp_ = TUtils::GetUserIpArg(Request_);

        TCheckResult checkResult = CheckSession();

        std::unique_ptr result = std::make_unique<TDecreaseSessionidLifetimeResult>();

        result->Status = TSessionUtils::StatusName(checkResult.Status);
        result->Comment = std::move(checkResult.Comment);

        if (checkResult.Status == ESessionStatus::VALID) {
            time_t newTs = checkResult.Sess.Now() - checkResult.Sess.GetValidTime() - 1;
            checkResult.Sess.SetTime(ToString(newTs));

            result->Session = checkResult.Sess.AsString();
        } else if (checkResult.Status == ESessionStatus::NEED_RESET) {
            result->Session = checkResult.Sess.AsString();
        }

        return result;
    }

    TDecreaseSessionidLifetimeProcessor::TCheckResult TDecreaseSessionidLifetimeProcessor::CheckSession() {
        ESessionStatus status;
        TString comment;
        std::vector<TString> uidsList;
        std::unique_ptr<TAuthIdChunk> authidChunk;

        NAuth::TSession sess = CheckCookieAndRestrictions(status, authidChunk, comment, uidsList);

        auto createResult = [&]() {
            return TCheckResult{
                .Status = status,
                .Comment = std::move(comment),
                .Sess = std::move(sess),
            };
        };

        if (!TSessionUtils::ValidCookieStatus(status)) {
            TLog::Debug() << "DecreaseSessionidLifetime: sessionid is invalid: " << status
                          << ". comment: " << comment;
            return createResult();
        }

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        {
            TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId());
            // just add default aliases to fetcher: it allowes to find account
            TAccountHelper(conv, Request_);
        }

        const TDbIndex availableIdx = fetcher.AddAttr(TAttr::ACCOUNT_IS_AVAILABLE);
        const TDbIndex glogoutIdx = fetcher.AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME);
        const TDbIndex revokeWebSessionsIdx = fetcher.AddAttr(TAttr::REVOKER_WEB_SESSIONS);
        const TDbIndex changeReasonIdx = fetcher.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON);
        const TDbIndex createRequiredIdx = fetcher.AddAttr(TAttr::PASSWORD_CREATING_REQUIRED);

        TStrongPwdHelper strongPwd(fetcher);

        fetcher.FetchByUid(sess.Uid());

        const TDbProfile* profile = fetcher.NextProfile();

        if (nullptr == profile) {
            comment = "user not found";
            status = ESessionStatus::INVALID;
            TLog::Debug() << "DecreaseSessionidLifetime: cookie <" << LogableSessionId_
                          << "...> of age=" << sess.Age()
                          << ", uid=" << sess.Uid()
                          << " is OK but account is not found in the database: "
                          << (sess.Age() <= 10 ? "just created" : "deleted") << "?";
            return createResult();
        }

        const bool available = profile->Get(availableIdx)->AsBoolean();
        const time_t glogout = profile->Get(glogoutIdx)->AsTime();
        const time_t revokeWebSessions = profile->Get(revokeWebSessionsIdx)->AsTime();
        const TInstant domainGlogout = profile->PddDomItem().Glogout();
        const bool changeReason = profile->Get(changeReasonIdx)->AsBoolean();
        const bool createRequired = profile->Get(createRequiredIdx)->AsBoolean();

        bool passwdExpired = strongPwd.PasswdExpired(profile, Blackbox_.StrongPwdExpireTime());
        bool haveExternalAuth = false;

        status = CheckUserStatus(sess,
                                 sess.DefaultIdx(),
                                 status,
                                 available,
                                 glogout,
                                 revokeWebSessions,
                                 domainGlogout.TimeT(),
                                 changeReason,
                                 createRequired,
                                 passwdExpired,
                                 comment,
                                 haveExternalAuth);

        // status possibly changed, check again
        if (!TSessionUtils::ValidCookieStatus(status)) {
            TLog::Debug() << "DecreaseSessionidLifetime: sessionid is invalid by attrs: " << status
                          << ". comment: " << comment;
            return createResult();
        }

        if (IsSessionKilled(sess, authidChunk->Id)) {
            comment = "session was glogouted";
            status = ESessionStatus::INVALID;
            TLog::Debug() << "DecreaseSessionidLifetime: session was glogouted";
            return createResult();
        }

        return createResult();
    }
}
