#include "deletion_operations.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/table_result.h>

namespace NPassport::NBb {
    TDeletionOperationsProcessor::TDeletionOperationsProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TDeletionOperationsProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::DeletionOperations);

        return checker;
    }

    static const TString GET_DELETION_OPERATION_COUNT_QUERY = "SELECT COUNT(uid) FROM account_deletion_operations WHERE ";
    static const TString GET_DELETION_OPERATION_QUERY = "SELECT uid FROM account_deletion_operations WHERE ";
    static const TTableResult::TStringVector DELETION_OPERATION_COLUMNS = {"uid"};

    std::unique_ptr<TTableResult> TDeletionOperationsProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& strDeletedAfter = TUtils::GetUIntArg(Request_, TStrings::DELETED_AFTER, true);
        const TString& strDeletedBefore = TUtils::GetUIntArg(Request_, TStrings::DELETED_BEFORE, true);
        const TString& strChunkCount = TUtils::GetUIntArg(Request_, TStrings::CHUNK_COUNT);
        const TString& strChunkNo = TUtils::GetUIntArg(Request_, TStrings::CHUNK_NO);

        unsigned chunkCount = strChunkCount.empty() ? 1 : IntFromString<ui32, 10>(strChunkCount);
        unsigned chunkNo = strChunkNo.empty() ? 0 : IntFromString<ui32, 10>(strChunkNo);

        if (chunkNo >= chunkCount) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "chunk_no should be less than chunk_count";
        }

        const TString whereCondition = NUtils::CreateStr(
            strDeletedAfter, "<UNIX_TIMESTAMP(started_at) AND UNIX_TIMESTAMP(started_at)<=", strDeletedBefore);

        // First, get total counts for each shard
        const TString countQuery = NUtils::CreateStr(GET_DELETION_OPERATION_COUNT_QUERY, whereCondition);

        std::vector<NDbPool::TNonBlockingHandle> handles;
        for (NDbPool::TDbPool& db : Blackbox_.ShardsMap().GetShards()) {
            NDbPool::TNonBlockingHandle sqlh(db);
            sqlh.SendQuery(countQuery);
            handles.push_back(std::move(sqlh));
        }

        for (auto& sqlh : handles) {
            std::unique_ptr<NDbPool::TResult> r = TUtils::WaitResult(sqlh, "dbpool exception in deletionOperations");

            int totalCount;
            if (!r->Fetch(totalCount)) {
                throw TBlackboxError(TBlackboxError::EType::Unknown)
                    << "Couldn't fetch deletion operations count";
            }

            if (totalCount > 0) {
                int limit = (totalCount - 1) / chunkCount + 1; // get chunk size (rounded up)
                int offset = limit * chunkNo;

                TString query = NUtils::CreateStr(
                    GET_DELETION_OPERATION_QUERY, whereCondition, " LIMIT ", limit, " OFFSET ", offset);
                sqlh.SendQuery(query);
            } else {
                sqlh = NDbPool::TNonBlockingHandle(); // delete handle if no uids there
            }
        }

        std::unique_ptr<TTableResult> result = std::make_unique<TTableResult>(
            TOutTokens::DELETION_OPERATIONS, DELETION_OPERATION_COLUMNS);

        for (auto& sqlh : handles) {
            if (!sqlh) {
                continue;
            }

            std::unique_ptr<NDbPool::TResult> r = TUtils::WaitResult(sqlh, "dbpool exception in deletionOperations");

            for (const NDbPool::TRow& row : r->Table()) {
                result->Rows.push_back({row[0].AsString()});
            }
        }

        return result;
    }
}
