#include "edit_totp.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/passport_wrapper.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/edit_totp_result.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_encryptor.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_profile.h>

#include <passport/infra/libs/cpp/utils/crypto/hash.h>

namespace NPassport::NBb {
    static const TString CHECK_FAILED = "Totp check failed: password and pin do not match the secret";
    static const TString TOTP_CHECKER_EXCEPTION = "TOTP checker exception checking password: ";
    static const TString FAILED_TO_ENCODE_SECRET = "Failed to encode secret";
    static const TString FAILED_TO_ENCODE_JUNK_SECRET = "Failed to encode junk secret";
    static const TString ADD_FAILED = "Failed to add TOTP secret: maximum reached";

    static void LogError(const TString& msg, const TString& uid) {
        TLog::Debug("%s. uid: %s", msg.c_str(), uid.c_str());
    }

    TEditTotpProcessor::TEditTotpProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TEditTotpProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::EditTotp);

        return checker;
    }

    std::unique_ptr<TEditTotpResult> TEditTotpProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        if (!Blackbox_.TotpEncryptor()) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "TOTP encryption is not initialized, unable to encrypt secret";
        }

        const TString& op = TUtils::GetCheckedArg(Request_, TStrings::OP);
        const TString& uidStr = TUtils::GetUIntArg(Request_, TStrings::UID, true);
        ui64 uidInt = TUtils::ToUInt(uidStr, TStrings::UID);
        ui64 secretId = TUtils::ToUInt(TUtils::GetUIntArg(Request_, TStrings::SECRET_ID, true), TStrings::SECRET_ID);

        std::unique_ptr<TEditTotpResult> result = std::make_unique<TEditTotpResult>();

        if (op == TStrings::CREATE) {
            const TString& totpPasswd = TUtils::GetCheckedArg(Request_, TStrings::PASSWORD);
            TString secret = NUtils::Base64url2bin(
                TUtils::GetCheckedArg(Request_, TStrings::TOTP_SECRET_ARG_NAME));
            const TString& pin = TUtils::GetCheckedArg(Request_, TStrings::PIN);

            TTotpProfile profile(uidInt, pin);
            AddSecretToResult(profile, *result, secret, secretId, uidStr, totpPasswd);
        } else if (op == TStrings::ADD) {
            const TString& totpPasswd = TUtils::GetCheckedArg(Request_, TStrings::PASSWORD);
            TString secret = NUtils::Base64url2bin(
                TUtils::GetCheckedArg(Request_, TStrings::TOTP_SECRET_ARG_NAME));

            TTotpProfile profile = ReadTotpProfileFromDb(Blackbox_, uidStr);
            if (!profile.IsInited()) {
                result->SetError("Failed to decode TOTP secret");
                return result;
            }
            AddSecretToResult(profile, *result, secret, secretId, uidStr, totpPasswd);
        } else if (op == TStrings::DELETE) {
            TTotpProfile profile = ReadTotpProfileFromDb(Blackbox_, uidStr);
            if (!profile.IsInited()) {
                result->SetError("Failed to decode TOTP secret");
                return result;
            }

            if (!profile.DropSecret(secretId)) {
                LogError(profile.LastError(), uidStr);
                result->SetError(profile.LastError());
                return result;
            }

            TString secretValue = Blackbox_.TotpEncryptor()->Encrypt(profile);
            result->SetSecretValue(secretValue, TStrings::EMPTY);
        } else if (op == TStrings::REPLACE) {
            const TString& totpPasswd = TUtils::GetCheckedArg(Request_, TStrings::PASSWORD);
            TString secret = NUtils::Base64url2bin(
                TUtils::GetCheckedArg(Request_, TStrings::TOTP_SECRET_ARG_NAME));
            ui64 oldSecretId = TUtils::ToUInt(TUtils::GetCheckedArg(Request_, TStrings::OLD_SECRET_ID),
                                              TStrings::OLD_SECRET_ID);

            TTotpProfile profile = ReadTotpProfileFromDb(Blackbox_, uidStr);
            if (!profile.IsInited()) {
                result->SetError("Failed to decode TOTP secret");
                return result;
            }

            if (!profile.DropSecret(oldSecretId)) {
                LogError(profile.LastError(), uidStr);
                result->SetError(profile.LastError());
                return result;
            }

            AddSecretToResult(profile, *result, secret, secretId, uidStr, totpPasswd);
        } else {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Invalid 'op' argument: " << InvalidValue(op);
        }

        return result;
    }

    void TEditTotpProcessor::AddSecretToResult(TTotpProfile& profile, TEditTotpResult& result, const TString& secret, ui64 secretId, const TString& uidStr, const TString& totpPasswd) const {
        NTotp::TTotpResult r;
        try {
            r = Blackbox_.TotpService().Check(NUtils::TCrypto::Sha256(profile.Pin() + secret), totpPasswd, 0);
        } catch (const std::exception& e) {
            TString msg;
            msg.reserve(128);
            msg.assign(TOTP_CHECKER_EXCEPTION).append(e.what());
            LogError(msg, uidStr);
            result.SetError(msg);
        }

        if (!r.Succeeded()) {
            if (result.Error() == TStrings::OK) { // if no error from previous check, set error
                result.SetError(CHECK_FAILED);
            }

            // we checked the password and check failed
            // so let's store it as junk_secret for future diagnostics
            try {
                TTotpProfile junkProfile = ReadTotpProfileFromDb(Blackbox_, uidStr, true);
                if (!junkProfile.IsInited() || profile.Pin() != junkProfile.Pin()) {
                    // we have pin changed, decided to drop whole junk secret
                    // if no profile found, create new anyway
                    junkProfile = TTotpProfile(profile.Uid(), profile.Pin());
                } else {
                    // first, check that we don't add duplicate id to junk_secret, drop copy if any
                    junkProfile.DropSecret(secretId);

                    // we need to check for junk secret overflow, delete oldest secret to free some space
                    while (junkProfile.SecretCount() >= Blackbox_.MultisecretLimit()) {
                        junkProfile.DropSecret(junkProfile.OldestSecretId());
                    }
                }
                AddJunkSecret(junkProfile, result, secret, secretId, uidStr);
            } catch (const std::exception& e) {
                TTotpProfile junkProfile = TTotpProfile(profile.Uid(), profile.Pin());
                AddJunkSecret(junkProfile, result, secret, secretId, uidStr);
            }

            return;
        }

        // need to update last TOTP check time in db
        if (Blackbox_.PassportWrapper() && !Blackbox_.PassportWrapper()->UpdateTotpCheckTime(uidStr, r.TotpTime())) {
            throw TBlackboxError(TBlackboxError::EType::DbException)
                << "Failed to update TOTP check time for uid=" << uidStr;
        }

        if (profile.SecretCount() >= Blackbox_.MultisecretLimit()) {
            result.SetError(ADD_FAILED);
            return;
        }

        if (!profile.AddSecret(secret, secretId, std::time(nullptr))) {
            LogError(profile.LastError(), uidStr);
            result.SetError(profile.LastError());
            return;
        }

        TString secretValue = Blackbox_.TotpEncryptor()->Encrypt(profile);
        if (secretValue.empty()) {
            LogError(FAILED_TO_ENCODE_SECRET, uidStr);
            result.SetError(FAILED_TO_ENCODE_SECRET);
            return;
        }

        result.SetSecretValue(secretValue, IntToString<10>(r.TotpTime()));
    }

    void TEditTotpProcessor::AddJunkSecret(TTotpProfile& junkProfile, TEditTotpResult& result, const TString& secret, ui64 secretId, const TString& uidStr) const {
        // yes, we are paranoid, drop secret if we already tried this id
        junkProfile.DropSecret(secretId);

        if (!junkProfile.AddSecret(secret, secretId, std::time(nullptr))) {
            LogError(junkProfile.LastError(), uidStr);
            return;
        }

        TString value = Blackbox_.TotpEncryptor()->Encrypt(junkProfile);
        if (value.empty()) {
            LogError(FAILED_TO_ENCODE_JUNK_SECRET, uidStr);
            return;
        }

        result.SetJunkSecretValue(value);
    }

    static const TString GET_TOTP_SECRET_QUERY = "SELECT value FROM attributes WHERE type=";
    static const TString GET_TOTP_UID_CONDITION = " AND uid=";

    TTotpProfile TEditTotpProcessor::ReadTotpProfileFromDb(const TBlackboxImpl& blackbox,
                                                           const TString& uid,
                                                           bool junk) {
        TString query = NUtils::CreateStr(
            GET_TOTP_SECRET_QUERY,
            junk ? TAttr::ACCOUNT_TOTP_JUNK_SECRET : TAttr::ACCOUNT_TOTP_SECRET,
            GET_TOTP_UID_CONDITION,
            uid);

        std::unique_ptr<NDbPool::TResult> dbResult;
        try {
            NDbPool::TBlockingHandle sqlh(blackbox.ShardsMap().GetPool(uid));
            dbResult = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception querying shard '%s' : '%s'", query.c_str(), e.what());
            throw TDbpoolError("dbpool exception in prove_key_diag fetch", e.what());
        }

        TString secretValue;
        if (!dbResult->Fetch(secretValue)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "User has no TOTP secret.";
        }
        if (secretValue.empty()) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "User has empty TOTP secret.";
        }

        return blackbox.TotpEncryptor()->Decrypt(TUtils::ToUInt(uid, TStrings::UID), secretValue);
    }
}
