#include "editsession.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/strong_pwd_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/session_utils.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/authid_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/create_session_result.h>
#include <passport/infra/daemons/blackbox/src/output/new_session_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/sessguard_chunk.h>

#include <passport/infra/libs/cpp/auth_core/sessguard_parser.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>

namespace NPassport::NBb {
    TEditSessionProcessor::TEditSessionProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : TSessionProcessorBase(impl, request)
    {
    }

    static const TString NO_GRANTS_FOR_EDITSESSION = "no grants for method=editsession";

    TGrantsChecker TEditSessionProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        if (!consumer.IsAllowed(TBlackboxMethods::CreateSession) && !consumer.IsAllowed(TBlackboxFlags::CreateStressSession)) {
            checker.Add(TString(NO_GRANTS_FOR_EDITSESSION));
        }

        checker.CheckNotEmptyArgAllowed(TStrings::GUARD_HOSTS, TBlackboxFlags::CreateGuard);

        return checker;
    }

    std::unique_ptr<TCreateSessionResult> TEditSessionProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        TUtils::CheckUserIpArg(Request_);

        const TString& operation = TUtils::GetCheckedArg(Request_, TStrings::OP);

        // session_id and host arguments must be present
        GetSessionidFromArgs();
        Host_ = TUtils::GetCheckedArg(Request_, TStrings::HOST);
        UserIp_ = TUtils::GetUserIpArg(Request_);

        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);

        TString comment("OK");
        std::unique_ptr<TAuthIdChunk> authidChunk;
        ESessionStatus status;
        std::vector<TString> uidsList;

        NAuth::TSession sess = CheckCookieAndRestrictions(status, authidChunk, comment, uidsList);

        if (!TSessionUtils::ValidCookieStatus(status)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Sessionid cookie not valid" << (comment == "OK" ? "" : ": " + comment);
        }

        // to edit full production session CreateSession required, for stress CreateStressSession is ok
        if (!consumer.IsAllowed(TBlackboxMethods::CreateSession)) {
            const bool throwAccessDeniedError = true;
            if (sess.IsStress()) {
                if (!consumer.IsAllowed(TBlackboxFlags::CreateStressSession)) {
                    TGrantsChecker(Request_, consumer, throwAccessDeniedError)
                        .Add(TString(NO_GRANTS_FOR_EDITSESSION));
                }
            } else {
                TGrantsChecker(Request_, consumer, throwAccessDeniedError)
                    .Add(TString(NO_GRANTS_FOR_EDITSESSION));
            }
        }

        bool newSessguardAllowed = false;
        TString domain;
        if (sess.Domsuff().find('_') != TString::npos) {
            domain.push_back('.');
        }
        domain.append(sess.Domsuff());
        NAuth::TSessionUtils::KeyspaceToDomain(domain);

        const TString& hostGuardspace = Blackbox_.SessGuardParser().GetGuardSpace(Host_, domain);

        if (hostGuardspace) { // this host supports sessguard
            const bool isEnabled = Blackbox_.GetGuardSpaceOptions(hostGuardspace).IsEnabled(UserIp_);
            const TString& sessguard = Request_.GetArg(TStrings::SESSGUARD);

            if (sessguard) {
                NAuth::TSessGuard guard = CheckSessGuard(sessguard, sess.AuthId(), consumer.GetName());
                if (guard.Status() != NAuth::TSessGuard::VALID) {
                    if (isEnabled) {
                        throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                            << "SessGuard is not valid";
                    }
                } else {
                    // sessguard was valid, creating new sessguard allowed
                    newSessguardAllowed = true;
                }
            } else { // no or empty sesguard
                const TString& requestId = Request_.GetArg(TStrings::REQUEST_ID);
                TLog::Debug() << "SessGuard empty, host=" << Host_ << ","
                              << " request_id=" << (requestId.empty() ? "<unknown>" : requestId)
                              << " peer='" << consumer.GetName() << "'";
                if (isEnabled) {
                    throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                        << "SessGuard is not valid";
                }

                // we are in dry-run mode with empty sessguard, creating new sessguard allowed
                newSessguardAllowed = true;
            }
        }

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();

        // we actually don't need the login value, but we ask it to find the account
        // else we fail if account is not disabled and never glogouted - db result will be empty
        fetcher.AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN);

        const TDbIndex availableIdx = fetcher.AddAttr(TAttr::ACCOUNT_IS_AVAILABLE);
        const TDbIndex glogoutIdx = fetcher.AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME);
        const TDbIndex revokeWebSessionsIdx = fetcher.AddAttr(TAttr::REVOKER_WEB_SESSIONS);
        const TDbIndex changeReasonIdx = fetcher.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON);
        const TDbIndex createRequiredIdx = fetcher.AddAttr(TAttr::PASSWORD_CREATING_REQUIRED);

        TStrongPwdHelper strongPwd(fetcher);

        fetcher.FetchByUids(uidsList);

        bool haveExternalAuth = false;

        // to have flag 'have_external_auth' set correctly,
        // we need first to delete user if we are in op=DELETE, and then update statuses and compute flags
        // or else we may delete the only external user later on

        if (operation == TStrings::DELETE) {
            if (Blackbox_.IsYandexIpCheckEnabled() && !UserIpIsYandex_) { // drop only external auth
                int idx = sess.FindUser(uid);
                if (idx >= 0) {
                    TSessionUtils::DropExternalAuth(sess, idx);
                }
            } else { // drop all users entirely
                sess.RemoveUser(uid);
            }

            const TString& newDefault = TUtils::GetUIntArg(Request_, TStrings::NEWDEFAULT);
            if (!newDefault.empty() && !sess.SetDefaultUser(newDefault)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "New default uid is missing in the session: " << InvalidValue(newDefault);
            }

            // drop cookie ext fields if no external sessions left
            if (!sess.HasExternalAuth()) {
                sess.SetExtUserIp(TStrings::EMPTY);
                sess.SetExtAuthTs(TStrings::EMPTY);
                sess.SetExtUpdateTs(TStrings::EMPTY);
            }
        }

        // check and update all users statuses
        for (unsigned idx = 0; idx < sess.UserCount(); ++idx) {
            const TString& currentUid = sess.Uid(idx);

            const TDbProfile* profile = fetcher.ProfileByUid(currentUid);

            if (nullptr == profile) {
                TLog::Debug() << "BlackBox error: cookie <" << LogableSessionId_
                              << "...> of age=" << sess.Age()
                              << ", uid=" << currentUid
                              << " is OK but account is not found in the database: "
                              << (sess.Age() <= 10 ? "just created" : "deleted") << "?";
                continue;
            }

            const bool available = profile->Get(availableIdx)->AsBoolean();
            const time_t glogout = profile->Get(glogoutIdx)->AsTime();
            const TInstant domainGlogout = profile->PddDomItem().Glogout();
            const time_t revokeWebSessions = profile->Get(revokeWebSessionsIdx)->AsTime();
            const bool changeReason = profile->Get(changeReasonIdx)->AsBoolean();
            const bool createRequired = profile->Get(createRequiredIdx)->AsBoolean();

            bool passwdExpired = strongPwd.PasswdExpired(profile, Blackbox_.StrongPwdExpireTime());

            TString userComment;
            // needs this to update Glogout and other flags in session
            CheckUserStatus(sess,
                            idx,
                            status,
                            available,
                            glogout,
                            revokeWebSessions,
                            domainGlogout.TimeT(),
                            changeReason,
                            createRequired,
                            passwdExpired,
                            userComment,
                            haveExternalAuth);
        }

        // for select operation check if we have this user
        if (operation == TStrings::SELECT) {
            if (!sess.SetDefaultUser(uid)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Can't select user that is missing in the session: " << InvalidValue(uid);
            }
        } else if (operation == TStrings::ADD) {
            bool yateamIntAuth = false;
            bool yateamExtAuth = false;

            if (Request_.HasArg(TStrings::YATEAM_AUTH)) {
                bool yateamAuth = TUtils::GetBoolArg(Request_, TStrings::YATEAM_AUTH);

                if (yateamAuth) {
                    if (!UserIpIsYandex_) { // don't allow internal auth from external IPs
                        throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                            << "Yateam auth requested but user_ip is not Yandex: " << InvalidValue(UserIp_);
                    }
                    yateamIntAuth = true;
                } else { // for external auth check and clean old sessions if needed
                    if (Blackbox_.IsYandexIpCheckEnabled() && (!ExtUserIpMatch_ || ExtSessionExpired_)) {
                        // either we checked the IP and it differs, or all external sessions expired
                        // so kill all existing external auths
                        for (unsigned idx = 0; idx < sess.UserCount();) {
                            if (!TSessionUtils::DropExternalAuth(sess, idx)) {
                                ++idx; // if user not deleted, take next one
                            }
                        }

                        // update per-session fields
                        NUtils::TIpAddr ip;
                        ip.Parse(UserIp_);
                        sess.SetExtUserIp(ip.ToBase64String());
                        ExtUserIpMatch_ = true; // now it matches, we've just set it
                        sess.SetExtAuthTs(TStrings::EMPTY);
                        sess.SetExtUpdateTs(TStrings::EMPTY);
                    }
                    yateamExtAuth = true;
                }
            }

            int idx = sess.FindUser(uid); // take existing user block or add a new one
            if (idx < 0) {
                idx = sess.AddUser(uid); // check for overflow below
            }

            if (sess.UserCount() > Blackbox_.MultisessionUserLimit()) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Can't add user, maximum user limit reached: "
                    << Blackbox_.MultisessionUserLimit();
            }

            // update user fields with given values
            if (yateamIntAuth) {
                sess.SetInternalAuth(true, idx);
                sess.SetGlogouted(false, idx); // clean glogout flag if any
                if (!sess.IsExternalAuth(idx)) {
                    sess.SetExtGlogouted(false, idx); // drop ext_glogout flag if user has no external auth
                }
            } else if (yateamExtAuth) {
                haveExternalAuth = true; // we added valid ext auth
                sess.SetExternalAuth(true, idx);
                sess.SetExtGlogouted(false, idx); // clean ext_glogout flag if any
                if (!sess.IsInternalAuth(idx)) {
                    sess.SetGlogouted(false, idx); // drop glogout flag if user has only external auth flag
                }
            } else {                           // not yateam, no int/ext auth
                sess.SetGlogouted(false, idx); // clean glogout flag if any
                if (!sess.IsExternalAuth(idx)) {
                    sess.SetExtGlogouted(false, idx); // drop ext_glogout flag if user has no external auth
                }
            }

            time_t authidTimestamp = sess.AuthIdTimestamp();

            if (Request_.HasArg(TStrings::PASSWORD_CHECK_TIME)) {
                time_t pwdCheckTime = TUtils::ToTime(TUtils::GetUIntArg(Request_, TStrings::PASSWORD_CHECK_TIME));
                time_t delta = pwdCheckTime - authidTimestamp;
                // if password check time < authid_time write 0, because password WAS checked
                // this shouldn't happen unless local time got significantly different between passport and blackbox
                sess.SetPasswordCheckDelta(delta < 0 ? 0 : delta, idx);
            }

            if (Request_.HasArg(TStrings::IS_LITE)) {
                if (TUtils::GetBoolArg(Request_, TStrings::IS_LITE)) {
                    sess.TurnLite(idx);
                }
            }

            if (Request_.HasArg(TStrings::HAVE_PASSWORD)) {
                sess.SetHavePassword(TUtils::GetBoolArg(Request_, TStrings::HAVE_PASSWORD), idx);
            }

            if (Request_.HasArg(TStrings::IS_YASTAFF)) {
                sess.SetStaff(TUtils::GetBoolArg(Request_, TStrings::IS_YASTAFF), idx);
            }

            if (Request_.HasArg(TStrings::IS_BETATESTER)) {
                sess.SetBetatester(TUtils::GetBoolArg(Request_, TStrings::IS_BETATESTER), idx);
            }

            if (Request_.HasArg(TStrings::IS_SCHOLAR)) {
                sess.SetScholar(TUtils::GetBoolArg(Request_, TStrings::IS_SCHOLAR), idx);
            }

            if (Request_.HasArg(TStrings::LANG)) {
                sess.SetLang(TUtils::GetUIntArg(Request_, TStrings::LANG), idx);
            }

            if (Request_.HasArg(TStrings::SOCIAL_ID)) {
                sess.SetSocialId(TUtils::GetUIntArg(Request_, TStrings::SOCIAL_ID), idx);
            }

            time_t logindelta = time(nullptr) - authidTimestamp;
            if (logindelta > 0) {
                sess.SetLoginDelta(logindelta, idx);
            }

            const TString& newDefault = TUtils::GetUIntArg(Request_, TStrings::NEWDEFAULT);
            if (!newDefault.empty() && !sess.SetDefaultUser(newDefault)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "New default uid is missing in the session: " << InvalidValue(newDefault);
            }
        }

        if (IsSessionKilled(sess, authidChunk->Id)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Sessionid cookie not valid: session logged out";
        }

        std::unique_ptr<TCreateSessionResult> result = std::make_unique<TCreateSessionResult>();
        result->AuthId = std::move(authidChunk);
        if (TUtils::GetBoolArg(Request_, TStrings::GET_LOGIN_ID)) {
            result->LoginId = TSessionUtils::GetLoginId(sess);
        }

        if (!sess.UserCount()) { // last user deleted, need to clean cookies
            result->NewSession = std::make_unique<TNewSessionChunk>(domain);

            if (newSessguardAllowed) { // valid sessguard and asked for new one, clean it too
                std::vector<TStringBuf> guardHosts = GetGuardHostsArg();
                if (!guardHosts.empty()) {
                    result->NewSessguards = GetSessguardCookies(guardHosts, sess.AuthId(), domain, 1, true);
                }
            }
            return result;
        }

        // check if custom create_time given
        TString createTime = TUtils::GetUIntArg(Request_, TStrings::CREATE_TIME);
        if (createTime.empty()) {
            createTime = IntToString<10>(std::time(nullptr));
        } else if (createTime.size() < 10) { // invalid timestamp
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid create_time value: " << InvalidValue(createTime);
        }

        sess.SetTime(createTime);

        if (haveExternalAuth && ExtUserIpMatch_) { // update external auth time too
            sess.SetExtUpdateTs(createTime);
            if (sess.ExtAuthTs().empty()) { // we added new external session
                sess.SetExtAuthTs(createTime);
            }
        }

        result->NewSession = std::make_unique<TNewSessionChunk>(sess, domain, createTime.empty());

        if (newSessguardAllowed) {
            std::vector<TStringBuf> guardHosts = GetGuardHostsArg();
            if (!guardHosts.empty()) {
                result->NewSessguards = GetSessguardCookies(guardHosts, sess.AuthId(), domain, NUtils::ToUInt(createTime, TStrings::CREATE_TIME));
            }
        }

        result->DefaultUid = sess.Uid();

        return result;
    }
}
