#include "email_bindings.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/list_result.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>

namespace NPassport::NBb {
    TEmailBindingsProcessor::TEmailBindingsProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TEmailBindingsProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::EmailBindings);

        return checker;
    }

    static const TString EMAIL_BINDINGS_QUERY_ =
        "SELECT uid FROM email_bindings WHERE address='";

    std::unique_ptr<TListResult> TEmailBindingsProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& email = TUtils::GetCheckedArg(Request_, TStrings::EMAIL);

        TString query;

        std::vector<NDbPool::TNonBlockingHandle> handles;
        for (NDbPool::TDbPool& db : Blackbox_.ShardsMap().GetShards()) {
            NDbPool::TNonBlockingHandle sqlh(db);
            if (query.empty()) {
                query = NUtils::CreateStr(EMAIL_BINDINGS_QUERY_, sqlh.EscapeQueryParam(email), "'");
            }

            sqlh.SendQuery(query);
            handles.push_back(std::move(sqlh));
        }

        TListResult::TStringVector vec;
        for (auto& sqlh : handles) {
            std::unique_ptr<NDbPool::TResult> r = TUtils::WaitResult(sqlh, "dbpool exception in EmailBindings");

            for (const NDbPool::TRow& row : r->Table()) {
                vec.push_back(row[0].AsString());
            }
        }

        std::unique_ptr<TListResult> result = std::make_unique<TListResult>(TOutTokens::UIDS);
        result->TotalCount = vec.size();
        result->Data = std::move(vec);

        return result;
    }
}
