#include "family_info.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/base_result_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/uid_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/family_info_result.h>

#include <passport/infra/libs/cpp/dbpool/exception.h>
#include <passport/infra/libs/cpp/dbpool/result.h>
#include <passport/infra/libs/cpp/tvm/common/public_keys.h>

namespace NPassport::NBb {
    TFamilyInfoProcessor::TFamilyInfoProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TFamilyInfoProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::FamilyInfo);

        checker.CheckArgIsTrue(TStrings::GET_PLACE, TBlackboxFlags::FamilyInfoPlace);

        if (Request_.HasArg(TStrings::GET_MEMBERS_INFO)) {
            TBaseResultHelper::CheckGrants(
                Blackbox_.DbFieldSettings(),
                Blackbox_.AttributeSettings(),
                Request_.HasArg(TStrings::USER_TICKET) ? TConsumer::ERank::HasCred
                                                       : TConsumer::ERank::NoCred,
                checker);
        }

        return checker;
    }

    static const TString STATUS_OK = "OK";
    static const TString STATUS_MISSING_FAMILY = "MISSING_FAMILY";
    static const TString STATUS_INVALID_USER_TICKET = "INVALID_USER_TICKET";
    static const TString STATUS_WRONG_USER = "WRONG_USER";

    std::unique_ptr<TFamilyInfoResult> TFamilyInfoProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& familyId = TUtils::GetCheckedArg(Request_, TStrings::FAMILY_ID);
        const bool getPlace = TUtils::GetBoolArg(Request_, TStrings::GET_PLACE);
        const bool getAllowMoreKids = TUtils::GetBoolArg(Request_, TStrings::GET_ALLOW_MORE_KIDS);

        std::unique_ptr<TFamilyInfoResult> result = std::make_unique<TFamilyInfoResult>();
        result->FamilyId = familyId;

        const EMembersInfo membersInfo = GetMembersInfo(Request_);

        TFetchFamilyParams fetchParams{
            .FamilyId = familyId,
            .GetPlace = getPlace,
            .MembersInfo = membersInfo,
        };
        if (!CheckUserTicket(fetchParams.UidFromCred, *result)) {
            return result;
        }

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId());
        TBaseResultHelper baseResultHelper(conv, Blackbox_, Request_);

        const TDbIndex isChildIdx = fetcher.AddAttr(TAttr::ACCOUNT_IS_CHILD);

        TString adminUid;
        ui16 kidsCount = 0;
        switch (FetchFromDb(fetcher, fetchParams, isChildIdx, adminUid, kidsCount)) {
            case EFetchStatus::Ok:
                break;
            case EFetchStatus::NoResult:
                result->Status.Value = STATUS_MISSING_FAMILY;
                result->Status.Description = NUtils::CreateStr(
                    "Family was not found: ", TString(InvalidValue(familyId)));
                return result;
            case EFetchStatus::VerifyFailed:
                result->Status.Value = STATUS_WRONG_USER;
                result->Status.Description = NUtils::CreateStr(
                    "default_uid from UserTicket == '", *fetchParams.UidFromCred,
                    "' does not belong to family ",
                    TString(InvalidValue(familyId)));
                return result;
        }

        bool success = true;
        if (membersInfo == EMembersInfo::None) {
            BuildSimpleResult(fetcher, getPlace, *result);
        } else {
            success = BuildCommonResult(fetcher,
                                        fetchParams.UidFromCred,
                                        getPlace,
                                        isChildIdx,
                                        baseResultHelper,
                                        *result);
        }

        if (getAllowMoreKids) {
            result->AllowMoreKids = (kidsCount < Blackbox_.FamilyKidsLimit());
        }

        if (success) {
            result->AdminUid = adminUid;
            result->Status.Value = STATUS_OK;
        }

        return result;
    }

    TFamilyInfoProcessor::EFetchStatus TFamilyInfoProcessor::FetchFromDb(TDbFetcher& fetcher,
                                                                         const TFetchFamilyParams& params,
                                                                         const int isChildIdx,
                                                                         TString& adminUid,
                                                                         ui16& kidsCount) {
        // `kid` is account with KIDDISH alias + without PORTAL alias
        fetcher.AddAlias(TAlias::PORTAL_LOGIN);
        fetcher.AddAlias(TAlias::KIDDISH);

        fetcher.FetchAliasesByFamilyId(
            ToInternalFamilyId(params.FamilyId),
            params.GetPlace);

        bool foundAnybody = false;
        bool foundCred = false;
        const TDbProfile* profile = nullptr;

        auto skipUser = [=, &params](const TDbProfile* profile, bool attributesFetched) -> bool {
            switch (params.MembersInfo) {
                case EMembersInfo::None:
                case EMembersInfo::All:
                    return false;

                case EMembersInfo::Kids:
                    return !profile->IsKid();

                case EMembersInfo::Children:
                    return attributesFetched && !profile->Get(isChildIdx)->AsBoolean();

                case EMembersInfo::Adults:
                    return profile->IsKid() || profile->Get(isChildIdx)->AsBoolean();
            }
        };

        kidsCount = 0;
        while ((profile = fetcher.NextProfile())) {
            foundAnybody = true;

            if (params.UidFromCred && profile->Uid() == *params.UidFromCred) {
                foundCred = true;
            }

            if (adminUid.empty()) {
                if (!profile->FamilyInfo()) {
                    TLog::Error() << "FamilyInfoProcessor: no family_info in profile (adminUid): "
                                  << profile->Uid();
                    throw TBlackboxError(TBlackboxError::EType::Unknown)
                        << "Internal error: no family info in fetcher";
                }
                adminUid = profile->FamilyInfo()->AdminUid;
            }

            if (profile->IsKid()) {
                ++kidsCount;
            }

            if (skipUser(profile, false)) {
                // do not fetch attrs for accounts if they are not required
                fetcher.ResetCurrentProfile();
            }
        }

        if (!foundAnybody) {
            return EFetchStatus::NoResult;
        }

        if (params.UidFromCred && !foundCred) {
            // uid from credential must belong to family. otherwise:
            // 1) detect this case
            // 2) skip visiting passportdbshard
            return EFetchStatus::VerifyFailed;
        }

        if (params.MembersInfo == EMembersInfo::None) {
            // we need only data from passportdbcentral
            return EFetchStatus::Ok;
        }

        fetcher.FetchAttrs();

        while ((profile = fetcher.NextProfile())) {
            if (skipUser(profile, true)) {
                fetcher.ResetCurrentProfile();
            }
        }

        return EFetchStatus::Ok;
    }

    TFamilyInfoProcessor::EMembersInfo TFamilyInfoProcessor::GetMembersInfo(const NCommon::TRequest& request) {
        if (!request.HasArg(TStrings::GET_MEMBERS_INFO)) {
            return EMembersInfo::None;
        }

        const TString& value = request.GetArg(TStrings::GET_MEMBERS_INFO);
        if (value == "all") {
            return EMembersInfo::All;
        }

        if (value == "kids") {
            return EMembersInfo::Kids;
        }

        if (value == "children") {
            return EMembersInfo::Children;
        }

        if (value == "adults") {
            return EMembersInfo::Adults;
        }

        throw TBlackboxError(TBlackboxError::EType::InvalidParams)
            << "Unsupported " << TStrings::GET_MEMBERS_INFO
            << " value: " << InvalidValue(value)
            << ". Could be: 'all' or 'kids'";
    }

    static const TString FAMILY_ID_PREFIX = "f";

    TString TFamilyInfoProcessor::ToInternalFamilyId(const TString& id) {
        TStringBuf res = id;
        if (!res.SkipPrefix(FAMILY_ID_PREFIX) || res.StartsWith("0")) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << TStrings::FAMILY_ID << " does not look like valid value: " << InvalidValue(id);
        }

        return TString(res);
    }

    TString TFamilyInfoProcessor::FromInternalFamilyId(const TStringBuf id) {
        return NUtils::CreateStr(FAMILY_ID_PREFIX, id);
    }

    bool TFamilyInfoProcessor::CheckUserTicket(std::optional<TString>& uid,
                                               TFamilyInfoResult& result) {
        if (!Request_.HasArg(TStrings::USER_TICKET)) {
            return true;
        }

        const TString& ut = Request_.GetArg(TStrings::USER_TICKET);
        NTvmAuth::TCheckedUserTicket ticket = Blackbox_.TvmClient().CheckUserTicket(ut);
        if (ticket.GetStatus() != NTvmAuth::ETicketStatus::Ok) {
            result.Status.Value = STATUS_INVALID_USER_TICKET;
            result.Status.Description = NUtils::CreateStr(
                "UserTicket is invalid: ", NTvmAuth::StatusToString(ticket.GetStatus()),
                ": ", TString(InvalidValue(NTvmAuth::NUtils::RemoveTicketSignature(ut))));
            return false;
        }

        if (ticket.GetDefaultUid() == 0) {
            result.Status.Value = STATUS_WRONG_USER;
            result.Status.Description = NUtils::CreateStr(
                "default_uid in UserTicket is '0'",
                ": ", TString(InvalidValue(NTvmAuth::NUtils::RemoveTicketSignature(ut))));
            return false;
        }

        uid = IntToString<10>(ticket.GetDefaultUid());
        return true;
    }

    void TFamilyInfoProcessor::BuildSimpleResult(TDbFetcher& fetcher,
                                                 const bool getPlace,
                                                 TFamilyInfoResult& result) {
        const TDbProfile* profile = nullptr;

        while ((profile = fetcher.NextProfile())) {
            // Here should be only regular users.
            // We need to protect most part of services: they don't know about kids
            if (profile->IsKid()) {
                continue;
            }

            TFamilyInfoResult::TUser user{
                .Uid = profile->Uid(),
            };

            if (getPlace) {
                if (!profile->FamilyInfo()) {
                    TLog::Error() << "FamilyInfoProcessor: no family_info in profile (get_place): "
                                  << profile->Uid();
                    throw TBlackboxError(TBlackboxError::EType::Unknown)
                        << "Internal error: no family info for profile";
                }

                user.Place = profile->FamilyInfo()->Place;
            }

            result.Users.push_back(std::move(user));
        }
    }

    bool TFamilyInfoProcessor::BuildCommonResult(TDbFetcher& fetcher,
                                                 const std::optional<TString>& uidFromCred,
                                                 const bool getPlace,
                                                 const int isChildIdx,
                                                 TBaseResultHelper& baseResultHelper,
                                                 TFamilyInfoResult& result) {
        const TDbProfile* profile = nullptr;

        while ((profile = fetcher.NextProfile())) {
            if (uidFromCred && *uidFromCred == profile->Uid() && profile->IsKid()) {
                // Users will be serialized only with non-empty adminUid.
                // It won't be set for non-OK status. Just clean up for consistency
                result.Users.clear();

                result.Status.Value = STATUS_WRONG_USER;
                result.Status.Description = NUtils::CreateStr(
                    "default_uid from UserTicket == '", *uidFromCred,
                    "' is kid");
                return false;
            }

            TFamilyInfoResult::TUser user{
                .Uid = profile->Uid(),
                .IsChild = profile->Get(isChildIdx)->AsBoolean(),
                .IsKid = profile->IsKid(),
                .BaseResult = std::make_shared<TBaseResult>(),
            };
            baseResultHelper.FillResults(*user.BaseResult, profile);
            user.BaseResult->Uid = TUidHelper::Result(profile, false, profile->PddDomItem());

            if (getPlace) {
                if (!profile->FamilyInfo()) {
                    TLog::Error() << "FamilyInfoProcessor: no family_info in profile (get_place): "
                                  << profile->Uid();
                    throw TBlackboxError(TBlackboxError::EType::Unknown)
                        << "Internal error: no family info for profile";
                }

                user.Place = profile->FamilyInfo()->Place;
            }

            result.Users.push_back(std::move(user));
        }

        return true;
    }
}
