#include "find_by_phone_numbers.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/find_by_phone_numbers_result.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <set>

namespace NPassport::NBb {
    static const TString PHONE_BINDINGS_QUERY = "SELECT number,uid FROM phone_bindings WHERE unix_timestamp(bound)>0 AND number IN (";

    TFindByPhoneNumbersProcessor::TFindByPhoneNumbersProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TFindByPhoneNumbersProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::FindByPhoneNumbers);

        return checker;
    }

    TFindByPhoneNumbersProcessor::TNumberUidPairs TFindByPhoneNumbersProcessor::ReadPhoneBindings(
        const TString& numbersList,
        std::set<TString>& foundUids) {
        TNumberUidPairs result;
        TString query = NUtils::CreateStr(PHONE_BINDINGS_QUERY, numbersList, ')');

        // iterate over all shards and join into table result
        std::vector<NDbPool::TNonBlockingHandle> handles;
        for (NDbPool::TDbPool& db : Blackbox_.ShardsMap().GetShards()) {
            NDbPool::TNonBlockingHandle sqlh(db);
            sqlh.SendQuery(query);
            handles.push_back(std::move(sqlh));
        }

        for (NDbPool::TNonBlockingHandle& sqlh : handles) {
            std::unique_ptr<NDbPool::TResult> res = TUtils::WaitResult(sqlh, "dbpool exception in FindByPhoneNumbers");

            TString number;
            TString uid;
            while (res->Fetch(number, uid)) {
                foundUids.insert(uid);
                result.emplace_back(std::move(number), std::move(uid));
            }
        }

        return result;
    }

    std::unique_ptr<TFindByPhoneNumbersResult> TFindByPhoneNumbersProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& numbersStr = TUtils::GetCheckedArg(Request_, TStrings::NUMBERS);
        std::vector<TStringBuf> numbers = NUtils::NormalizeListValue<TStringBuf>(numbersStr, ",");

        if (numbers.size() > Blackbox_.FindByPhoneNumbersLimit()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Invalid number value: too many numbers in request:" << numbers.size()
                << ", allowed: " << Blackbox_.FindByPhoneNumbersLimit();
        }

        std::unique_ptr<TFindByPhoneNumbersResult> result = std::make_unique<TFindByPhoneNumbersResult>();
        TFindByPhoneNumbersResult::TPhoneNumbersMap phoneNumbersMap;

        TString numbersList;
        numbersList.reserve(numbersStr.size());

        // 1. Sanitize numbers and read from phone_bindings
        for (const TStringBuf n : numbers) {
            const TString num = TUtils::SanitizePhoneNumber(n);

            NUtils::AppendSeparated(numbersList, ',', num);
            phoneNumbersMap.emplace(num, TFindByPhoneNumbersResult::TUidVector());
        }

        if (numbersList.empty()) {
            return result;
        }

        std::set<TString> foundUids;
        TNumberUidPairs numberUidPairs = ReadPhoneBindings(numbersList, foundUids);

        // 2. Get user profiles for all found uids
        std::vector<TString> uidsList(foundUids.begin(), foundUids.end());

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();

        fetcher.AddAlias(TAlias::PORTAL_LOGIN);
        fetcher.AddAlias(TAlias::LITE_LOGIN);
        fetcher.AddAlias(TAlias::SOCIAL_LOGIN);
        fetcher.AddAlias(TAlias::PDD_MASTER_LOGIN);
        fetcher.AddAlias(TAlias::NEOPHONISH);

        const TDbIndex availableIdx = fetcher.AddAttr(TAttr::ACCOUNT_IS_AVAILABLE);
        const TDbIndex changeReasonIdx = fetcher.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON);
        const TDbIndex createRequiredIdx = fetcher.AddAttr(TAttr::PASSWORD_CREATING_REQUIRED);

        // here we ask for attributes: 3, 22, 23
        // let's set condition to "type < 4 or type in (22, 23)" to save network bandwidth
        fetcher.SetAttrsBoundary(4);

        fetcher.FetchByUids(uidsList);

        // 3. Filter out uids that are notfound, empty, wrong alias types, disabled, change_pwd, create_pwd
        for (const TString& currentUid : uidsList) {
            const TDbProfile* profile = fetcher.ProfileByUid(currentUid);

            if (!profile) { // skip not found users
                foundUids.erase(currentUid);
                continue;
            }

            bool hasAlias = profile->HasAlias(TAlias::PORTAL_LOGIN) ||
                            profile->HasAlias(TAlias::LITE_LOGIN) ||
                            profile->HasAlias(TAlias::SOCIAL_LOGIN) ||
                            profile->HasAlias(TAlias::PDD_MASTER_LOGIN) ||
                            profile->HasAlias(TAlias::NEOPHONISH);

            if (!hasAlias) { // other accounts not supported
                foundUids.erase(currentUid);
                continue;
            }

            const bool available = profile->Get(availableIdx)->AsBoolean();
            const bool changeReason = profile->Get(changeReasonIdx)->AsBoolean();
            const bool createRequired = profile->Get(createRequiredIdx)->AsBoolean();

            if (!available || changeReason || createRequired) { // account blocked in some way
                foundUids.erase(currentUid);
                continue;
            }
        }

        // 4. Build a result
        for (auto& it : numberUidPairs) {
            TFindByPhoneNumbersResult::TUidVector& uids = phoneNumbersMap[std::move(it.first)];
            if (foundUids.find(it.second) != foundUids.end()) {
                uids.push_back(std::move(it.second));
            }
        }

        result->Data = std::move(phoneNumbersMap);

        return result;
    }
}
