#include "find_pdd_accounts.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/domain/domain_fetcher.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/list_result.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>

#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

namespace NPassport::NBb {
    const TString MaxPddUidsCount = "100";

    TFindPddAccountsProcessor::TFindPddAccountsProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TFindPddAccountsProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::FindPddAccounts);

        return checker;
    }

    std::unique_ptr<TListResult> TFindPddAccountsProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& login = Request_.GetArg(TStrings::LOGIN);
        const TString& domid = TUtils::GetUIntArg(Request_, TStrings::DOMAIN_ID);
        const TString domain = NIdn::UtfToPunycode(
            NUtils::TolowerCopy(Request_.GetArg(TStrings::DOMAIN_)));

        const TString& offset = TUtils::GetUIntArg(Request_, TStrings::OFFSET);
        TString limit = TUtils::GetUIntArg(Request_, TStrings::LIMIT);
        const TString& sort = Request_.GetArg(TStrings::SORT);

        // sanitize login
        for (size_t idx = 0; idx < Blackbox_.MinCharsCountBeforeWildcard() && idx < login.size(); ++idx) {
            if ('*' == login[idx]) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Wildcard is restricted for prefix shorter than "
                    << Blackbox_.MinCharsCountBeforeWildcard() << " chars: "
                    << InvalidValue(login);
            }
        }

        std::unique_ptr<TListResult> result = std::make_unique<TListResult>(TOutTokens::UIDS);

        TString masterDomid;

        if (!domid.empty()) { // convert to master domain_id
            TDomainFetcher::TResult domRes;

            if (!(domRes = Blackbox_.HostedDomains().FindById(domid)) && // look in DB if not found
                !(domRes = Blackbox_.HostedDomains().FindInDb(domid, true)))
            {
                result->TotalCount = 0;
                return result;
            }

            masterDomid = domRes.Domain->Id();
        }

        if (!domain.empty()) { // convert to master domain_id
            TDomainFetcher::TResult domRes;

            if (!(domRes = Blackbox_.HostedDomains().Find(domain)) && // look in DB if not found
                !(domRes = Blackbox_.HostedDomains().FindInDb(domain, false)))
            {
                result->TotalCount = 0;
                return result;
            }

            if (!masterDomid.empty() && masterDomid != domRes.Domain->Id()) { // 2 different domains specified
                result->TotalCount = 0;
                return result;
            }

            if (masterDomid.empty()) {
                masterDomid = domRes.Domain->Id();
            }
        }

        if (masterDomid.empty()) { // neither domain nor domain_id specified
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "unknown domain: please specify domain or domain_id";
        }

        bool singleAccount = !login.empty() && (login.find('*') == TString::npos); // non-wildcard login

        TString sortKey;
        TString sortMode;

        if (!sort.empty()) {
            size_t dot = sort.find('.');
            sortKey.assign(NUtils::TolowerCopy(sort.substr(0, dot)));

            if (sortKey != TStrings::LOGIN && sortKey != TStrings::UID) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "invalid sort key: sorting by login or uid allowed: " << InvalidValue(sortKey);
            }

            if (sortKey == TStrings::LOGIN) {
                sortKey.assign(TStrings::VALUE);
            }

            if (dot != TString::npos) {
                sortMode.assign(NUtils::TolowerCopy(sort.substr(dot + 1)));
                if (!sortMode.empty() && sortMode != "asc" && sortMode != "desc") {
                    throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                        << "invalid sort mode: only 'asc' or 'desc' is allowed: " << InvalidValue(sortMode);
                }
            }
        }

        TString queryCond(" WHERE type");

        // optimize search if login is not specified:
        // in this case it is enough to count/select master pdd accounts and don't look at aliases at all
        queryCond.append(login.empty() ? "=7" : " IN (7,8)");
        const TString uidClause = login.empty() ? "uid" : "DISTINCT uid"; // use DISTINCT only when two aliases searched
        queryCond.append(" AND value");

        TString dbLogin;
        dbLogin.reserve(login.size() + 10);
        dbLogin.append(masterDomid).push_back('/');

        if (singleAccount) {
            queryCond.append("='");
            dbLogin.append(login);
        } else {
            queryCond.append(" LIKE '");
            // in LIKE clause we need to quote _ and %
            if (login.empty()) {
                dbLogin.push_back('%');
            } else {
                for (const char c : login) {
                    switch (c) {
                        case '*':
                            dbLogin.push_back('%');
                            break;
                        case '_':
                            dbLogin.append("\\_");
                            break;
                        case '%':
                            dbLogin.append("\\%");
                            break;
                        default:
                            dbLogin.push_back(c);
                    }
                }
            }
        }
        NUtils::Tolower(dbLogin);

        std::unique_ptr<NDbPool::TResult> res;
        try {
            NDbPool::TBlockingHandle sqlh(Blackbox_.CentralSlowDb());
            dbLogin = sqlh.EscapeQueryParam(dbLogin);
            queryCond.append(dbLogin).push_back('\'');

            TString query;

            if (!singleAccount) { // get count first
                bool ignoreCount = TUtils::GetBoolArg(Request_, TStrings::IGNORE_COUNT);

                if (ignoreCount) {
                    result->TotalCount = -1;
                } else { // need to get total count first
                    query.assign("SELECT count(").append(uidClause).append(") FROM aliases").append(queryCond);

                    // TLog::Error("QUERY: '%s'", query.c_str());
                    const NDbPool::TTable table = sqlh.Query(query)->ExctractTable();
                    if (!table.empty() && !table[0][0].IsNull()) {
                        int totalCount = table[0][0].AsInt();
                        result->TotalCount = totalCount;

                        if (!totalCount) { // no matching records
                            return result;
                        }
                    }
                }

                if (!sortKey.empty()) {
                    queryCond.append(" ORDER BY ").append(sortKey);
                    if (!sortMode.empty()) {
                        queryCond.append(" ").append(sortMode);
                    }
                }

                // if no limit given, set limit to 100
                if (limit.empty()) {
                    limit = MaxPddUidsCount;
                }

                queryCond.append(" LIMIT ");
                if (!offset.empty()) {
                    queryCond.append(offset).push_back(',');
                }

                queryCond.append(limit);
            }
            // main db request
            query.assign("SELECT ").append(uidClause).append(" FROM aliases FORCE KEY (type)").append(queryCond);

            // TLog::Error("QUERY: '%s'", query.c_str());
            res = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception in findPddAccounts: %s", e.what());
            throw TDbpoolError("dbpool exception in findPddAccounts", e.what());
        }

        TListResult::TStringVector data;
        data.reserve(res->size());

        for (const NDbPool::TRow& row : res->Table()) {
            if (!row[0].IsNull()) {
                data.push_back(row[0].AsString());
            }
        }

        if (singleAccount) { // possibly no accounts found
            result->TotalCount = data.size();
        }

        result->Data = std::move(data);

        return result;
    }
}
