#include "get_all_tracks.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/all_tracks_result.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/track_result.h>

#include <passport/infra/libs/cpp/json/reader.h>

namespace NPassport::NBb {
    TGetAllTracksProcessor::TGetAllTracksProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TGetAllTracksProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::GetAllTracks);

        return checker;
    }

    static const TString GET_ALL_TRACKS_QUERY_BEGIN_ =
        "SELECT UNIX_TIMESTAMP(created),UNIX_TIMESTAMP(expired),content,track_id FROM tracks WHERE uid=";
    static const TString GET_ALL_TRACKS_QUERY_END_ = " LIMIT 1000";

    std::unique_ptr<TAllTracksResult> TGetAllTracksProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID);
        std::unique_ptr<TAllTracksResult> res = std::make_unique<TAllTracksResult>();

        std::unique_ptr<NDbPool::TResult> dbResult;
        TString query = NUtils::CreateStr(GET_ALL_TRACKS_QUERY_BEGIN_, uid, GET_ALL_TRACKS_QUERY_END_);

        try {
            NDbPool::TBlockingHandle sqlh(Blackbox_.ShardsMap().GetPool(uid));
            dbResult = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception querying shard '%s' : '%s'", query.c_str(), e.what());
            throw TDbpoolError("dbpool exception in track fetch", e.what());
        }

        res->Tracks.reserve(dbResult->size());
        for (const NDbPool::TRow& row : dbResult->Table()) {
            TString content = row[2].AsString();
            std::unique_ptr<rapidjson::Document> rapidValue = std::make_unique<rapidjson::Document>();
            if (!NJson::TReader::DocumentAsObject(content, *rapidValue)) {
                TLog::Error("Error: broken json in shard db (tracks). uid: '%s'", uid.c_str());
            }

            std::unique_ptr<TTrackResult> track = std::make_unique<TTrackResult>(uid, row[3].AsString());
            track->Created = row[0].AsString();
            track->Expired = row[1].AsString();
            track->Content = content;
            track->ContentRapidObj = std::move(rapidValue);

            res->Tracks.push_back(std::move(track));
        }

        return res;
    }
}
