#include "get_debug_user_ticket.h"

#include "oauth.h"
#include "sessionid.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/rate_limiter.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/oauth/scopes.h>
#include <passport/infra/daemons/blackbox/src/oauth/status.h>
#include <passport/infra/daemons/blackbox/src/output/get_debug_user_ticket_result.h>
#include <passport/infra/daemons/blackbox/src/output/oauth_result.h>

namespace NPassport::NBb {
    TGetDebugUserTicketProcessor::TGetDebugUserTicketProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TGetDebugUserTicketProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        // This method does not require grants - it is by design: PASSP-30722
        // It requires ServiceTicket - this check was done earlier

        return checker;
    }

    std::unique_ptr<TGetDebugUserTicketResult> TGetDebugUserTicketProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);
        if (!Blackbox_.RateLimiter().CheckLimit(
                NUtils::CreateStr("get_debug_user_ticket_", consumer.GetClientId()),
                TRateLimiterSettings::GetDebugUserTicket))
        {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Too many requests to method=get_debug_user_ticket from tvmid="
                << consumer.GetClientId();
        }

        TOAuthProcessor oauthProc(Blackbox_, Request_);

        std::unique_ptr<TOAuthResult> oauthResult = oauthProc.ProcessImpl(
            consumer,
            TOAuthProcessor::TOptions{
                .UserIp = Request_.GetRemoteAddr(),
            });

        const bool isTokenValid = oauthResult->OauthChunk.Status.Status() == TOAuthStatus::Valid;
        if (isTokenValid && !oauthResult->Uid) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "OAuth token is valid but does not contain uid (USER):"
                << " this method allowes to get only USER-ticket."
                << " Please use token with uid";
        }

        const bool addBbSessionid = TUtils::GetBoolArg(Request_, TStrings::ADD_SCOPE_SESSIONID);
        const bool addBbSessguard = TUtils::GetBoolArg(Request_, TStrings::ADD_SCOPE_SESSGUARD);
        if ((addBbSessionid || addBbSessguard) && isTokenValid) {
            const bool isXtoken =
                oauthResult->OauthChunk.TokenInfo
                    ? oauthResult->OauthChunk.TokenInfo->IsXToken()
                    : false;
            if (!isXtoken) {
                throw TBlackboxError(TBlackboxError::EType::Unknown)
                    << "Impossible to add scope '" << TSessionidProcessor::SCOPE_BB_SESSIONID
                    << "' or '" << TSessionidProcessor::SCOPE_BB_SESSGUARD
                    << "' to UserTicket: oauth token must be xtoken";
            }
        }

        auto res = std::make_unique<TGetDebugUserTicketResult>();
        res->Status = oauthResult->OauthChunk.Status.AsString();
        res->Comment = oauthResult->OauthChunk.Comment;

        if (isTokenValid) {
            // there is no way in method=sessionid to create userticket with only 'bb:sessguard'
            if (addBbSessionid || addBbSessguard) {
                oauthResult->OauthChunk.TokenInfo->AddScope(
                    TOAuthSingleScope(TSessionidProcessor::SCOPE_BB_SESSIONID.copy()));
            }
            if (addBbSessguard) {
                oauthResult->OauthChunk.TokenInfo->AddScope(
                    TOAuthSingleScope(TSessionidProcessor::SCOPE_BB_SESSGUARD.copy()));
            }

            res->UsetTicket = oauthProc.BuildUserTicket(
                TUtils::ToUInt(oauthResult->Uid->Uid, TStrings::UID),
                consumer,
                oauthResult->OauthChunk.TokenInfo->GetScopeCollection());
        }

        return res;
    }
}
