#include "get_device_public_key.h"

#include "check_device_signature.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/get_device_public_key_result.h>

#include <passport/infra/libs/cpp/utils/crypto/rsa.h>

namespace NPassport::NBb {
    TGetDevicePublicKeyProcessor::TGetDevicePublicKeyProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TGetDevicePublicKeyProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::GetDevicePublicKey);

        return checker;
    }

    std::unique_ptr<TGetDevicePublicKeyResult> TGetDevicePublicKeyProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& deviceId = TUtils::GetCheckedArg(Request_, TStrings::DEVICE_ID);

        TString query;

        // get public key
        TString publicKey;
        TString version;
        TString ownerId;
        try {
            NDbPool::TBlockingHandle handle(Blackbox_.CentralDb());

            query = NUtils::CreateStr(
                "SELECT public_key,version,owner_id FROM device_public_key WHERE device_id='",
                handle.EscapeQueryParam(deviceId),
                "'");

            if (!handle.Query(query)->Fetch(publicKey, version, ownerId)) {
                const TString details = NUtils::CreateStr("Public key is absent for device_id=", deviceId);
                TLog::Debug() << "getDevicePublicKey: " << details;
                return std::make_unique<TGetDevicePublicKeyResult>(
                    TCheckDeviceSignatureProcessor::CDS_STATUS_PUBLIC_KEY_NOT_FOUND,
                    publicKey,
                    version,
                    ownerId,
                    details);
            }
        } catch (const NDbPool::TException& e) {
            TLog::Debug() << "Failed to get public_key for device_id: <" << query << ">. " << e.what();
            throw TDbpoolError("Failed to get public_key for device_id", e.what());
        }

        // check version
        if ("1" != version) {
            const TString details = NUtils::CreateStr("Supported versions: 1. got: ", version);
            TLog::Debug() << "getDevicePublicKey: " << details;
            return std::make_unique<TGetDevicePublicKeyResult>(
                TCheckDeviceSignatureProcessor::CDS_STATUS_PUBLIC_KEY_UNSUPPORTED_VERSION,
                publicKey,
                version,
                ownerId,
                details);
        }

        // parse public key
        NUtils::TRsaPublicEvp rsa;
        try {
            rsa = NUtils::TRsaPublicEvp::FromPem(publicKey);
        } catch (const std::exception& e) {
            const TString details = NUtils::CreateStr("Public key is invalid: ", e.what());
            TLog::Debug() << "checkDeviceSignature: " << details;
            return std::make_unique<TGetDevicePublicKeyResult>(
                TCheckDeviceSignatureProcessor::CDS_STATUS_PUBLIC_KEY_INVALID,
                publicKey,
                version,
                ownerId,
                details);
        }

        return std::make_unique<TGetDevicePublicKeyResult>(
            TCheckDeviceSignatureProcessor::CDS_STATUS_OK,
            publicKey,
            version,
            ownerId,
            TString());
    }
}
