#include "get_oauth_tokens.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/account_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/oauth_attrs_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/strong_pwd_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/oauth/fetcher.h>
#include <passport/infra/daemons/blackbox/src/output/get_oauth_tokens_result.h>

namespace NPassport::NBb {
    TGetOAuthTokensProcessor::TRequestParams
    TGetOAuthTokensProcessor::TRequestParams::Create(const NCommon::TRequest& request) {
        TRequestParams res{
            .Uid = TUtils::GetUIntArg(request, TStrings::UID, true),
            .Fullinfo = TUtils::GetBoolArg(request, TStrings::FULL_INFO),
            .NeedLoginId = TUtils::GetBoolArg(request, TStrings::GET_LOGIN_ID),
            .XtokenOnly = TUtils::GetBoolArg(request, TStrings::XTOKEN_ONLY),
            .DeviceId = request.GetArg(TStrings::DEVICE_ID),
        };

        TStringBuf clientId = TUtils::GetUIntArg(request, TStrings::CLIENT_ID);
        if (clientId) {
            res.ClientId = IntFromString<ui32, 10>(clientId);
        }

        return res;
    }

    TGetOAuthTokensProcessor::TGetOAuthTokensProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGetOAuthTokensProcessor::~TGetOAuthTokensProcessor() = default;

    TGrantsChecker TGetOAuthTokensProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::GetOAuthTokens);
        checker.CheckHasArgAllowed(TStrings::FULL_INFO, TBlackboxFlags::FullInfo);

        TOAuthAttrsHelper::CheckGrants(checker);
        TAccountHelper::CheckGrants(checker);

        return checker;
    }

    std::unique_ptr<TGetOAuthTokensResult> TGetOAuthTokensProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        Params_ = TRequestParams::Create(Request_);

        // fetchers and helpers can throw, so keep them at the very begining
        TOAuthMultiFetcher oauthFetcher(Blackbox_.OauthConfig());
        TOAuthAttrsHelper attrs(oauthFetcher, Request_);
        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        AddParamsToFetcher(fetcher);

        TOAuthMultiFetcher::TTokens tokens = FetchEntriesByUid(fetcher, oauthFetcher);
        if (tokens.empty()) {
            return std::make_unique<TGetOAuthTokensResult>();
        }

        const TDbProfile& profile = *fetcher.NextProfile();
        tokens = FetchClientInfoAndFilterTokens(std::move(tokens), profile, oauthFetcher);
        if (tokens.empty()) {
            return std::make_unique<TGetOAuthTokensResult>();
        }

        return BuildResult(attrs, std::move(tokens), Params_->NeedLoginId);
    }

    void TGetOAuthTokensProcessor::AddParamsToFetcher(TDbFetcher& fetcher) {
        AttrIndexes_ = TAttrIndexes{
            .Available = fetcher.AddAttr(TAttr::ACCOUNT_IS_AVAILABLE),
            .Glogout = fetcher.AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME),
            .RevokerTokens = fetcher.AddAttr(TAttr::REVOKER_TOKENS),
            .ChangeReason = fetcher.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON),
            .CreateRequired = fetcher.AddAttr(TAttr::PASSWORD_CREATING_REQUIRED),
        };

        StrongPwdHelper_ = std::make_unique<TStrongPwdHelper>(fetcher);

        TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId());
        // just add default aliases to fetcher: it allowes to find account
        TAccountHelper(conv, Request_);
    }

    TOAuthMultiFetcher::TTokens
    TGetOAuthTokensProcessor::FetchEntriesByUid(TDbFetcher& fetcher,
                                                const TOAuthMultiFetcher& oauthFetcher) const {
        // All queries will be performed in parallel way

        TOAuthMultiFetcher::TDbHandles tokenAttrsHandles =
            oauthFetcher.SendRequestForTokenAttrs(Params_->Uid.get(),
                                                  Params_->ClientId,
                                                  Params_->DeviceId);

        fetcher.FetchByUid(Params_->Uid.get());

        const TDbProfile* profile = fetcher.NextProfile();
        if (!profile) {
            LogDebugMessage("uid was not found");
            return {};
        }
        // to find this profile again later
        fetcher.ResetProfiles();

        TOAuthMultiFetcher::TTokens tokens = oauthFetcher.WaitTokenAttrs(std::move(tokenAttrsHandles));
        if (tokens.empty()) {
            LogDebugMessage("there are no tokens for uid");
            return {};
        }

        return tokens;
    }

    TOAuthMultiFetcher::TTokens
    TGetOAuthTokensProcessor::FetchClientInfoAndFilterTokens(TOAuthMultiFetcher::TTokens tokens,
                                                             const TDbProfile& profile,
                                                             const TOAuthMultiFetcher& oauthFetcher) const {
        if (Params_->XtokenOnly) {
            tokens = FilterByXToken(std::move(tokens));
            if (tokens.empty()) {
                LogDebugMessage("there are no tokens for uid after filtering by xtoken");
                return {};
            }
        }

        tokens = oauthFetcher.FetchClientAttrs(std::move(tokens));
        for (auto& [id, token] : tokens) {
            if (token.Error.Error() == TOAuthError::Ok) {
                token.Error = UpdateTokenStatus(profile, token);
            }
        }

        if (!Params_->Fullinfo) {
            tokens = FilterByTokenStatus(std::move(tokens));
            if (tokens.empty()) {
                LogDebugMessage("there are no tokens for uid after filtering by status");
                return {};
            }
        }

        return tokens;
    }

    TOAuthError TGetOAuthTokensProcessor::UpdateTokenStatus(const TDbProfile& profile,
                                                            const TOAuthMultiFetcher::TToken& token) const {
        if (!profile.Get(AttrIndexes_.Available)->AsBoolean()) {
            return TOAuthError::AccountDisabled;
        }

        if (StrongPwdHelper_->PasswdExpired(&profile, Blackbox_.StrongPwdExpireTime())) {
            return TOAuthError::ExpiredPassword;
        }

        const time_t issuedTime = token.Info->GetMinimumIssueTime();

        time_t glogoutTime = profile.Get(AttrIndexes_.Glogout)->AsTime();
        if (issuedTime < glogoutTime) {
            return TOAuthError::GLogout;
        }

        time_t revokeTokensTime = profile.Get(AttrIndexes_.RevokerTokens)->AsTime();
        if (issuedTime < revokeTokensTime) {
            return TOAuthError::Revoked;
        }

        const TInstant domainGlogout = profile.PddDomItem().Glogout();
        if (issuedTime < domainGlogout.TimeT()) {
            return TOAuthError::GLogout;
        }

        if (profile.Get(AttrIndexes_.ChangeReason)->AsBoolean()) {
            return TOAuthError::PasswordChangeRequired;
        }

        if (profile.Get(AttrIndexes_.CreateRequired)->AsBoolean()) {
            return TOAuthError::PasswordCreateRequired;
        }

        return TOAuthError::Ok;
    }

    template <typename Func>
    static TOAuthMultiFetcher::TTokens EraseIfNot(TOAuthMultiFetcher::TTokens tokens, Func func) {
        for (auto it = tokens.begin(); it != tokens.end();) {
            if (func(it->second)) {
                ++it;
            } else {
                tokens.erase(it++);
            }
        }

        return tokens;
    }

    TOAuthMultiFetcher::TTokens TGetOAuthTokensProcessor::FilterByXToken(TOAuthMultiFetcher::TTokens tokens) {
        return EraseIfNot(
            std::move(tokens),
            [](const TOAuthMultiFetcher::TToken& token) -> bool {
                return token.Info->IsXToken();
            });
    }

    TOAuthMultiFetcher::TTokens TGetOAuthTokensProcessor::FilterByTokenStatus(TOAuthMultiFetcher::TTokens tokens) {
        return EraseIfNot(
            std::move(tokens),
            [](const TOAuthMultiFetcher::TToken& token) -> bool {
                return token.Error.Error() == TOAuthError::Ok;
            });
    }

    std::unique_ptr<TGetOAuthTokensResult>
    TGetOAuthTokensProcessor::BuildResult(const TOAuthAttrsHelper& attrsHelper,
                                          TOAuthMultiFetcher::TTokens tokens,
                                          bool needLoginId) {
        std::unique_ptr res = std::make_unique<TGetOAuthTokensResult>();
        res->Tokens.reserve(tokens.size());

        for (auto& [id, token] : tokens) {
            TOAuthChunk out{
                .TokenAttrs = attrsHelper.TokenAttrsChunk(token.Info.get()),
                .ClientAttrs = attrsHelper.ClientAttrsChunk(token.Info.get()),
                .Status = token.Error.ConvertToStatus(),
                .Comment = token.Error.Msg(),
            };

            if (needLoginId) {
                out.LoginId = token.Info->GetLoginId();
            }
            out.TokenInfo = std::move(token.Info);

            res->Tokens.push_back(std::move(out));
        }

        return res;
    }

    void TGetOAuthTokensProcessor::LogDebugMessage(TStringBuf description) const {
        TLog::Debug() << "GetOAuthTokens: " << description << ": " << Params_->Uid.get();
    }
}
