#include "get_track.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/track_result.h>

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

namespace NPassport::NBb {
    TGetTrackProcessor::TGetTrackProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TGetTrackProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::GetTrack);

        return checker;
    }

    static const TString TRACK_QUERY_START =
        "SELECT UNIX_TIMESTAMP(created),UNIX_TIMESTAMP(expired),content "
        "FROM tracks WHERE uid=";
    static const TString TRACK_QUERY_MID = " and track_id='";
    static const TString TRACK_QUERY_END = "'";
    static TString TrackQuery(const TString& uid, const TString& track_id) {
        return NUtils::CreateStr(TRACK_QUERY_START, uid, TRACK_QUERY_MID, track_id, TRACK_QUERY_END);
    }

    std::unique_ptr<TTrackResult> TGetTrackProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID);
        const TString& trackId = TUtils::GetCheckedArg(Request_, TStrings::TRACK_ID);

        std::unique_ptr<TTrackResult> res = std::make_unique<TTrackResult>(uid, trackId);

        NDbPool::TDbPool& db = Blackbox_.ShardsMap().GetPool(uid);

        std::unique_ptr<NDbPool::TResult> dbResult;
        {
            TString query;
            try {
                NDbPool::TBlockingHandle sqlh(db);
                query = TrackQuery(uid, sqlh.EscapeQueryParam(NUtils::TolowerCopy(trackId)));
                dbResult = sqlh.Query(query);
            } catch (const NDbPool::TException& e) {
                TLog::Debug("BlackBox: dbpool exception querying shard '%s' : '%s'", query.c_str(), e.what());
                throw TDbpoolError("dbpool exception in track fetch", e.what());
            }
        }

        if (!dbResult->Fetch(res->Created, res->Expired, res->Content)) {
            return res;
        }

        std::unique_ptr<rapidjson::Document> rapidValue = std::make_unique<rapidjson::Document>();
        if (!NJson::TReader::DocumentAsObject(res->Content, *rapidValue)) {
            TLog::Error("Error: broken json in shard db (tracks). uid: '%s', track_id: '%s'", uid.c_str(), trackId.c_str());
        }

        res->ContentRapidObj = std::move(rapidValue);

        return res;
    }
}
