#include "hosted_domains.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/domain/domain_fetcher.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/table_result.h>

#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    static const TString DOMAINS_HOSTS_QUERY_SEL = "SELECT m.domain_id,m.name,m.admin_uid,m.mx,m.default_uid,m2.name,m.enabled,m.options,m.ts";
    static const TString DOMAINS_HOSTS_QUERY_SLAVE = ",NULL as domain_id,s.name";
    static const TString DOMAINS_HOSTS_QUERY_SLAVE_EXT = ",s.domain_id,s.name,s.admin_uid,s.mx,s.default_uid,m.name,s.enabled,s.options,s.ts";
    static const TString DOMAINS_HOSTS_QUERY_FROM = " FROM domains m LEFT JOIN domains m2 ON m.master_domain_id=m2.domain_id";
    static const TString DOMAINS_HOSTS_QUERY_JOIN = " LEFT JOIN domains s ON s.master_domain_id=m.domain_id";
    static const TString DOMAINS_HOSTS_QUERY_ORDER = " ORDER BY m.domain_id";

    static const TTableResult::TStringVector HOSTED_DOMAINS_COLUMNS = {
        "domid",
        "domain",
        "admin",
        "mx",
        "default_uid",
        "master_domain",
        "ena",
        "options",
        "born_date",
        "slaves",
    };

    THostedDomainsProcessor::THostedDomainsProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker THostedDomainsProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::HostedDomains);

        return checker;
    }

    // TODO: drop after PASSP-33935
    static const TString METHOD_NAME = "hosted_domains";

    std::unique_ptr<TTableResult> THostedDomainsProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& admin = TUtils::GetUIntArg(Request_, TStrings::DOMAIN_ADMIN);
        const TString& domid = TUtils::GetUIntArg(Request_, TStrings::DOMAIN_ID);
        const TString& rawDomain = Request_.GetArg(TStrings::DOMAIN_);

        if (domid.empty() && rawDomain.empty() && admin.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "No selection filters, please specify domain, domain_id or domain_admin argument";
        }

        const TString domain = NIdn::UtfToPunycode(
            NUtils::TolowerCopy(rawDomain));
        bool aliases = TUtils::GetBoolArg(Request_, TStrings::ALIASES);

        std::unique_ptr<TTableResult> result = std::make_unique<TTableResult>(
            TOutTokens::HOSTED_DOMAINS, HOSTED_DOMAINS_COLUMNS);

        if (domid.empty() && domain.empty() && admin.empty()) {
            return result; // domain was given but has invalid format, return empty result
        }

        // TODO: allow to use cache with `admin` too
        const bool onlyAdmin = !admin.empty() && domain.empty() && domid.empty();
        if (onlyAdmin) {
            TLog::Warning() << "HostedDomains: performance issue:"
                            << " there is no cache for case with domain_admin";
        }

        result->Rows = !onlyAdmin && consumer.IsMethodCacheable(METHOD_NAME)
                           ? FetchFromCache(admin, domid, domain, aliases)
                           : FetchFromDb(admin, domid, domain, aliases);

        return result;
    }

    static void FixBooleanColumns(TTableResult::TStringVector& row) {
        for (int idx : {
                 3, // mx
                 6, // ena
             }) {
            row[idx] = TDbValue::AsBoolean(row[idx])
                           ? TStrings::ONE
                           : TStrings::ZERO;
        }
    }

    std::vector<TTableResult::TStringVector>
    THostedDomainsProcessor::FetchFromDb(const TString& admin,
                                         const TString& domid,
                                         const TString& domain,
                                         bool aliases) const {
        TString query(DOMAINS_HOSTS_QUERY_SEL);

        if (aliases) {
            query.append(!admin.empty() && domid.empty() && domain.empty()
                             ? DOMAINS_HOSTS_QUERY_SLAVE
                             : DOMAINS_HOSTS_QUERY_SLAVE_EXT);
        }

        query.append(DOMAINS_HOSTS_QUERY_FROM);

        if (aliases) {
            query.append(DOMAINS_HOSTS_QUERY_JOIN);
        }

        std::unique_ptr<NDbPool::TResult> res;
        try {
            NDbPool::TBlockingHandle sqlh(Blackbox_.CentralDb());
            TString whereClause;

            if (!domid.empty()) {
                whereClause.append("m.domain_id=").append(domid);
            }

            if (!domain.empty()) {
                NUtils::AppendSeparated(whereClause, " AND ", "m.name='").append(sqlh.EscapeQueryParam(domain)).push_back('\'');
            }

            if (!admin.empty()) {
                NUtils::AppendSeparated(whereClause, " AND ", "m.admin_uid=").append(admin);
            }

            query.append(" WHERE ").append(whereClause).append(DOMAINS_HOSTS_QUERY_ORDER);

            // TLog::Error("QUERY: '%s'", query.c_str());
            res = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception in hostedDomains: %s", e.what());
            throw TDbpoolError("dbpool exception in hostedDomains", e.what());
        }

        // store results
        TString lastId;
        TString slaves;

        std::vector<TTableResult::TStringVector> rows;
        // Sql result contains rows count equal slave count. but minimum count = 1
        // We need display slaves + master (N+1)
        // The worst case: N = 1
        rows.reserve(2 * res->size());
        int masterIdx = -1;

        for (const NDbPool::TRow& row : res->Table()) {
            TString id = row[0].AsString();

            if (lastId != id) {
                // id changed, store previous slaves row if any
                if (masterIdx != -1) {
                    rows[masterIdx].push_back(slaves);
                    masterIdx = -1;
                }
                // and fill up row for new id
                lastId = id;
                slaves.clear();

                TTableResult::TStringVector currentRow;
                currentRow.reserve(9);
                for (unsigned i = 0; i < 9 && i < (unsigned)row.size(); ++i) {
                    currentRow.push_back(row[i].IsNull() ? TStrings::EMPTY : row[i].AsString());
                }

                FixBooleanColumns(currentRow);

                rows.push_back(std::move(currentRow));
                if (aliases) {
                    masterIdx = rows.size() - 1;
                }
            }

            if (row.size() > 10 && !row[10].IsNull()) {
                NUtils::AppendSeparated(slaves, ',', row[10].AsString());
            }

            if (aliases && !row[9].IsNull()) {
                TTableResult::TStringVector currentRow;
                currentRow.reserve(11);
                for (unsigned i = 9; i < 19 && i < (unsigned)row.size(); ++i) {
                    currentRow.push_back(row[i].IsNull() ? TStrings::EMPTY : row[i].AsString());
                }
                currentRow.push_back(TStrings::EMPTY);

                FixBooleanColumns(currentRow);

                rows.push_back(std::move(currentRow));
            }
        }

        // and don't forget the last slave
        if (masterIdx != -1) {
            rows[masterIdx].push_back(slaves);
        }

        return rows;
    }

    static const TString& AsString(bool value) {
        return value ? TStrings::ONE : TStrings::ZERO;
    }

    static TTableResult::TStringVector DomainToRow(const TDomain& domain,
                                                   const TDomainCache& domainsCache) {
        const TDomain* master = nullptr;
        if (domain.IsSlave()) {
            master = domainsCache.List.FindById(domain.Master());
            if (!master) {
                TLog::Warning()
                    << "HostedDomains: failed to find master domain by id=" << domain.Master()
                    << " for domain='" << domain.AsciiName() << "'";
            }
        }

        return TTableResult::TStringVector{
            domain.Id(),
            domain.AsciiName(),
            domain.AdminUid(),
            AsString(domain.Mx()),
            domain.DefaultUid(),
            master ? master->AsciiName() : TString(),
            AsString(domain.Ena()),
            domain.RawOptions(),
            domain.BornDate(),
        };
    };

    std::vector<TTableResult::TStringVector>
    THostedDomainsProcessor::FetchFromCache(const TString& admin,
                                            const TString& domid,
                                            const TString& domain,
                                            bool aliases) const {
        const TDomainFetcher::TResult fetchResult =
            domain ? Blackbox_.HostedDomains().Find(domain, TDomainFetcher::EPolicy::MasterOrAlias)
                   : Blackbox_.HostedDomains().FindById(domid, TDomainFetcher::EPolicy::MasterOrAlias);
        if (!fetchResult) {
            return {};
        }

        if (domid && domain && fetchResult.Domain->Id() != domid) {
            return {};
        }
        if (admin && fetchResult.Domain->AdminUid() != admin) {
            return {};
        }

        std::vector<TTableResult::TStringVector> res;
        res.reserve(1 + (aliases ? fetchResult.Domain->Slaves().size() : 0));
        res.push_back(DomainToRow(*fetchResult.Domain, *fetchResult.Cache));

        if (aliases) {
            TString slaves;
            for (const TString& id : fetchResult.Domain->Slaves()) {
                const TDomain* s = fetchResult.Cache->List.FindById(id);
                if (!s) {
                    TLog::Warning() << "HostedDomains: failed to find slave domain by id=" << id
                                    << " for domain='" << fetchResult.Domain->AsciiName() << "'";
                    continue;
                }

                NUtils::AppendSeparated(slaves, ',', s->AsciiName());

                res.push_back(DomainToRow(*s, *fetchResult.Cache));
                res.back().push_back({}); // slaves
            }

            res[0].push_back(std::move(slaves));
        }

        return res;
    }
}
