#include "login.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/badauth/facade.h>
#include <passport/infra/daemons/blackbox/src/badauth/helper.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/base_result_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/partitions_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/strong_pwd_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/uid_helper.h>
#include <passport/infra/daemons/blackbox/src/loggers/tskvlog.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/misc/passport_wrapper.h>
#include <passport/infra/daemons/blackbox/src/misc/password_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/perimeter.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/oauth/config.h>
#include <passport/infra/daemons/blackbox/src/oauth/fetcher.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>
#include <passport/infra/daemons/blackbox/src/output/dbfields_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/emails_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/login_result.h>
#include <passport/infra/daemons/blackbox/src/output/oauth_result.h>
#include <passport/infra/daemons/blackbox/src/output/uid_chunk.h>
#include <passport/infra/daemons/blackbox/src/staff/staff_info.h>
#include <passport/infra/daemons/blackbox/src/totp/checker.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_encryptor.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_profile.h>

#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionparser.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/common/private_key.h>
#include <passport/infra/libs/cpp/tvm/signer/signer.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/openssl/include/openssl/sha.h>

#include <library/cpp/digest/old_crc/crc.h>

#include <set>

namespace NPassport::NBb {
    static const TString REFERER("referer");
    static const TString USERAGENT("useragent");
    static const TString USER_AGENT_HEADER("user-agent");
    static const TString RETPATH("retpath");
    static const TString X_RETPATH_HEADER("X-Retpath");
    static const TString CGI_SSL("ssl");
    static const TString SSL_HEADER("X-HTTPS-Req");
    static const TString YANDEXUID("yandexuid");
    static const TString AUTH_TYPE("authtype");
    static const TString CALENDAR_AUTH_TYPE("calendar");
    static const TString WEB_AUTH_TYPE("web");
    static const TString WEBDAV_AUTH_TYPE("webdav");
    static const TString OAUTHCREATE_AUTH_TYPE("oauthcreate");
    static const TString VERIFY_AUTH_TYPE("verify");

    static const TString CAPTCHA("captcha");
    static const TString CAPTCHA_ENTERED("captcha entered");

    static const TString SSL_COMMENT("ssl=1");

    static const TString PROXYIP("proxyip");
    static const TString X_FWD_FOR("x-forwarded-for");
    static const TString X_FWD("x-forwarded");
    static const TString FWD_FOR("forwarded-for");
    static const TString FWD("forwarded");
    static const TString PROXY("proxy-user");

    TLoginProcessor::TLoginProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TString TLoginProcessor::GetUidLogin(const TString& uid) {
        TString res("UID=");
        res.append(uid);

        return res;
    }

    TString TLoginProcessor::CheckForProxy(const NCommon::TRequest& req) {
        TString header;
        if (req.HasArg(PROXYIP)) {
            header.assign(req.GetArg(PROXYIP));
        } else if (req.HasArg(X_FWD_FOR)) {
            header.assign(req.GetArg(X_FWD_FOR));
        } else if (req.HasArg(FWD) && req.HasArg(PROXY)) {
            header.assign(req.GetArg(PROXY));
        } else if (req.HasArg(FWD)) {
            header.assign(req.GetArg(FWD));
        } else if (req.HasArg(FWD_FOR)) {
            header.assign(req.GetArg(FWD_FOR));
        } else if (req.HasArg(X_FWD)) {
            header.assign(req.GetArg(X_FWD));
        } else {
            return TStrings::EMPTY;
        }

        return NUtils::TIpAddr::Normalize(header);
    }

    TGrantsChecker TLoginProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::Login);

        const TString& uid = Request_.GetArg(TStrings::UID);
        if (!uid.empty() && !consumer.IsAllowed(TBlackboxFlags::LoginByUid)) {
            checker.Add("no grants for LoginByUid()");
        }

        int apiVersion = 1;
        TryIntFromString<10>(Request_.GetArg(TStrings::API_VERSION), apiVersion);
        if (apiVersion > 1) {
            // Is the caller capable to resist bruteforce by showing CAPTCHA or delaying the response?
            if (!consumer.CanCaptcha() && !consumer.CanDelay()) {
                checker.Add("CAPTCHA or DELAY required for ver=2");
            }
        }

        checker.CheckHasArgAllowed(TStrings::FULL_INFO, TBlackboxFlags::FullInfo);
        checker.CheckHasArgAllowed(TStrings::FIND_BY_PHONE_ALIAS, TBlackboxFlags::FindByPhoneAlias);
        checker.CheckHasArgAllowed(TStrings::ALLOW_SCHOLAR, TBlackboxFlags::ScholarLogin);

        TPartitionsHelper::CheckGrants(Blackbox_.PartitionsSettings(), checker);
        TBaseResultHelper::CheckGrants(
            Blackbox_.DbFieldSettings(),
            Blackbox_.AttributeSettings(),
            TConsumer::ERank::HasCred,
            checker);

        return checker;
    }

    std::unique_ptr<TLoginResult>
    TLoginProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& remoteIp = Request_.GetRemoteAddr();

        TUtils::CheckUserIpArg(Request_);

        // get (optional) uid and sanitize as uint if it is not empty
        Uid_ = TUtils::GetUIntArg(Request_, TStrings::UID);

        TUtils::CheckUserTicketAllowed(Request_);

        // Check for password argument - must be present
        Password_ = TUtils::GetCheckedArg(Request_, TStrings::PASSWORD);
        Login_ = Request_.GetArg(TStrings::LOGIN);

        // Requested API version
        TString versionStr = TUtils::GetUIntArg(Request_, TStrings::API_VERSION);
        int apiVersion = versionStr ? IntFromString<int, 10>(versionStr) : 1;

        if (apiVersion > 1) {
            VerComment_ = NUtils::CreateStr("ver=", versionStr);

            // Is the caller capable to resist bruteforce by showing CAPTCHA or delaying the response?
            CanResist_ = true;
        } else {
            // version 1, scholar login not supported
            if (Request_.HasArg(TStrings::ALLOW_SCHOLAR)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Scholar authorization is not supported for ver=1";
            }
        }

        if (!Uid_.empty()) {
            HaveLogin_ = false;
        } else if (!Login_.empty()) {
            switch (TUtils::SanitizeLogin(Login_)) {
                case ESanitizeStatus::Ok:
                    break;
                case ESanitizeStatus::TooLong: {
                    TLoginStatus status(TLoginStatus::PAIR_NOT_FOUND, "Login is too long", apiVersion);
                    return GetErrorLogin(status);
                }
                case ESanitizeStatus::InvalidChars: {
                    TLog::Debug("BlackBox: invalid characters in login <%s>, ignored.", Login_.c_str());
                    TLoginStatus status(TLoginStatus::PAIR_NOT_FOUND, "Invalid characters in login", apiVersion);
                    return GetErrorLogin(status);
                }
            }

            if (Login_.find('@') == Login_.size() - 1) {
                TLog::Debug("BlackBox: empty domain part in login <%s>, ignored.", Login_.c_str());
                TLoginStatus status(TLoginStatus::PAIR_NOT_FOUND, "Empty domain part in login", apiVersion);
                return GetErrorLogin(status);
            }

            HaveLogin_ = true;
        } else {
            TLoginStatus status(TLoginStatus::PAIR_NOT_FOUND, "Empty login and uid", apiVersion);
            return GetErrorLogin(status);
        }

        InitParameters();

        // login value to be written to the badauth_login table
        TString badauthLogin = GetBadauthLogin();
        bool getBadauthCounts = TUtils::GetBoolArg(Request_, TStrings::GET_BADAUTH_COUNTS);

        // The real work starts HERE
        //
        // Unlike the status BlackBox traditionally returns to the caller,
        // this value is not overridden by badauth and other high-level logic
        // so it shows actual knowledge of the login-password in question.

        // Encapsulates both real and conventional statuses, as well as bruteforce
        // resist policy (SHOW_CAPTCHA and delay); initially, is set to
        // (real=UNKNOWN,conventional=INVALID)
        TLoginStatus retStatus(apiVersion);
        TBadauthHelper badauth = Blackbox_.Badauth()->FactoryWrapper(badauthLogin, UserIp_, PasswdHash_, AuthType_);

        if (!ProcessRestrict(badauth, consumer, remoteIp, retStatus)) {
            std::unique_ptr<TLoginResult> loginResult = GetErrorLogin(retStatus);
            if (getBadauthCounts) {
                loginResult->BadauthCounts = TBadauthCountersChunk();
                badauth.Counts().GetValues(*loginResult->BadauthCounts);
            }
            badauth.IncrementCounters(AuthlogComment_, Password_.size());
            return loginResult;
        }

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId(), Sids_, ForceSid2_);
        TUtils::CheckFindByPhoneAliasArg(Request_, fetcher);

        // Now do check login/password and fetch any dbfields requested

        TBaseResultHelper baseResult(conv, Blackbox_, Request_);
        // SOX: создаем объект, проверяющий состояние строгой парольной политики
        TStrongPwdHelper strongPwd(fetcher);

        TFetchLoginStatus loginStatus;
        const TDbProfile* profile = FetchLoginDbFields(fetcher, loginStatus);

        // SOX: если включена строгая парольная политика - проверяем требуемые ограничения
        bool expired = strongPwd.PasswdExpired(profile, Blackbox_.StrongPwdExpireTime()) || loginStatus.PasswordChangeRequired;

        retStatus.SetScopeComment(loginStatus.Comment);

        // Translate true status to legacy status, format various comment strings
        retStatus.Assign(loginStatus.Status, ShallRestrict_, CanResist_, consumer.CanCaptcha(), expired);

        std::unique_ptr<TLoginResult> loginResult = std::make_unique<TLoginResult>();

        loginResult->TotpCheckTime = loginStatus.TotpCheckTime;
        loginResult->ConnectionId = loginStatus.ConnectionId;
        loginResult->AllowedSecondSteps = loginStatus.AllowedSecondSteps;
        if (getBadauthCounts) {
            loginResult->BadauthCounts = TBadauthCountersChunk();
            badauth.Counts().GetValues(*loginResult->BadauthCounts);
        }

        bool showFullInfo = TUtils::GetBoolArg(Request_, TStrings::FULL_INFO);
        if (retStatus == TLoginStatus::PAIR_VALID || retStatus == TLoginStatus::PAIR_SECOND_STEP || (showFullInfo && profile)) {
            loginResult->Uid = TUidHelper::Result(profile, false, profile->PddDomItem());

            baseResult.FillResults(*loginResult, profile);

            if (RestrictSession_) {
                loginResult->RestrictSession = RestrictSession_;
            }
            if (ScholarSession_) {
                loginResult->ScholarSession = true;
            }
        }

        loginResult->Comment = retStatus.Comment();
        loginResult->ApiVersion = retStatus.ApiVersion();
        if (retStatus.ApiVersion() > 1) {
            loginResult->LoginStatus = retStatus.AccountStatus();
            loginResult->LoginStatusStr = TLoginStatus::AccountStatusName(retStatus.AccountStatus());
            loginResult->PasswordStatus = retStatus.PasswordStatus();
            loginResult->PasswordStatusStr = TLoginStatus::PasswordStatusName(retStatus.PasswordStatus());
            if (!ScholarSession_) {
                loginResult->ResistPolicy = (TLoginResult::EResistPolicy)retStatus.Policy();
                if (retStatus.Policy() == TLoginStatus::resist_Delay) {
                    loginResult->ResistDelay = retStatus.Delay();
                }
            }
        } else {
            loginResult->LoginStatus = retStatus.GetLegacyStatus();
            loginResult->LoginStatusStr = retStatus.LegacyStatusAsString();
        }

        if (TUtils::GetBoolArg(Request_, TStrings::GET_USER_TICKET)) {
            if (retStatus == TLoginStatus::PAIR_VALID) {
                if (!profile) {
                    throw TBlackboxError(TBlackboxError::EType::Unknown) << "Profile cannot be empty";
                }
                NTicketSigner::TUserSigner us;
                ui64 uid = TUtils::ToUInt(profile->Uid(), TStrings::UID);
                us.AddUid(uid);
                us.SetDefaultUid(uid);
                us.SetEntryPoint(consumer.GetClientId());
                us.SetEnv(Blackbox_.UserTicketEnv());
                us.AddScope("bb:password");
                NTvmCommon::TPrivateKey::TKey k = Blackbox_.TvmPrivateKeys().GetKey();
                loginResult->UserTicket = us.SerializeV3(
                    *k,
                    time(nullptr) + Blackbox_.UserTicketTtl());
            }
        }

        // Log this
        LoginLog(retStatus.AuthFlag(), remoteIp);
        if (retStatus.AuthFlag() == TAuthLog::OK) {
            // limit successful auth records to keep history smaller
            if (badauth.Counts().AuthLogInLimits()) {
                AuthLog(retStatus.AuthFlag(), AuthlogComment_);
                badauth.IncrementAuthlog(AuthlogComment_);
            }
        } else {
            AuthLog(retStatus.AuthFlag(), AuthlogComment_);
        }

        // Write failed attempts to badauth (writes only if the limit is not exceeded yet)
        if (loginStatus.Second() == TLoginStatus::pwd_Bad || loginStatus.First() == TLoginStatus::acc_Not_Found) {
            if (!ScholarSession_) {
                badauth.IncrementCounters(AuthlogComment_, Password_.size());
            }
            // report bruteforce if needed
            BruteforceLog(badauth.Counts());
        }

        return loginResult;
    }

    void TLoginProcessor::InitParameters() {
        TPartitionsHelper::ParsePartitionArg(
            Blackbox_.PartitionsSettings(),
            Request_,
            TPartitionsHelper::TSettings{
                .Method = "login",
                .ForbidArray = true,
                .ForbidNonDefault = true,
            });

        // Check for the sid argument - optional (and is ignored in the "internal"
        // LDAP-based version)
        Sid_ = Request_.GetArg(TStrings::SID); // is sanitized later in DbFetcher
        if (Sid_ == TStrings::BANK_SID) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "unsupported sid=" << Sid_;
        }

        if (HaveLogin_) {
            // Save original login value for the purpose of writing to auth.log
            AuthLogin_ = TUtils::TolowerUtf(Login_);
            NUtils::EscapeEol(Login_);

            // Do some (login,sid) magic: we may need to strip off @domain.tld part,
            // substitute actual sids value for a macro and decide whether this request
            // goes to Yandex users or mail-4-domains (PDD) users. In the latter case
            // we also learn a few things about the domain in question.
            LoginType_ = Blackbox_.TranslateLoginSids(Login_, Sid_, PddDomain_, Request_.GetArg(TStrings::COUNTRY));

            if (!Sid_.empty()) {
                Sids_ = NUtils::ToVector(Sid_, ',');
            }
            ForceSid2_ = LoginType_ == ELoginType::Yandex_Force_Sid2 ||
                         LoginType_ == ELoginType::Hosted;
            if (LoginType_ == ELoginType::Hosted) {
                Mail4Domains_ = true;
            }

            // Next, for Yandex (as opposed to PDD) logins replace '.' with '-'; internally
            // this function checks to see that login is not an e-mail and for e-mails
            // does nothing.
            if (LoginType_ != ELoginType::Hosted && Sid_ != TStrings::FEDERAL_SID) {
                TUtils::DotsToHyphens(Login_, false);
            }
        } else { // no login
            // uid should be already sanitized
            if (Blackbox_.IsMail4domainsUid(Uid_)) {
                Mail4Domains_ = true;
                ForceSid2_ = true;
                LoginType_ = ELoginType::Hosted;
            } else {
                Mail4Domains_ = false;
                LoginType_ = ELoginType::Yandex;
            }
        }

        InitOptionalParams();

        // Compute password's digest (for logging and badauth)
        PasswdHash_.assign(PasswordHash(Password_, Blackbox_.PasswordHashSecret()));

        // User IP must be present among parameters in all remaining cases.
        UserIp_ = TUtils::GetUserIpArg(Request_); // throws on insanity
        TUtils::GetUserPortArg(Request_);         // not used now, but let it be valid for future
    }

    void TLoginProcessor::InitOptionalParams() {
        // Determine other (optional) parameters written to auth.log
        Referer_ = Request_.HasArg(REFERER) ? Request_.GetArg(REFERER) : Request_.GetHeader(REFERER);
        if (Referer_.size() > 500) {
            Referer_.resize(500);
        }
        NUtils::EscapeEol(Referer_);

        UserAgent_ = Request_.HasArg(USERAGENT) ? Request_.GetArg(USERAGENT) : Request_.GetHeader(USER_AGENT_HEADER);
        if (UserAgent_.size() > 500) {
            UserAgent_.resize(500);
        }
        NUtils::EscapeEol(UserAgent_);

        Retpath_ = Request_.HasArg(RETPATH) ? Request_.GetArg(RETPATH) : Request_.GetHeader(X_RETPATH_HEADER);
        NUtils::EscapeEol(Retpath_);

        Ssl_ = Request_.HasArg(CGI_SSL) ? Request_.GetArg(CGI_SSL) : Request_.GetHeader(SSL_HEADER);
        NUtils::EscapeEol(Ssl_);

        YandexUid_ = Request_.HasArg(YANDEXUID) ? Request_.GetArg(YANDEXUID) : Request_.GetCookie(YANDEXUID);

        // Try determine original IP in case user IP is in fact a proxy's IP
        ProxyIp_ = CheckForProxy(Request_);
        NUtils::EscapeEol(ProxyIp_);

        // from and authtype are optional; if present, are used for logging
        AuthType_ = Request_.GetArg(AUTH_TYPE);

        if (AuthType_.empty()) { // empty authtype is allowed, but we need to see in logs
            TLog::Info("BlackBox: invalid authtype in request: '%s'", AuthType_.c_str());
        }

        if (!Blackbox_.IsAllowedAuthtype(AuthType_)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "unsupported authtype: " << InvalidValue(AuthType_);
        }

        AllowScholar_ = TUtils::GetBoolArg(Request_, TStrings::ALLOW_SCHOLAR);
    }

    bool TLoginProcessor::ProcessRestrict(TBadauthHelper& badauth,
                                          const TConsumer& consumer,
                                          const TString& remoteIp,
                                          TLoginStatus& status) {
        // shallRestrict
        //
        // Shall we check badauth? if so, are limits excedeeded?
        if (consumer.CanCaptcha() && Request_.GetArg(CAPTCHA) == TStrings::NO) {
            CaptchaEntered_ = true;
            LoginlogComment_.assign(CAPTCHA_ENTERED);
        }
        bool respectCaptcha = CaptchaEntered_ && status.ApiVersion() < 2;

        if (respectCaptcha ||                                // captcha already entered
            consumer.LoginSafety() == TConsumer::login_Weak) // consumer doesn't need badauth
        {
            ShallRestrict_ = false;
        } else {
            ShallRestrict_ = !HaveLoginAttempts(badauth, consumer.CanCaptcha()); // read counters and check limits
        }

        // May use this comment later
        status.FormatRestrictComment(AuthlogComment_, LoginlogComment_);

        if (ShallRestrict_ && !CanResist_) { // ver=1
            status.Assign(TLoginStatus::PAIR_UNKNOWN, ShallRestrict_, CanResist_, consumer.CanCaptcha());
            LoginLog(status.AuthFlag(), remoteIp);
            AuthLog(status.AuthFlag(), AuthlogComment_);
            return false;
        }

        if (LoginType_ == ELoginType::Alien) {
            status.Assign(TLoginStatus::PAIR_NOT_FOUND, ShallRestrict_, CanResist_, consumer.CanCaptcha());
            LoginLog(status.AuthFlag(), remoteIp);
            AuthLog(status.AuthFlag(), AuthlogComment_);
            return false;
        }

        return true;
    }

    namespace {
        /// convert int value 0..15 to hex char
        inline char Int2hex(int n) {
            if (0 <= n && n <= 9) {
                return '0' + n;
            }
            if (10 <= n && n <= 15) {
                return 'A' + n - 10;
            }

            return ' '; // error
        }

        const TString VALID_LOGIN_CHARS = "@._-+=";
    }

    // prepare login to use in badauth tables
    TString TLoginProcessor::GetBadauthLogin() const {
        if (Login_.empty()) {
            return GetUidLogin(Uid_);
        }

        const size_t len = Login_.size();

        TString res;
        res.reserve(len * 3);

        // encode all non-alnum characters
        char hex[3] = {'%', ' ', ' '};

        for (size_t i = 0; i < len; ++i) {
            const char c = Login_[i];
            if (isalnum(c) || VALID_LOGIN_CHARS.find(c) != TString::npos) {
                res.append(1, c);
            } else {
                hex[1] = Int2hex((c >> 4) & 0xF);
                hex[2] = Int2hex(c & 0xF);
                res.append(hex, 3);
            }
        }

        if (Mail4Domains_ && res.find('@') == TString::npos) {
            res.append(1, '@').append(PddDomain_.AsciiName());
        }

        return res;
    }

    bool TLoginProcessor::HaveLoginAttempts(TBadauthHelper& badauth, bool canCaptcha) {
        Whitelisted_ = Blackbox_.Badauth() &&
                       (Blackbox_.Badauth()->IsWhitelisted(UserIp_) ||
                        Blackbox_.Badauth()->IsWhitelisted(ProxyIp_));

        badauth.GetTotalCounts(AuthlogComment_);

        TBadauthCounts& cnts = badauth.Counts();

        if (!cnts.IsOk()) {
            return true; // ignore failures
        }

        // first check if already banned by repeated (login,pwd,ip,authtype) tuple
        TBadauthCheckResult check = cnts.RepeatInLimits(AuthlogComment_);
        if (!check.IsOk()) {
            if (Blackbox_.StatboxLogger()) {
                Blackbox_.StatboxLogger()->LogBadauthBan(Login_, Uid_, YandexUid_, check.Rule, check.Count, UserIp_, canCaptcha);
            }
            return false; // banned, but not bruteforce
        }

        // Check experimental logic with half ipv6 address used and log result
        TString tmpComment;
        TBadauthCheckResult checkHalfIp = cnts.CountsInLimitsHalfIp(Whitelisted_, Password_.size(), tmpComment);
        if (!checkHalfIp.IsOk()) {
            bool shouldBanByHalfIp = TExperiment::Get().IsInHalfIpExperiment(UserIp_);
            if (Blackbox_.StatboxLogger()) {
                Blackbox_.StatboxLogger()->LogBadauthBan(Login_, Uid_, YandexUid_, checkHalfIp.Rule, checkHalfIp.Count, UserIp_, canCaptcha, !shouldBanByHalfIp);
            }
            if (shouldBanByHalfIp) {
                // keep comment that we got when checking by half ip
                AuthlogComment_ = std::move(tmpComment);
                return false;
            }
        }

        // if counts does not exceed limits return true, we have login attempts
        TBadauthCheckResult checkAll = cnts.CountsInLimits(Whitelisted_, Password_.size(), AuthlogComment_);
        if (!checkAll.IsOk()) {
            if (Blackbox_.StatboxLogger()) {
                Blackbox_.StatboxLogger()->LogBadauthBan(Login_, Uid_, YandexUid_, checkAll.Rule, checkAll.Count, UserIp_, canCaptcha);
            }
            return false;
        }

        return true;
    }

    void TLoginProcessor::AuthLog(TAuthLog::EFlag status, const TString& comment) const {
        // whiteliste, ver
        if (Blackbox_.AuthLogger() && !Uid_.empty()) {
            const TString& sslStr = (Ssl_.empty() || Ssl_ == TStrings::ZERO) ? TStrings::EMPTY : SSL_COMMENT;

            TString realComment = NUtils::CreateStrExt(comment.size() + 15, VerComment_); // reserve for 'whitelisted' + separators
            if (Whitelisted_) {
                NUtils::AppendSeparated(realComment, ';', "whitelisted");
            }
            NUtils::AppendSeparated(realComment, ';', comment);

            Blackbox_.AuthLogger()->Write(Uid_, AuthLogin_, Sid_, AuthType_, status, realComment, sslStr, CaptchaEntered_, UserIp_, ProxyIp_, YandexUid_, Referer_, Retpath_, UserAgent_);
        }
    }

    void TLoginProcessor::LoginLog(TAuthLog::EFlag status, const TString& serviceIp) const {
        if (Blackbox_.LoginLogger()) {
            TString realComment = NUtils::CreateStrExt(AuthlogComment_.size() + LoginlogComment_.size() + 15, VerComment_); // reserve for 'whitelisted' + separators
            if (Whitelisted_) {
                NUtils::AppendSeparated(realComment, ';', "whitelisted");
            }
            NUtils::AppendSeparated(realComment, ';', AuthlogComment_);
            NUtils::AppendSeparated(realComment, ';', LoginlogComment_);

            Blackbox_.LoginLogger()->Error("uid=%s login=%s password=%s ip_from=%s ip_prox=%s ip_internal=%s flag=%d authtype=%s comment=\"%s\" reqid=%s",
                                           Uid_.c_str(),
                                           AuthLogin_.c_str(),
                                           PasswdHash_.c_str(),
                                           UserIp_.c_str(),
                                           ProxyIp_.c_str(),
                                           serviceIp.c_str(),
                                           status,
                                           AuthType_.c_str(),
                                           realComment.c_str(),
                                           NUtils::GetThreadLocalRequestId().c_str());
        }
    }

    void TLoginProcessor::BruteforceLog(TBadauthCounts& counts) const {
        if (ShallRestrict_) {
            return; // already banned
        }

        TString msg;
        TBadauthCheckResult check = counts.CountsInLimits(Whitelisted_, Password_.size(), msg);

        // Here we rely on the fact that badauth counters already were incremented in appropriate way
        // so if check is not OK now, it means it was OK in the beginning but now it became banned
        // also, repeated attempt is not considered bruteforce, since we need only one event for each bruteforce
        // so if it is first attempt with any counter over limit - we log it as bruteforce start
        if (counts.RepeatCount < 2 && !check.IsOk()) {
            // msg already filled up with correct message
            AuthLog(TAuthLog::BRUTEFORCE, msg);
        }
    }

    time_t TLoginProcessor::LastValidTotpTime(const TDbValue* tm) {
        if (tm && !tm->Value.empty()) {
            try {
                return TUtils::ToTime(tm->Value);
            } catch (const std::logic_error&) {
                TLog::Error("BlackBox: bad lexical cast on lastValidTotpTime=<%s>",
                            tm->Value.c_str());
            }
        }
        return 0;
    }

    const TDbProfile*
    TLoginProcessor::FetchLoginDbFields(TDbFetcher& fetcher, TFetchLoginStatus& status) {
        if (Password_.size() > 255) {
            TString p(Password_);
            if (p.size() > 1024) {
                p.resize(1024);
            }
            TLog::Debug("BlackBox: too long password=<%s>, return BAD_PASSWORD.", p.c_str());
            status = TLoginStatus::PAIR_NOT_FOUND;
            return nullptr;
        }

        const TDbIndex loginValue = fetcher.AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN);
        const TDbIndex isAvailable = fetcher.AddAttr(TAttr::ACCOUNT_IS_AVAILABLE);
        const TDbIndex encryptedPwd = fetcher.AddAttr(TAttr::PASSWORD_ENCRYPTED);
        const TDbIndex changeReason = fetcher.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON);
        const TDbIndex createRequired = fetcher.AddAttr(TAttr::PASSWORD_CREATING_REQUIRED);
        const TDbIndex totpSecret = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET);
        const TDbIndex lastValidTime = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_CHECK_TIME);
        const TDbIndex enableAppPwd = fetcher.AddAttr(TAttr::ACCOUNT_ENABLE_APP_PASSWORD);
        const TDbIndex glogoutItem = fetcher.AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME);
        const TDbIndex revokeAppPassword = fetcher.AddAttr(TAttr::REVOKER_APP_PASSWORDS);
        const TDbIndex allowWebdavByPassword = fetcher.AddAttr(TAttr::ACCOUNT_ALLOW_WEBDAV_BY_PASSWORD);
        const TDbIndex allowCalendarByPassword = fetcher.AddAttr(TAttr::ACCOUNT_ALLOW_CALENDAR_BY_PASSWORD);
        const TDbIndex scholarAlias = fetcher.AddAlias(TAlias::SCHOLAR);
        const TDbIndex scholarPwd = AllowScholar_ ? fetcher.AddAttr(TAttr::ACCOUNT_SCHOLAR_PASSWORD) : -1;

        if (!HaveLogin_) {
            fetcher.FetchByUid(Uid_);
        } else {
            fetcher.FetchByLogin(Login_, PddDomain_.Id(), Sids_, PddDomain_.DefaultUid(), AllowScholar_);
        }

        const TDbProfile* profile = fetcher.NextProfile();
        if (nullptr == profile) {
            status = TLoginStatus::PAIR_NOT_FOUND;
            return nullptr;
        }

        if (HaveLogin_) { // update uid if we came with login
            Uid_ = profile->Uid();
        } else { // and update login if we came with uid
            Login_ = AuthLogin_ = profile->Get(loginValue)->Value;
        }

        const TString& scholarLogin = profile->Get(scholarAlias)->Value;
        if (HaveLogin_ && scholarLogin == Login_) {
            // user found by scholar login, check only scholar password
            if (AllowScholar_) {
                status = CheckPasswordScholar(profile->Get(scholarPwd)->Value);
                const bool available = profile->Get(isAvailable)->AsBoolean();
                if (!available) {
                    status.SetFirst(TLoginStatus::acc_Disabled);
                }
                ScholarSession_ = true;
                return profile;
            }

            status = TLoginStatus::PAIR_NOT_FOUND;
            return nullptr;
        }

        const bool applicationPasswords = profile->Get(enableAppPwd)->AsBoolean();
        const bool twoFAEnabled = !profile->Get(totpSecret)->Value.empty();
        // for authtype in ['web','oauthcreate','verify'] always use web password
        const bool passwordOnlyAuthtype = AuthType_.empty() || AuthType_ == WEB_AUTH_TYPE || AuthType_ == OAUTHCREATE_AUTH_TYPE || AuthType_ == VERIFY_AUTH_TYPE;

        const bool checkAsOAuth = (twoFAEnabled || applicationPasswords) && !passwordOnlyAuthtype;

        // if password change required you can login only from web or oauthcreate, other authtypes will fail with bad password
        if (AuthType_ != WEB_AUTH_TYPE && AuthType_ != OAUTHCREATE_AUTH_TYPE) {
            if (profile->Get(changeReason)->AsBoolean()) {
                if (checkAsOAuth) {
                    Blackbox_.OauthConfig().LogOAuthErrorByAlias(Uid_,
                                                                 TOAuthStatboxMsg::USER_PWD_CHANGE,
                                                                 UserIp_,
                                                                 Request_.GetConsumerFormattedName(),
                                                                 Request_.GetRemoteAddr());
                }
                status = TLoginStatus::PAIR_BAD_PASSWORD;
                return profile;
            }

            if (profile->Get(createRequired)->AsBoolean()) {
                if (checkAsOAuth) {
                    Blackbox_.OauthConfig().LogOAuthErrorByAlias(Uid_,
                                                                 TOAuthStatboxMsg::USER_PWD_CREATE,
                                                                 UserIp_,
                                                                 Request_.GetConsumerFormattedName(),
                                                                 Request_.GetRemoteAddr());
                }
                status = TLoginStatus::PAIR_BAD_PASSWORD;
                return profile;
            }
        }

        if (Blackbox_.IsYaTeam()) { // check password for yandex-team
            std::optional<TString> allowedSecondSteps;
            status = YandexTeamCheckPassword(allowedSecondSteps, status.PasswordChangeRequired);
            status.AllowedSecondSteps = std::move(allowedSecondSteps);
            return profile;
        }

        if (checkAsOAuth) { // check password as oauth token
            status = CheckPasswordAsOAuth(profile->Get(glogoutItem)->AsTime(),
                                          profile->Get(revokeAppPassword)->AsTime(),
                                          profile->PddDomItem().Glogout().TimeT(),
                                          status.Comment,
                                          status.ConnectionId);
        } else { // check portal password (hash in db or TOTP for 2FA users)

            if (AuthType_ == WEBDAV_AUTH_TYPE && !profile->Get(allowWebdavByPassword)->AsBoolean()) {
                TLog::Info() << "BlackBox: webdav login with portal password blocked for uid " << Uid_;
                status = TLoginStatus::PAIR_BAD_PASSWORD;
                return profile;
            }
            if (Blackbox_.BlockCalendarByPassword() &&
                AuthType_ == CALENDAR_AUTH_TYPE &&
                !profile->Get(allowCalendarByPassword)->AsBoolean())
            {
                TLog::Info() << "BlackBox: calendar login with portal password blocked for uid " << Uid_;
                status = TLoginStatus::PAIR_BAD_PASSWORD;
                return profile;
            }

            const TString& dbSecret = profile->Get(totpSecret)->Value;

            if (!dbSecret.empty()) { // check password as totp secret
                if (!Blackbox_.TotpEncryptor()) {
                    TLog::Error("BlackBox: TOTP encryption is not initialized, login failed for user %s", Login_.c_str());
                    status = TLoginStatus::PAIR_BAD_PASSWORD;
                } else {
                    try {
                        ui64 u = TUtils::ToUInt(Uid_, TStrings::UID);
                        TTotpProfile totpProfile = Blackbox_.TotpEncryptor()->Decrypt(u, dbSecret);
                        TString totpPassword = TUtils::NormalizeTotp(Password_);

                        NTotp::TTotpResult r;
                        const TString& secretId = TUtils::GetUIntArg(Request_, TStrings::SECRET_ID);
                        if (secretId.empty()) {
                            r = totpProfile.Check(Blackbox_.TotpService(),
                                                  totpPassword,
                                                  LastValidTotpTime(profile->Get(lastValidTime)));
                        } else {
                            r = totpProfile.Check(Blackbox_.TotpService(),
                                                  totpPassword,
                                                  LastValidTotpTime(profile->Get(lastValidTime)),
                                                  TUtils::ToUInt(secretId, TStrings::SECRET_ID));
                        }

                        status = r.Succeeded() ? TLoginStatus::PAIR_VALID : TLoginStatus::PAIR_BAD_PASSWORD;
                        status.TotpCheckTime = r.TotpTime();

                        // on success update last TOTP check time in db
                        if (r.Succeeded() &&
                            Blackbox_.PassportWrapper() &&
                            !Blackbox_.PassportWrapper()->UpdateTotpCheckTime(Uid_, r.TotpTime()))
                        {
                            status = TLoginStatus::PAIR_BAD_PASSWORD;
                            status.Comment = "failed to update TOTP check time";

                            NUtils::AppendSeparated(AuthlogComment_, ';', "totp_check_time update failed");
                        }
                    } catch (std::invalid_argument& e) {
                        TLog::Error("%s", e.what());
                        status = TLoginStatus::PAIR_BAD_PASSWORD;
                    }
                }
            } else { // check as plain db password
                const TDbValue* pwd = profile->Get(encryptedPwd);
                if (pwd->Exists) {
                    if (Blackbox_.PasswordChecker().PasswordMatches(
                            Password_,
                            pwd->Value,
                            Uid_)) {
                        status = TLoginStatus::PAIR_VALID;
                    } else {
                        status = TLoginStatus::PAIR_BAD_PASSWORD;
                    }
                } else {
                    status = TLoginStatus::PAIR_BAD_PASSWORD;
                }
            }
        }

        const bool available = profile->Get(isAvailable)->AsBoolean();
        if (!available) {
            if (checkAsOAuth) {
                Blackbox_.OauthConfig().LogOAuthErrorByAlias(Uid_,
                                                             TOAuthStatboxMsg::USER_DISABLED,
                                                             UserIp_,
                                                             Request_.GetConsumerFormattedName(),
                                                             Request_.GetRemoteAddr());
            }
            status.SetFirst(TLoginStatus::acc_Disabled);
        }

        return profile;
    }

    TLoginStatus::TStatusPair TLoginProcessor::CheckPasswordScholar(const TString& pwdHash) {
        if (pwdHash && Blackbox_.PasswordChecker().PasswordMatches(Password_, pwdHash, Uid_)) {
            return TLoginStatus::PAIR_VALID;
        }
        return TLoginStatus::PAIR_BAD_PASSWORD;
    }

    TLoginStatus::TStatusPair TLoginProcessor::YandexTeamCheckPassword(std::optional<TString>& allowedSecondSteps, bool& passwordChangeRequired) {
        // yandex-team: check password in Perimeter
        TLoginStatus::TStatusPair status = TLoginStatus::PAIR_BAD_PASSWORD;
        if (Blackbox_.PerimeterWrapper()) {
            status = CheckPasswordPerimeter(allowedSecondSteps, passwordChangeRequired);
        }
        return status;
    }

    TLoginStatus::TStatusPair TLoginProcessor::CheckPasswordAsOAuth(time_t glogoutTime,
                                                                    time_t revokeAppPasswordTime,
                                                                    time_t domainGlogout,
                                                                    TString& comment,
                                                                    TString& connectionId) {
        TOAuthSingleFetcher oauthFetcher(Blackbox_.OauthConfig(), UserIp_, Request_.GetConsumerFormattedName(), Request_.GetRemoteAddr());

        std::unique_ptr<TOAuthTokenInfo> impl = oauthFetcher.CheckTokenByAlias(Password_, Uid_);

        NUtils::AppendSeparated(AuthlogComment_, ';', "AP=1");

        if (impl) {
            NUtils::AppendSeparated(AuthlogComment_, ';', impl->AuthLogComment());

            // need to check scopes for given authtype
            if (!AuthType_.empty()) {
                TString scope = TStrings::APP_PASSWORD_PREFIX + AuthType_;
                if (!impl->HasScopes(scope)) {
                    Blackbox_.OauthConfig().LogOAuth(*impl,
                                                     UserIp_,
                                                     TOAuthStatboxMsg::ERROR,
                                                     TOAuthStatboxMsg::SCOPES_WRONG,
                                                     Request_.GetConsumerFormattedName(),
                                                     Request_.GetRemoteAddr());
                    comment.assign("Application password doesn't have scope " + scope);
                    return TLoginStatus::PAIR_BAD_PASSWORD;
                }
            }

            // need to check glogout and passwords revoke time
            time_t issuedTime = impl->IssueTimeTs;
            if (issuedTime < glogoutTime) {
                Blackbox_.OauthConfig().LogOAuth(*impl,
                                                 UserIp_,
                                                 TOAuthStatboxMsg::ERROR,
                                                 TOAuthStatboxMsg::USER_DISABLED,
                                                 Request_.GetConsumerFormattedName(),
                                                 Request_.GetRemoteAddr());
                comment.assign("Application password expired - user globally logged out");
                return TLoginStatus::PAIR_BAD_PASSWORD;
            }
            if (issuedTime < revokeAppPasswordTime) {
                Blackbox_.OauthConfig().LogOAuth(*impl,
                                                 UserIp_,
                                                 TOAuthStatboxMsg::ERROR,
                                                 "user.revoked_app_password",
                                                 Request_.GetConsumerFormattedName(),
                                                 Request_.GetRemoteAddr());
                comment.assign("Application password expired - user globally revoked app passwords");
                return TLoginStatus::PAIR_BAD_PASSWORD;
            }
            if (issuedTime < domainGlogout) {
                Blackbox_.OauthConfig().LogOAuth(*impl,
                                                 UserIp_,
                                                 TOAuthStatboxMsg::ERROR,
                                                 TOAuthStatboxMsg::USER_DISABLED,
                                                 Request_.GetConsumerFormattedName(),
                                                 Request_.GetRemoteAddr());
                comment.assign("Application password expired - domain globally logged out");
                return TLoginStatus::PAIR_BAD_PASSWORD;
            }

            connectionId.assign("t:");
            connectionId.append(impl->TokenId);

            return TLoginStatus::PAIR_VALID;
        }

        return TLoginStatus::PAIR_BAD_PASSWORD;
    }

    TLoginStatus::TStatusPair
    TLoginProcessor::CheckPasswordPerimeter(std::optional<TString>& allowedSecondSteps, bool& passwordChangeRequired) {
        TString msg;
        TString requestId;

        ENetworkKind ipKind = Blackbox_.CheckYandexIp(UserIp_, Request_, Uid_);

        ui64 uid = TUtils::ToUInt(Uid_, TStrings::UID);
        bool isRobot = Blackbox_.StaffInfo()->IsRobot(uid);

        TPerimeterWrapper::EStatus status = Blackbox_.PerimeterWrapper()->Login(
            Login_, Password_, UserIp_, AuthType_,
            ipKind, isRobot, requestId, msg, allowedSecondSteps, passwordChangeRequired);

        // store request id to login log
        NUtils::AppendSeparated(LoginlogComment_, "; ", "request_id=").append(requestId);

        switch (status) {
            case TPerimeterWrapper::LoginOk:
                RestrictSession_ = ipKind != ENetworkKind::Internal;
                return TLoginStatus::PAIR_VALID;
            case TPerimeterWrapper::SecondStep:
                return TLoginStatus::PAIR_SECOND_STEP;
            case TPerimeterWrapper::Rejected:
                TLog::Debug() << "Password check for user '" << Login_
                              << "' rejected by Perimeter with message '" << msg
                              << "' perimeter_request_id=" << requestId
                              << ", user_ip " << UserIp_ << '(' << ipKind << ')';
                return TLoginStatus::PAIR_BAD_PASSWORD;
            case TPerimeterWrapper::Error:
                TLog::Warning() << "Password check for user '" << Login_
                                << "' failed with message '" << msg
                                << "' perimeter_request_id=" << requestId
                                << ", user_ip " << UserIp_ << '(' << ipKind << ')';
                return TLoginStatus::PAIR_BAD_PASSWORD;
        };

        return TLoginStatus::PAIR_UNKNOWN;
    }

    std::unique_ptr<TLoginResult> TLoginProcessor::GetErrorLogin(const TLoginStatus& status) {
        std::unique_ptr<TLoginResult> loginResult = std::make_unique<TLoginResult>();
        loginResult->Comment = status.Comment();
        loginResult->ApiVersion = status.ApiVersion();

        if (status.ApiVersion() > 1) {
            loginResult->LoginStatus = status.AccountStatus();
            loginResult->LoginStatusStr = TLoginStatus::AccountStatusName(status.AccountStatus());
            loginResult->PasswordStatus = status.PasswordStatus();
            loginResult->PasswordStatusStr = TLoginStatus::PasswordStatusName(status.PasswordStatus());
        } else {
            loginResult->LoginStatus = status.GetLegacyStatus();
            loginResult->LoginStatusStr = status.LegacyStatusAsString();
        }

        return loginResult;
    }

    inline TString TLoginProcessor::PasswordHash(const TStringBuf pwd, const TStringBuf secret) {
        TString out = NUtils::Bin2hex(NUtils::TCrypto::Sha256(NUtils::CreateStr(pwd, secret)));
        out.resize(16);
        return out;
    }
}
