#include "loginoccupation.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/domain/domain_fetcher.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/loginoccupation_result.h>

#include <passport/infra/libs/cpp/auth_core/public_id.h>
#include <passport/infra/libs/cpp/auth_core/public_id_encryptor.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    static const TString CENTRAL_DB_QUERY("SELECT uid,type,value FROM aliases WHERE ");
    static const TString CENTRAL_DB_QUERY_ALIASES("(type IN (1,2,3,4,19,20,23) AND value IN (");
    static const TString CENTRAL_DB_QUERY_SYNTH("(type IN (6,10,17,18,21) AND value IN (");
    static const TString CENTRAL_DB_QUERY_MAIL("(type=5 AND value IN (");
    static const TString CENTRAL_DB_QUERY_ALT_DOMAIN("(type=9 AND value IN (");
    static const TString CENTRAL_DB_QUERY_PDD("(type IN (7,8) AND value IN (");

    static const TString RESERVED_LOGINS_QUERY("SELECT '',1,login FROM reserved_logins WHERE login IN (");
    static const TString QUERY_OR(" OR ");
    static const TString ORDER_BY(" ORDER BY uid,type");

    TString TLoginOccupationProcessor::MakeArgList(const TLoginList& l, NDbPool::TBlockingHandle& sqlh) {
        TString result;

        for (const TString& arg : l) {
            NUtils::AppendSeparated(result, ',', "'").append(sqlh.EscapeQueryParam(arg)).append('\'');
        }

        return result;
    }

    TLoginOccupationStatusMap TLoginOccupationProcessor::FindLoginsInDb(
        const TLoginOccupationProcessor::TLoginList& accLogins,
        const TLoginOccupationProcessor::TLoginList& mailLogins,
        const TLoginOccupationProcessor::TLoginList& synthLogins,
        bool isPdd) {
        if (accLogins.empty() && mailLogins.empty() && synthLogins.empty()) { // nothing to do
            return {};
        }

        TString loginsAllArg;
        TString loginsAccArg;
        TString loginsMailArg;
        TString loginsSynthArg;

        // remember the mapping of PDD and AltDomain logins to their original form
        using TPddMap = std::multimap<TString, TString>;
        TPddMap pddMap;
        std::map<TString, TString> altDomainMap;
        std::unique_ptr<NDbPool::TResult> result;

        try {
            NDbPool::TBlockingHandle sqlh(Blackbox_.CentralDb());

            loginsAccArg = TUtils::TolowerUtf(MakeArgList(accLogins, sqlh));
            loginsMailArg = NUtils::TolowerCopy(MakeArgList(mailLogins, sqlh));
            loginsSynthArg = NUtils::TolowerCopy(MakeArgList(synthLogins, sqlh));

            // normalize all non-email accounts at once (convert '.' to '-')
            TUtils::DotsToHyphens(loginsAccArg, false);
            TUtils::DotsToHyphens(loginsSynthArg, false);

            loginsAllArg.append(loginsAccArg);

            if (!isPdd) { // only if non-pdd mode!
                NUtils::AppendSeparated(loginsAllArg, ',', loginsMailArg);
            }

            NUtils::AppendSeparated(loginsAllArg, ',', loginsSynthArg);

            // first, check aliases in centralDb
            TString condition;

            if (!loginsAccArg.empty()) {
                condition = NUtils::CreateStr(
                    CENTRAL_DB_QUERY_ALIASES,
                    loginsAccArg,
                    "))");
            }

            if (!loginsSynthArg.empty()) {
                NUtils::AppendSeparated(condition, QUERY_OR, CENTRAL_DB_QUERY_SYNTH);
                NUtils::Append(condition, loginsSynthArg, "))");
            }

            if (!loginsMailArg.empty()) {
                if (isPdd) { // treat emails as pdd accounts
                    TString domList;

                    for (const TString& login : mailLogins) {
                        size_t atPos = login.find('@');
                        if (atPos == TString::npos || atPos == 0 || atPos + 1 == login.size()) {
                            continue;
                        }

                        TString lowcase = NUtils::TolowerCopy(login);
                        TString loginPart = lowcase.substr(0, atPos);
                        TString domPart = lowcase.substr(atPos + 1);

                        TDomainFetcher::TResult domRes;
                        if (!(domRes = Blackbox_.HostedDomains().Find(domPart)) &&
                            !(domRes = Blackbox_.HostedDomains().FindInDb(domPart))) {
                            continue;
                        }

                        // convert to internal form '<domid>/<login>', and keep the original email in a map
                        TString internalForm = NUtils::CreateStr(domRes.Domain->Id(), '/', sqlh.EscapeQueryParam(loginPart));

                        NUtils::AppendSeparated(domList, ',', "'").append(internalForm).append('\'');

                        pddMap.insert(std::make_pair(internalForm, login));
                    }

                    if (!domList.empty()) {
                        NUtils::AppendSeparated(condition, QUERY_OR, CENTRAL_DB_QUERY_PDD);
                        NUtils::Append(condition, domList, "))");
                    }
                } else { // treat emails as lite aliases or alt domain accounts
                    NUtils::AppendSeparated(condition, QUERY_OR, CENTRAL_DB_QUERY_MAIL);
                    NUtils::Append(condition, loginsMailArg, "))");
                    TString domList;

                    for (const TString& login : mailLogins) {
                        size_t atPos = login.find('@');
                        if (atPos == TString::npos || atPos == 0 || atPos + 1 == login.size()) {
                            continue;
                        }

                        TString loginPart = NUtils::TolowerCopy(login.substr(0, atPos));
                        TString domPart = NUtils::TolowerCopy(login.substr(atPos + 1));

                        // look up in configured alt domains
                        const TYandexDomains::TDomainData* domData = Blackbox_.YandexDomains().Find(domPart);
                        if (domData == nullptr || domData->GetDomid().empty()) {
                            continue;
                        }

                        // don't forget to normalize login part
                        TUtils::DotsToHyphens(loginPart, false);
                        TString internalForm = NUtils::CreateStr(domData->GetDomid(), '/', sqlh.EscapeQueryParam(loginPart));

                        NUtils::AppendSeparated(domList, ',', "'").append(internalForm).append('\'');

                        altDomainMap.insert(std::make_pair(internalForm, login));
                    }

                    if (!domList.empty()) {
                        NUtils::AppendSeparated(condition, QUERY_OR, CENTRAL_DB_QUERY_ALT_DOMAIN);
                        NUtils::Append(condition, domList, "))");
                    }
                }
            }

            TString query;
            if (!condition.empty()) {
                query = NUtils::CreateStr(CENTRAL_DB_QUERY, condition);
            }

            if (!loginsAllArg.empty()) { // also check in reserved logins
                NUtils::Append(query,
                               query.empty() ? "" : " UNION ",
                               RESERVED_LOGINS_QUERY,
                               loginsAllArg,
                               ")");
            }

            if (query.empty()) {
                return {};
            }
            // we need order by type to show uid of lowest found alias type
            // in case of portal and mail alias clash we'll take uid of portal user
            query.append(ORDER_BY);

            // TLog::Error("QUERY: check login centralDb query: %s", query.c_str());

            result = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            throw TDbpoolError("dbpool exception checking logins", e.what());
        }

        TLoginOccupationStatusMap foundLogins;

        for (const NDbPool::TRow& row : result->Table()) {
            const TString& uid = row[0].AsString();
            const TString& type = row[1].AsString();
            const TString& value = row[2].AsString();
            TExperiment::Get().RunAliasCheck();
            if (type == TAlias::PDD_MASTER_LOGIN || type == TAlias::PDD_ALIAS_LOGIN) { // convert PDD back to original login
                std::pair<TPddMap::const_iterator, TPddMap::const_iterator> range = pddMap.equal_range(value);
                for (TPddMap::const_iterator it = range.first; it != range.second; ++it) {
                    foundLogins.emplace(TUtils::TolowerUtf(it->second), TLoginOccupationStatus(TLoginOccupationStatus::Busy, uid));
                }
            } else if (type == TAlias::ALT_DOMAIN_LOGIN) { // convert AltDomain to original form
                std::map<TString, TString>::const_iterator it = altDomainMap.find(value);
                if (it != altDomainMap.end()) {
                    foundLogins.emplace(NUtils::TolowerCopy(it->second), TLoginOccupationStatus(TLoginOccupationStatus::Busy, uid));
                }
            } else if (type == TAlias::PUBLICID || type == TAlias::OLDPUBLICID) { // custom status for public_ic
                foundLogins.emplace(NUtils::TolowerCopy(value), TLoginOccupationStatus(TLoginOccupationStatus::PublicId, uid));
            } else {
                foundLogins.emplace(TUtils::TolowerUtf(value), TLoginOccupationStatus(TLoginOccupationStatus::Busy, uid));
            }
        }

        return foundLogins;
    }

    TLoginOccupationProcessor::TLoginOccupationProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TLoginOccupationProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::LoginOccupation);

        return checker;
    }

    std::unique_ptr<TLoginOccupationResult> TLoginOccupationProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& loginsArg = TUtils::GetCheckedArg(Request_, TStrings::LOGINS);

        bool ignoreStoplist = TUtils::GetBoolArg(Request_, TStrings::IGNORE_STOPLIST);
        bool isPdd = TUtils::GetBoolArg(Request_, TStrings::IS_PDD);

        TLoginList logins;
        NUtils::Transform(loginsArg, ',', [&logins](const TStringBuf buf) {
            if (buf.empty()) {
                return;
            }
            logins.push_back(TString(buf));
            NUtils::Trim(logins.back());
        });

        TLoginOccupationStatusMap statuses;
        // keep references to check the db
        TLoginList accLogins;
        TLoginList mailLogins;
        TLoginList synthLogins;

        // separate plain account logins, mail logins and synthetic logins into 3 different lists
        TLoginList::iterator it = logins.begin();
        while (it != logins.end()) {
            TLoginList::iterator p = it++;

            // minimal sanitizing to skip malformed
            if (p->empty() || TUtils::SanitizeLogin(*p, 128) != ESanitizeStatus::Ok || p->front() == '@' || p->back() == '@') {
                statuses.emplace(*p, TLoginOccupationStatus(TLoginOccupationStatus::Malformed));
                logins.erase(p);
                continue;
            }

            // check emails in db as is
            if (p->find('@') != TString::npos) {
                mailLogins.splice(mailLogins.end(), logins, p);
                continue;
            }

            // not allowed if decrypts as valid public_id, no matter which uid
            if (Blackbox_.PublicIdEncryptor()) {
                NAuth::TPublicId id = Blackbox_.PublicIdEncryptor()->Decrypt(*p);
                if (id.Status() == NAuth::TPublicId::VALID && id.Uid() != 0) {
                    statuses.emplace(*p, TLoginOccupationStatus(TLoginOccupationStatus::PublicId));
                    logins.erase(p);
                    continue;
                }
            }

            // not a email or public_id - normalize and check in stop words
            TString tmpLogin = TUtils::NormalizeLogin(*p);

            if (!ignoreStoplist && Blackbox_.IsStopWord(tmpLogin)) {
                statuses.emplace(*p, TLoginOccupationStatus(TLoginOccupationStatus::Stop));
                logins.erase(p);
            } else {
                if (TUtils::IsSyntheticLogin(tmpLogin)) {
                    synthLogins.splice(synthLogins.end(), logins, p);
                } else {
                    accLogins.splice(accLogins.end(), logins, p);
                }
            }
        }

        TLoginOccupationStatusMap foundLogins = FindLoginsInDb(accLogins, mailLogins, synthLogins, isPdd);

        // join lists back for convenience
        logins.splice(logins.end(), accLogins);
        logins.splice(logins.end(), mailLogins);
        logins.splice(logins.end(), synthLogins);

        for (const TString& login : logins) {
            // normalize and lowercase logins
            TString tmpLogin = TUtils::TolowerUtf(login);
            TUtils::DotsToHyphens(tmpLogin, false);
            TLoginOccupationStatusMap::iterator it = foundLogins.find(tmpLogin);

            if (it != foundLogins.end()) {
                statuses.emplace(login, std::move(it->second));
            } else {
                statuses.emplace(login, TLoginOccupationStatus(TLoginOccupationStatus::Free));
            }
        }

        std::unique_ptr<TLoginOccupationResult> result = std::make_unique<TLoginOccupationResult>();
        result->Statuses = std::move(statuses);

        return result;
    }
}
