#include "phone_bindings.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/phone_bindings_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/phone_bindings_result.h>

#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NBb {
    static const TString PHONE_BINDINGS_CURRENT_QUERY = "SELECT 'current' AS type,number,phone_id,uid,unix_timestamp(bound),flags FROM phone_bindings";
    static const TString CURRENT_TIMECHECK = " AND unix_timestamp(bound)>0 ";
    static const TString PHONE_BINDINGS_UNBOUND_QUERY = "SELECT 'unbound' AS type,number,phone_id,uid,0 as 'unix_timestamp(bound)',flags FROM phone_bindings";
    static const TString UNBOUND_TIMECHECK = " AND unix_timestamp(bound)=0 ";
    static const TString UNION = " UNION ";
    static const TString PHONE_BINDINGS_HISTORY_QUERY = "SELECT 'history' AS type,number,'' AS phone_id,uid,unix_timestamp(bound),0 AS flags FROM phone_bindings_history";

    TPhoneBindingsProcessor::TPhoneBindingsProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TPhoneBindingsProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::PhoneBindings);

        return checker;
    }

    std::unique_ptr<TPhoneBindingsResult> TPhoneBindingsProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& requestType = TUtils::GetCheckedArg(Request_, TStrings::TYPE);

        const bool needCurrent = requestType == TStrings::CURRENT || requestType == TStrings::ALL;
        const bool needUnbound = requestType == TStrings::UNBOUND || requestType == TStrings::ALL;
        const bool needHistory = requestType == TStrings::HISTORY || requestType == TStrings::ALL;

        if (!needCurrent && !needUnbound && !needHistory) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Unknown type argument: " << InvalidValue(requestType);
        }

        const TString& numbersStr = Request_.GetArg(TStrings::NUMBERS);
        const TString& phoneidsStr = Request_.GetArg(TStrings::PHONE_IDS);

        if (requestType != TStrings::CURRENT && numbersStr.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Missing " << TStrings::NUMBERS << " argument, not allowed with type=" << requestType;
        }

        const TString& boundAfter = TUtils::GetUIntArg(Request_, TStrings::BOUND_AFTER);

        const TString whereClause = BuildWhereClause(numbersStr, phoneidsStr, boundAfter);
        const TString query = BuildQuery(whereClause, needCurrent, needUnbound, needHistory);
        // TLog::Error("QUERY: '%s'", query.c_str());

        std::unique_ptr<TPhoneBindingsChunk> chunk = std::make_unique<TPhoneBindingsChunk>();

        // iterate over all shards and join into table result
        std::vector<NDbPool::TNonBlockingHandle> handles;
        for (NDbPool::TDbPool& db : Blackbox_.ShardsMap().GetShards()) {
            NDbPool::TNonBlockingHandle sqlh(db);
            sqlh.SendQuery(query);
            handles.push_back(std::move(sqlh));
        }

        for (auto& sqlh : handles) {
            std::unique_ptr<NDbPool::TResult> res = TUtils::WaitResult(sqlh, "dbpool exception in PhoneBindings");

            for (const NDbPool::TRow& row : res->Table()) {
                TPhoneBindingsChunk::TPhoneBindingsItem item;
                if (row.size() != 6) {
                    throw TBlackboxError(TBlackboxError::EType::Unknown)
                        << "BlackBox internal error: unexpected response from phone_bindings table";
                }

                item.Type = row[0].IsNull() ? TStrings::EMPTY : row[0].AsString();
                item.Number = row[1].IsNull() ? TStrings::EMPTY : row[1].AsString();
                item.PhoneId = row[2].IsNull() ? TStrings::EMPTY : row[2].AsString();
                item.Uid = row[3].IsNull() ? TStrings::EMPTY : row[3].AsString();
                item.Bound = row[4].IsNull() ? TStrings::EMPTY : row[4].AsString();
                item.Flags = row[5].IsNull() ? TStrings::EMPTY : row[5].AsString();

                chunk->Bindings.push_back(std::move(item));
            }
        }

        std::unique_ptr<TPhoneBindingsResult> result = std::make_unique<TPhoneBindingsResult>();
        result->PhoneBindingsChunk = std::move(chunk);
        return result;
    }

    TString TPhoneBindingsProcessor::BuildWhereClause(const TString& numbersStr,
                                                      const TString& phoneidsStr,
                                                      const TString& boundAfter) {
        if (numbersStr.empty() && phoneidsStr.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Both numbers and phone_ids parameters are empty, please specify one";
        }

        return NUtils::CreateStr(
            " WHERE ",
            numbersStr ? BuildWhereClauseWithNumbers(numbersStr)
                       : BuildWhereClauseWithPhoneids(phoneidsStr),
            boundAfter ? NUtils::CreateStr(" AND bound>=from_unixtime(", boundAfter, ')')
                       : "");
    }

    TString TPhoneBindingsProcessor::BuildWhereClauseWithNumbers(const TString& numbersStr) {
        std::vector<TStringBuf> numbers = NUtils::NormalizeListValue<TStringBuf>(numbersStr, ",");

        if (numbers.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Invalid numbers value";
        }

        TString whereClause;
        whereClause.reserve(numbersStr.size() + 20);
        whereClause.append("number in (");

        for (const TStringBuf n : numbers) {
            whereClause.append(TUtils::SanitizePhoneNumber(n));
            whereClause.push_back(',');
        }

        whereClause.back() = ')';

        return whereClause;
    }

    TString TPhoneBindingsProcessor::BuildWhereClauseWithPhoneids(const TString& phoneidsStr) {
        std::vector<TStringBuf> phoneids = NUtils::NormalizeListValue<TStringBuf>(phoneidsStr, ",");

        if (phoneids.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Invalid phone_ids value";
        }

        // we are here if we have type=='current', so it's safe to add phone_id check to where clause
        TString whereClause;
        whereClause.reserve(phoneidsStr.size() + 20);
        whereClause.append("phone_id in (");

        for (const TStringBuf id : phoneids) {
            if (!NUtils::DigitsOnly(id)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Invalid phone_id format: " << InvalidValue(id);
            }

            whereClause.append(id.begin(), id.end());
            whereClause.push_back(',');
        }
        whereClause.back() = ')';

        return whereClause;
    }

    TString TPhoneBindingsProcessor::BuildQuery(const TString& whereClause,
                                                bool needCurrent,
                                                bool needUnbound,
                                                bool needHistory) const {
        TString query;

        if (needCurrent) {
            NUtils::Append(query, PHONE_BINDINGS_CURRENT_QUERY, whereClause, CURRENT_TIMECHECK);

            const TString& flags = TUtils::GetUIntArg(Request_, TStrings::IGNOREBINDLIMIT);
            if (!flags.empty()) { // add flags, checked above that it's a number
                NUtils::Append(query, " AND flags=", flags);
            }
        }

        if (needUnbound) {
            NUtils::AppendSeparated(query, UNION, PHONE_BINDINGS_UNBOUND_QUERY);
            NUtils::Append(query, whereClause, UNBOUND_TIMECHECK);

            const TString& flags = TUtils::GetUIntArg(Request_, TStrings::IGNOREBINDLIMIT);
            if (!flags.empty()) { // add flags, checked above that it's a number
                NUtils::Append(query, " AND flags=", flags);
            }
        }

        if (needHistory) {
            NUtils::AppendSeparated(query, UNION, PHONE_BINDINGS_HISTORY_QUERY).append(whereClause);
        }

        return query;
    }
}
