#include "phone_operations.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/list_result.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>

namespace NPassport::NBb {
    TPhoneOperationsProcessor::TPhoneOperationsProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TPhoneOperationsProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::PhoneOperations);

        return checker;
    }

    std::unique_ptr<TListResult> TPhoneOperationsProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& finishedBefore = TUtils::GetUIntArg(Request_, TStrings::FINISHED_BEFORE, true);

        TString query;
        TQueries::AppendPhoneOperationsMethod(query, finishedBefore);

        std::vector<NDbPool::TNonBlockingHandle> handles;
        for (NDbPool::TDbPool& db : Blackbox_.ShardsMap().GetShards()) {
            NDbPool::TNonBlockingHandle sqlh(db);
            sqlh.SendQuery(query);
            handles.push_back(std::move(sqlh));
        }

        TListResult::TStringVector operations;
        for (auto& sqlh : handles) {
            std::unique_ptr<NDbPool::TResult> result = TUtils::WaitResult(sqlh, "dbpool exception in PhoneOperations");

            operations.reserve(operations.size() + result->size());
            for (const NDbPool::TRow& row : result->Table()) {
                TString value = row[0].IsNull() ? TStrings::EMPTY : row[0].AsString();

                if (value.empty()) {
                    continue;
                }

                operations.push_back(std::move(value));
            }
        }

        std::unique_ptr<TListResult> res = std::make_unique<TListResult>(
            TOutTokens::PHONE_OPERATIONS, TOutTokens::OPERATION);
        res->TotalCount = operations.size();
        res->Data = std::move(operations);

        return res;
    }
}
