#include "prove_key_diag.h"

#include "edit_totp.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/prove_key_diag_result.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_profile.h>

namespace NPassport::NBb {
    TProveKeyDiagProcessor::TProveKeyDiagProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TProveKeyDiagProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::ProveKeyDiag);

        return checker;
    }

    std::unique_ptr<TProveKeyDiagResult> TProveKeyDiagProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        if (!Blackbox_.TotpEncryptor()) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "TOTP encryption is not initialized, unable to decrypt secret";
        }

        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);
        const TString& timestamp = TUtils::GetUIntArg(Request_, TStrings::TIMESTAMP, true);
        const TString& skew = TUtils::GetUIntArg(Request_, TStrings::SKEW, true);
        TString secretSalt = NUtils::Base64url2bin(TUtils::GetCheckedArg(Request_, TStrings::SECRET_SALT));
        TString secretId = NUtils::Base64url2bin(TUtils::GetCheckedArg(Request_, TStrings::SECRET_ID));
        TString pinSecretSalt = NUtils::Base64url2bin(TUtils::GetCheckedArg(Request_, TStrings::PIN_SECRET_SALT));
        TString pinSecretId = NUtils::Base64url2bin(TUtils::GetCheckedArg(Request_, TStrings::PIN_SECRET_ID));
        TString totpSalt = NUtils::Base64url2bin(TUtils::GetCheckedArg(Request_, TStrings::TOTP_SALT));
        TString totpId = NUtils::Base64url2bin(TUtils::GetCheckedArg(Request_, TStrings::TOTP_ID));
        bool useJunk = TUtils::GetBoolArg(Request_, TStrings::USE_JUNK_SECRET);

        TTotpProfile totpProfile = TEditTotpProcessor::ReadTotpProfileFromDb(Blackbox_, uid, useJunk);

        std::unique_ptr<TProveKeyDiagResult> result = std::make_unique<TProveKeyDiagResult>();
        result->IsSecretIdCorrect = totpProfile.DiagSecret(secretId, secretSalt);
        result->IsPinSecretCorrect = totpProfile.DiagPinSecret(pinSecretId, pinSecretSalt);

        time_t timestampInt = IntFromString<time_t, 10>(timestamp);
        result->IsTotpCorrect = totpProfile.DiagTotp(Blackbox_.TotpService(), totpId, timestampInt, totpSalt);

        if (result->IsPinSecretCorrect.first && result->IsSecretIdCorrect.first && !result->IsTotpCorrect.first) {
            // secret is correct, user entered correct pin, but TOTP is unacceptable.
            // assume user's time is incorrect. now we'll try to determine it
            result->CorrectTimestamp =
                totpProfile.DiagTime(Blackbox_.TotpService(), totpId, skew, timestampInt, totpSalt);
        }

        return result;
    }
}
