#include "pwdhistory.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/password_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/pwdhistory_result.h>

#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NBb {
    TPwdHistoryProcessor::TPwdHistoryProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TPwdHistoryProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::PwdHistory);

        return checker;
    }

    std::unique_ptr<TPwdHistoryResult> TPwdHistoryProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        // get values and check that uid and depth are uint numbers (throw INVALID_PARAMS if not)
        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID, true);
        const TString& depth = TUtils::GetUIntArg(Request_, TStrings::DEPTH, true);
        const TString& password = TUtils::GetCheckedArg(Request_, TStrings::PASSWORD);
        const TString& reason = Request_.GetArg(TStrings::REASON);

        const TString query = BuildQuery(uid, reason, depth);

        std::unique_ptr<NDbPool::TResult> res;
        try {
            NDbPool::TBlockingHandle sqlh(Blackbox_.ShardsMap().GetPool(uid));
            res = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception in pwdHistory: %s", e.what());
            throw TDbpoolError("dbpool exception in pwdHistory", e.what());
        }

        std::unique_ptr<TPwdHistoryResult> result = std::make_unique<TPwdHistoryResult>();
        for (const NDbPool::TRow& row : res->Table()) {
            // find matching password
            if (Blackbox_.PasswordChecker().PasswordMatches(password, row[0].AsString(), uid)) {
                result->Reason = row[1].AsString();
                break;
            }
        }

        return result;
    }

    TString TPwdHistoryProcessor::BuildQuery(const TString& uid, const TString& reason, const TString& depth) {
        TString query = NUtils::CreateStrExt(
            100,
            "SELECT encrypted_password,reason FROM password_history WHERE uid=",
            uid);

        std::vector<TStringBuf> reasons = NUtils::NormalizeListValue<TStringBuf>(reason, ",");
        if (!reasons.empty()) {
            NUtils::Append(query, " AND reason in (");

            for (TStringBuf r : reasons) {
                if (!NUtils::DigitsOnly(r)) {
                    throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                        << "Invalid reason argument: " << InvalidValue(r);
                }
                NUtils::Append(query, r, ",");
            }

            query.back() = ')';
        }

        NUtils::Append(query, " ORDER BY ts DESC LIMIT ", depth);

        return query;
    }
}
