#include "sessionid.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/base_result_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/display_name_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/partitions_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/strong_pwd_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/uid_helper.h>
#include <passport/infra/daemons/blackbox/src/loggers/authlog.h>
#include <passport/infra/daemons/blackbox/src/loggers/tskvlog.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/misc/ip_comparator.h>
#include <passport/infra/daemons/blackbox/src/misc/lrcs_cookie_parser.h>
#include <passport/infra/daemons/blackbox/src/misc/sess_kill_wrapper.h>
#include <passport/infra/daemons/blackbox/src/misc/session_utils.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/account_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/auth_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/authid_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/dbfields_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/emails_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/new_session_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/resigned_cookies.h>
#include <passport/infra/daemons/blackbox/src/output/sessguard_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/session_result.h>
#include <passport/infra/daemons/blackbox/src/output/typed_value_result.h>
#include <passport/infra/daemons/blackbox/src/output/uid_chunk.h>

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/sessguard.h>
#include <passport/infra/libs/cpp/auth_core/sessguard_parser.h>
#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/common/private_key.h>
#include <passport/infra/libs/cpp/tvm/signer/signer.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/yp_cookie_parser/yp_cookie_parser.h>

#include <util/datetime/base.h>

namespace NPassport::NBb {
    static const TString AID_COMMENT("aid=");
    static const TString TTL_COMMENT("ttl=");
    static const TString WEB_AUTH_TYPE("web");
    static const TString HOST_EQUAL("host=");
    static const TString SAFE_USER("safe=1");

    TSessionProcessorBase::TSessionProcessorBase(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    NAuth::TSession TSessionProcessorBase::CheckCookieAndRestrictions(ESessionStatus& status,
                                                                      std::unique_ptr<TAuthIdChunk>& authidChunk,
                                                                      TString& comment,
                                                                      std::vector<TString>& uidsList) {
        // Invoke libauth to parse cookie value.
        NAuth::TSession sess = Blackbox_.SessionParser().ParseCookie(SessionId_, Host_);
        TExperiment::Get().RunCookieCheck();
        if (NAuth::NSessionCodes::OK != sess.LastErr()) {
            TLog::Debug()
                << "Error: Session error <" << sess.LastErrAsString()
                << ">. Session info: " << LogableSessionId_ << "...";
        }

        // Convert Sesssion status to our own status
        status = TSessionUtils::Session2UserStatus(sess.IsValid());

        if (!TSessionUtils::ValidCookieStatus(status)) {
            return sess;
        }

        // Init external session flags for yateam
        if (Blackbox_.IsYandexIpCheckEnabled()) {
            time_t extAuthTs = TUtils::ToTime(sess.ExtAuthTs());
            time_t extUpdateTs = TUtils::ToTime(sess.ExtUpdateTs());
            time_t now = std::time(nullptr);

            if (now - extAuthTs > 24 * 3600 || now - extUpdateTs > NAuth::TSession::RESTRICTED_COOKIE_EXPIRE_TIME) {
                ExtSessionExpired_ = true;
            }

            UserIpIsYandex_ = Blackbox_.CheckYandexIp(UserIp_, Request_) != ENetworkKind::External; // TMP: robots are internal now!
            if (!sess.ExtUserIp().empty()) {
                NUtils::TIpAddr extIp = NAuth::TSessionUtils::ParseEncodedIp(sess.ExtUserIp());
                ExtUserIpMatch_ = Blackbox_.IpComparator().Equal(extIp, UserIp_);
            }
        }

        if (!Blackbox_.IsStressSessionAllowed() && sess.IsStress()) {
            TLog::Warning() << "Blocked stress testing session cookie <" << LogableSessionId_
                            << "...>, user ip <" << UserIp_
                            << "> from peer ip <" << Request_.GetRemoteAddr()
                            << ">, referrer <" << Request_.GetHeader("referer")
                            << ">, user agent <" << Request_.GetHeader("user-agent") << ">";
            comment.assign("Session generated for stress testing must not appear in production environment");
            status = ESessionStatus::INVALID;
            return sess;
        }

        // Forbid restricted cookies from ip other than original one
        authidChunk = ParseAuthId(sess);

        if (!authidChunk->Id.empty() && // error if we do have authid but with invalid fields
            (authidChunk->Timestamp.empty() || authidChunk->Host.empty()))
        {
            TLog::Debug() << "Cookie <" << LogableSessionId_
                          << "...> has invalid authid (" << authidChunk->Id << ")";
            comment.assign("Sessionid has broken authid format");
            status = ESessionStatus::INVALID;
            return sess;
        }

        if (sess.IsRestricted() && !Blackbox_.IpComparator().Equal(authidChunk->Ip, UserIp_)) {
            TLog::Debug() << "Cookie <" << LogableSessionId_
                          << "...> is restricted to ip " << authidChunk->Ip
                          << " which does not match user ip " << UserIp_;
            comment.assign("restricted cookie auth ip does not match with userip");
            status = ESessionStatus::INVALID;
            return sess;
        }

        if (!sess.Safe()) {
            comment.assign("Session with secure=0 not supported");
            status = ESessionStatus::INVALID;
            return sess;
        }

        uidsList.clear();
        for (unsigned i = 0; i < sess.UserCount(); ++i) {
            const TString& uid = sess.Uid(i);
            uidsList.push_back(uid);
        }

        // if we have sess_kill pool, send request
        if (Blackbox_.SesskillWrapper()) {
            SesskillSqlh_ = Blackbox_.SesskillWrapper()->SendRequest(uidsList, authidChunk->Id);
        }

        return sess;
    }

    bool TSessionProcessorBase::IsSessionKilled(const NAuth::TSession& sess, const TString& authid) const {
        if (!Blackbox_.SesskillWrapper() || !SesskillSqlh_) {
            return false;
        }

        TSessKillWrapper::TUidLoginDeltaMap deltas;
        for (unsigned i = 0; i < sess.UserCount(); ++i) {
            const TString& uid = sess.Uid(i);
            long loginDelta = sess.LoginDelta(i);

            deltas.insert(std::make_pair(uid, loginDelta));
        }

        // check there if the session is logouted
        if (Blackbox_.SesskillWrapper()->IsKilled(std::move(SesskillSqlh_), deltas, authid, sess.AuthIdTimestamp())) {
            TLog::Debug() << "Cookie <" << LogableSessionId_
                          << "...> is probably stolen (user logged out)";
            return true;
        }

        return false;
    }

    std::unique_ptr<TAuthIdChunk> TSessionProcessorBase::ParseAuthId(TString& authid) const {
        NAuth::TSession::TAuthid id;
        id.Str = std::move(authid);
        if (!NAuth::TSessionParser::ParseAuthId(id)) {
            authid = std::move(id.Str);
            if (!authid.StartsWith("noauth:")) {
                TLog::Debug() << "Bad format of authid: <" << authid
                              << ">, cookie <" << LogableSessionId_ << "...>";
            }
            return std::make_unique<TAuthIdChunk>();
        }
        authid = std::move(id.Str);
        NUtils::TIpAddr ip = NAuth::TSessionUtils::ParseEncodedIp(id.Ip);
        if (ip == NUtils::TIpAddr()) {
            return std::make_unique<TAuthIdChunk>();
        }

        // repair possibly broken authid
        authid.reserve(48); // timestamp_ms + ip in base64 + hostid + some extra space
        authid.assign(id.Time).push_back(':');
        authid.append(ip.ToBase64String()).push_back(':');
        authid.append(id.Host);

        return std::make_unique<TAuthIdChunk>(authid, id.Time, ip, id.Host);
    }

    std::unique_ptr<TAuthIdChunk> TSessionProcessorBase::ParseAuthId(NAuth::TSession& session) const {
        TString authid = session.AuthId();
        std::unique_ptr<TAuthIdChunk> result = ParseAuthId(authid);
        if (!result->Id.empty()) {
            session.SetAuthId(result->Id);
        }
        return result;
    }

    // get guard hosts as a vector, without empty, duplicates etc
    std::vector<TStringBuf> TSessionProcessorBase::GetGuardHostsArg() const {
        return NUtils::NormalizeListValue<TStringBuf>(Request_.GetArg(TStrings::GUARD_HOSTS), ",");
    }

    bool TSessionProcessorBase::HasAnotherGuardHost(const std::vector<TStringBuf>& guardHosts,
                                                    const TString& hostGuardspace,
                                                    const TString& domain) const {
        for (const TStringBuf& host : guardHosts) {
            const TString& guardspace = Blackbox_.SessGuardParser().GetGuardSpace(host, domain);
            if (!guardspace.empty() && guardspace != hostGuardspace) {
                return true;
            }
        }
        return false;
    }

    std::unique_ptr<TSessguardChunk> TSessionProcessorBase::GetSessguardCookies(const std::vector<TStringBuf>& guardHosts,
                                                                                const TString& authid,
                                                                                const TString& domain,
                                                                                time_t createTime,
                                                                                bool deleteGuards) const {
        std::unique_ptr<TSessguardChunk> res = std::make_unique<TSessguardChunk>();

        NAuth::TSessGuard guard(authid);

        for (const TStringBuf& host : guardHosts) {
            if (!host.EndsWith(domain)) {
                TLog::Debug() << "Will not make SessGuard for guard_host " << host
                              << ": it does not correspond to domain " << domain;
                continue; // don't create sessguards for wrong domains
            }
            TString guardDomain;
            TString hostStr(host);
            TString guardStr = Blackbox_.SessGuardParser().MakeGuardStr(guard, hostStr, guardDomain, createTime);
            if (guardStr.empty() || guardDomain.empty()) {
                continue; // ignore unsupported domains and services
            }

            if (deleteGuards) {
                res->Cookies.emplace(hostStr, TCookieChunk(TStrings::EMPTY, guardDomain, 1)); // empty body, ttl=1
            } else {
                res->Cookies.emplace(hostStr, TCookieChunk(guardStr, guardDomain, std::numeric_limits<i32>::max()));
            }
        }

        return res;
    }

    TSessionidProcessor::ESessionStatus TSessionProcessorBase::CheckUserStatus(NAuth::TSession& sess,
                                                                               int idx,
                                                                               ESessionStatus status,
                                                                               bool available,
                                                                               time_t glogout,
                                                                               time_t revokeWebSessions,
                                                                               time_t domainGlogout,
                                                                               bool changeReason,
                                                                               bool createRequired,
                                                                               bool passwdExpired,
                                                                               TString& comment,
                                                                               bool& haveExternal) const {
        if (idx < 0) { // user was not in cookie: can happen only when default uid is not found
            // TODO: PASSP-15058: get rid of this branch
            TLog::Debug() << "SessionProcessorBase: experiment: this case should never happen: idx must be >= 0";
            comment.assign("default user is not in this cookie");
            return ESessionStatus::INVALID;
        }

        // first of all, update cookie glogout flags:
        // cookie may be edited/resigned later and timestamps will update
        // but we need to keep glogout state even if account is not valid right now
        // or else flags in db may change and resigned cookie will become valid
        if (sess.Ts() < glogout || sess.Ts() < revokeWebSessions || sess.Ts() < domainGlogout) {
            sess.SetGlogouted(true, idx);
        }
        time_t extUpdateTs = TUtils::ToTime(sess.ExtUpdateTs());
        if (extUpdateTs > 0 &&
            sess.IsExternalAuth(idx) &&
            (extUpdateTs < glogout || extUpdateTs < revokeWebSessions || extUpdateTs < domainGlogout)) {
            sess.SetExtGlogouted(true, idx);
        }
        // we'll check glogouted flag below depending on internal/external session logic

        if (!available) {
            comment.assign("account disabled");
            return ESessionStatus::DISABLED;
        }

        if (changeReason) {
            comment.assign("password change required");
            return ESessionStatus::INVALID;
        }

        if (createRequired) {
            comment.assign("password create required");
            return ESessionStatus::INVALID;
        }

        if (passwdExpired) {
            comment.assign("user has strongpwd policy and password has expired");
            return ESessionStatus::INVALID;
        }

        if (!Blackbox_.IsLiteSessionAllowed() && sess.IsLite(idx)) {
            comment.assign("lites are not supported");
            return ESessionStatus::INVALID;
        }

        // session status may be changed if some external needs reset and overrides cookie status
        // and all consecutive users should still have original status by default
        ESessionStatus userStatus = status;

        if (Blackbox_.IsYandexIpCheckEnabled() && !sess.IsRestricted()) {
            // Yandex-team internal/external authorization check
            bool intAuth = sess.IsInternalAuth(idx);
            bool extAuth = sess.IsExternalAuth(idx);
            bool legacyUser = !intAuth && !extAuth; // log these later?

            if (UserIpIsYandex_ && (intAuth || legacyUser)) {
                if (sess.IsGlogouted(idx)) {
                    comment.assign("user has logged out globally");
                    return ESessionStatus::INVALID;
                }

                if (legacyUser) { // upgrade sessions with no flags
                    sess.SetInternalAuth(true, idx);
                }

                return userStatus;
            }
            // either IP is external or yandex IP + ext_auth flag
            if (sess.IsExtGlogouted(idx)) {
                comment.assign("user has logged out globally");
                return ESessionStatus::INVALID;
            }

            if (!extAuth) {
                comment = NUtils::CreateStr("internal session not allowed from external IP '", UserIp_, "'");
                return ESessionStatus::INVALID;
            }

            // check ext_user_ip match, ext_auth_ts and ext_update_ts
            if (!ExtUserIpMatch_) {
                comment.assign("external auth ip does not match with userip");
                return ESessionStatus::INVALID;
            }

            // check if expired
            time_t now = std::time(nullptr);

            if (ExtSessionExpired_) {
                comment.assign("external session expired");
                return ESessionStatus::INVALID;
            }

            // check if valid but needs reset
            if (TSessionUtils::ValidCookieStatus(userStatus)) {
                haveExternal = true;
                if (now - extUpdateTs > NAuth::TSession::RESTRICTED_COOKIE_VALID_TIME) {
                    sess.OverrideStatus(NAuth::TSession::EStatus::NEED_RESET);
                    return ESessionStatus::NEED_RESET;
                }
            } else {
                TLog::Error("Impossible condition in checkUserStatus()");
            }

        } else { // legacy glogout logic, with no internal/external sessions
            if (sess.IsGlogouted(idx)) {
                comment.assign("user has logged out globally");
                return ESessionStatus::INVALID;
            }
        }

        return userStatus;
    }

    NAuth::TSessGuard TSessionProcessorBase::CheckSessGuard(const TString& guardStr,
                                                            const TString& authId,
                                                            const TString& consumerName) const {
        NAuth::TSessGuard guard = Blackbox_.SessGuardParser().ParseGuard(guardStr, Host_);
        TExperiment::Get().RunCookieCheck();
        if (guard.Status() == NAuth::TSessGuard::INVALID) {
            const TString& requestId = Request_.GetArg(TStrings::REQUEST_ID);
            TLog::Debug() << "SessGuard invalid, guard="
                          << TStringBuf(guardStr).Chop(5) << "...,"
                          << " host=" << Host_ << ", error='" << guard.ErrMsg() << "',"
                          << " request_id=" << (requestId.empty() ? "<unknown>" : requestId)
                          << " peer='" << consumerName << "'";
            return guard;
        }

        if (guard.AuthId() != authId) {
            const TString& requestId = Request_.GetArg(TStrings::REQUEST_ID);
            TLog::Debug() << "SessGuard authId mismatch, guard="
                          << TStringBuf(guardStr).Chop(5) << "...,"
                          << " host=" << Host_ << ", guard.authId='" << guard.AuthId() << "',"
                          << " Session.authId=" << authId
                          << " request_id=" << (requestId.empty() ? "<unknown>" : requestId)
                          << " peer='" << consumerName << "'";
            return {""}; // empty guard with status==INVALID
        }

        return guard;
    }

    void TSessionProcessorBase::GetSessionidFromArgs() {
        SessionId_ = TUtils::GetCheckedArgSession(Request_, TStrings::SESSION_ID);

        LogableSessionId_ = TStringBuf(SessionId_).RBefore('.');
    }

    const TString TSessionidProcessor::SCOPE_BB_SESSIONID = "bb:sessionid";
    const TString TSessionidProcessor::SCOPE_BB_SESSGUARD = "bb:sessguard";

    TSessionidProcessor::TSessionidProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : TSessionProcessorBase(impl, request)
    {
    }

    TGrantsChecker TSessionidProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::Cookie);

        checker.CheckHasArgAllowed(TStrings::FULL_INFO, TBlackboxFlags::FullInfo);
        checker.CheckArgIsTrue(TStrings::RESIGN, TBlackboxFlags::ResignCookie);
        checker.CheckNotEmptyArgAllowed(TStrings::RESIGN_FOR_DOMAINS, TBlackboxFlags::ResignForDomains);
        checker.CheckNotEmptyArgAllowed(TStrings::GUARD_HOSTS, TBlackboxFlags::CreateGuard);
        checker.CheckHasArgAllowed(TStrings::ALLOW_SCHOLAR, TBlackboxFlags::ScholarSession);
        checker.CheckHasArgAllowed(TStrings::ALLOW_FEDERAL, TBlackboxFlags::FederalAccounts);

        TPartitionsHelper::CheckGrants(Blackbox_.PartitionsSettings(), checker);
        TBaseResultHelper::CheckGrants(
            Blackbox_.DbFieldSettings(),
            Blackbox_.AttributeSettings(),
            TConsumer::ERank::HasCred,
            checker);

        return checker;
    }

    std::unique_ptr<TSessionResult> TSessionidProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);
        TUtils::CheckUserTicketAllowed(Request_);
        TUtils::CheckUserIpArg(Request_);

        // session_id and host arguments must be present
        GetSessionidFromArgs();
        Host_ = TUtils::GetCheckedArg(Request_, TStrings::HOST);
        UserIp_ = TUtils::GetUserIpArg(Request_);
        const TString& userPort = TUtils::GetUserPortArg(Request_);

        TString comment("OK");

        std::unique_ptr<TAuthIdChunk> authidChunk;
        ESessionStatus status;
        std::vector<TString> uidsList;

        NAuth::TSession sess = CheckCookieAndRestrictions(status, authidChunk, comment, uidsList);

        if (!TSessionUtils::ValidCookieStatus(status)) {
            return GetErrorSessionId(status, sess, comment);
        }

        TString authidStr = authidChunk->Id;

        if (sess.LastErr() != NAuth::NSessionCodes::OK) {
            comment.assign(sess.LastErrAsString());
        }

        bool hasValidSessguard = false;
        bool newSessguardAllowed = false;
        TString domain;
        if (sess.Domsuff().find('_') != TString::npos) {
            domain.push_back('.');
        }
        domain.append(sess.Domsuff());
        NAuth::TSessionUtils::KeyspaceToDomain(domain);

        const TString& hostGuardspace = Blackbox_.SessGuardParser().GetGuardSpace(Host_, domain);

        if (hostGuardspace) { // this host supports sessguard
            const TSessGuardOptions& options = Blackbox_.GetGuardSpaceOptions(hostGuardspace);
            const TString& sessguard = Request_.GetArg(TStrings::SESSGUARD);

            if (sessguard) {
                NAuth::TSessGuard guard = CheckSessGuard(sessguard, sess.AuthId(), consumer.GetName());
                if (guard.Status() != NAuth::TSessGuard::VALID) {
                    if (options.IsEnabled(UserIp_)) {
                        return GetErrorSessguard(sess, hostGuardspace);
                    }
                } else {
                    hasValidSessguard = true;
                    if (std::time(nullptr) - guard.CreateTime() > 30 * 24 * 3600) {
                        // if sessguard is older than 30 days it is better to refresh it (with cookie)
                        if (status == ESessionStatus::VALID) {
                            sess.OverrideStatus(NAuth::TSession::EStatus::NEED_RESET);
                            status = ESessionStatus::NEED_RESET;
                        }
                    }
                    // sessguard is valid, creating new sessguard allowed
                    newSessguardAllowed = true;
                }
            } else { // no or empty sessguard
                const TString& requestId = Request_.GetArg(TStrings::REQUEST_ID);
                TLog::Debug() << "SessGuard empty, host=" << Host_ << ","
                              << " request_id=" << (requestId.empty() ? "<unknown>" : requestId)
                              << " peer='" << consumer.GetName() << "'";
                if (options.IsEnabled(UserIp_)) {
                    return GetErrorSessguard(sess, hostGuardspace);
                }

                if (status == ESessionStatus::VALID && options.IsNeedResetStatus(UserIp_)) {
                    sess.OverrideStatus(NAuth::TSession::EStatus::NEED_RESET);
                    status = ESessionStatus::NEED_RESET;
                }

                // we are in dry-run mode with empty sessguard, creating new sessguard allowed
                newSessguardAllowed = true;
            }
        }

        // need to fetch information from db
        bool multisession = TUtils::GetBoolArg(Request_, TStrings::MULTISESSION);

        int defaultIdx = sess.DefaultIdx();

        const TString& defaultUid = TUtils::GetUIntArg(Request_, TStrings::DEFAULT_UID);

        if (!defaultUid.empty()) {
            defaultIdx = sess.FindUser(defaultUid);
            if (defaultIdx < 0) {
                defaultIdx = uidsList.size();
                uidsList.push_back(defaultUid);
            }
        }

        if (!multisession && uidsList.size() > 1) { // in no multisession leave only default user
            TString defaultUid = std::move(uidsList[defaultIdx]);
            defaultIdx = 0;
            uidsList.resize(1);
            uidsList[defaultIdx] = std::move(defaultUid);
        }

        TPartitionsHelper::ParsePartitionArg(
            Blackbox_.PartitionsSettings(),
            Request_,
            TPartitionsHelper::TSettings{
                .Method = "sessionid",
                .ForbidNonDefault = true,
            });

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId());

        TBaseResultHelper baseResult(conv, Blackbox_, Request_);
        TStrongPwdHelper strongPwd(fetcher);

        fetcher.AddAttr(TAttr::ACCOUNT_BETATESTER);
        fetcher.AddAlias(TAlias::YANDEXOID_LOGIN);
        const TDbIndex availableIdx = fetcher.AddAttr(TAttr::ACCOUNT_IS_AVAILABLE);
        const TDbIndex glogoutIdx = fetcher.AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME);
        const TDbIndex revokeWebSessionsIdx = fetcher.AddAttr(TAttr::REVOKER_WEB_SESSIONS);
        const TDbIndex changeReasonIdx = fetcher.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON);
        const TDbIndex createRequiredIdx = fetcher.AddAttr(TAttr::PASSWORD_CREATING_REQUIRED);
        const TDbIndex havePwdIdx = fetcher.AddAttr(TAttr::ACCOUNT_HAVE_PASSWORD);
        const TDbIndex federalIdx = fetcher.AddAlias(TAlias::FEDERAL);
        const TDbIndex regCompletionRecommendedIdx = fetcher.AddAttr(TAttr::ACCOUNT_COMPLETION_RECOMMENDED);

        fetcher.FetchByUids(uidsList);

        // prepare common result part
        std::unique_ptr<TSessionResult> sessionResult = std::make_unique<TSessionResult>();

        sessionResult->IsMultisession = multisession;
        sessionResult->Comment = comment;
        sessionResult->Age = sess.Age() < 0 ? 0 : sess.Age();
        sessionResult->Expires = sess.GetExpireTime();
        sessionResult->Ttl = sess.Ttl();

        if (TUtils::GetBoolArg(Request_, TStrings::AUTHID)) {
            sessionResult->AuthId = std::move(authidChunk);
        }

        if (!authidStr.empty()) {
            sessionResult->ConnectionId = NUtils::CreateStr("s:", authidStr);
        }

        if (TUtils::GetBoolArg(Request_, TStrings::GET_LOGIN_ID)) {
            sessionResult->LoginId = TSessionUtils::GetLoginId(sess);
        }

        sessionResult->SpecialKind = sess.IsStress() ? TSessionResult::Stress : TSessionResult::None;

        bool showFullInfo = TUtils::GetBoolArg(Request_, TStrings::FULL_INFO);
        bool haveExternalAuth = false;

        NTicketSigner::TUserSigner userSigner(TUtils::GetBoolArg(Request_, TStrings::GET_USER_TICKET));
        TString validUids;
        validUids.reserve(10 * sess.UserCount());
        TString validDefaultUid;

        bool allowScholar = TUtils::GetBoolArg(Request_, TStrings::ALLOW_SCHOLAR);
        bool allowFederal = TUtils::GetBoolArg(Request_, TStrings::ALLOW_FEDERAL);

        const bool hasRegCompletionRecommendedParam = Request_.HasArg(TStrings::GET_REG_COMPLETION_RECOMMENDATION_WITH_YP);

        // then fill up users
        for (const TString& currentUid : uidsList) {
            int idx = sess.FindUser(currentUid);
            TSessionUserPtr sessionUserUp = std::make_unique<TSessionUser>();
            TSessionUser& sessionUser = *sessionUserUp;
            sessionResult->Users.push_back({currentUid, std::move(sessionUserUp)});

            if (idx < 0) {
                // lets log this case here, before other checks, to see all such cases for sure
                // if uid from request is not in cookie it means we have alien default uid
                TLog::Debug() << "Default_uid not in cookie. peer=" << consumer.GetName()
                              << ", default_uid=" << defaultUid
                              << ", cookie=<" << LogableSessionId_
                              << "...>, peer_ip=<" << Request_.GetRemoteAddr() << ">";
                sessionUser.Status = ESessionStatus::INVALID;
                sessionUser.Comment = "default user is not in this cookie";
                continue;
            }

            sessionUser.Comment = comment;

            const TDbProfile* profile = fetcher.ProfileByUid(currentUid);

            if (nullptr == profile) {
                TLog::Debug() << "BlackBox error: cookie <" << LogableSessionId_
                              << "...> of age=" << sess.Age()
                              << ", uid=" << currentUid
                              << " is OK but account is not found in the database: "
                              << (sess.Age() <= 10 ? "just created" : "deleted") << "?";
                sessionUser.Status = ESessionStatus::INVALID;
                sessionUser.Comment = "cookie is OK but account not found";
                continue;
            }

            const bool available = profile->Get(availableIdx)->AsBoolean();
            const time_t glogout = profile->Get(glogoutIdx)->AsTime();
            const time_t revokeWebSessions = profile->Get(revokeWebSessionsIdx)->AsTime();
            const TInstant domainGlogout = profile->PddDomItem().Glogout();
            const bool changeReason = profile->Get(changeReasonIdx)->AsBoolean();
            const bool createRequired = profile->Get(createRequiredIdx)->AsBoolean();
            const bool isFederal = !profile->Get(federalIdx)->Value.empty();
            const bool passwdExpired = strongPwd.PasswdExpired(profile, Blackbox_.StrongPwdExpireTime());

            if (hasRegCompletionRecommendedParam) {
                // check lrcs key in yp cookie
                bool readyToShowStatus = false;

                TString lrcs;
                TStringBuf ypCookie = Request_.GetArg(TStrings::GET_REG_COMPLETION_RECOMMENDATION_WITH_YP);
                if (NPassport::NYpCookie::TryGetYandexPermanentCookieKey(TStrings::LRCS_COOKIE_NAME, ypCookie, lrcs)) {
                    // show registration suggest if lrcs is in the past
                    readyToShowStatus = !TLrcsCookieParser::IsLrcsNotTooOld(lrcs, Blackbox_.LrcsTtl());
                }

                sessionUser.IsRegCompletionRecommended = readyToShowStatus && profile->Get(regCompletionRecommendedIdx)->AsBoolean();
            }

            sessionUser.Status = CheckUserStatus(sess,
                                                 idx,
                                                 status,
                                                 available,
                                                 glogout,
                                                 revokeWebSessions,
                                                 domainGlogout.TimeT(),
                                                 changeReason,
                                                 createRequired,
                                                 passwdExpired,
                                                 sessionUser.Comment,
                                                 haveExternalAuth);

            if (!allowScholar && sess.IsScholar(idx)) {
                sessionUser.Comment.assign("scholar sessions not allowed");
                sessionUser.Status = ESessionStatus::INVALID;
            }

            if (isFederal && !allowFederal) {
                if (TExperiment::Get().RestrictFederalUsers) {
                    sessionUser.Comment.assign("federal accounts not allowed");
                    sessionUser.Status = ESessionStatus::INVALID;
                } else {
                    TLog::Debug() << "Federal account not allowed in method=sessionid. "
                                  << "Uid=" << profile->Uid() << ", "
                                  << "consumer=" << consumer.GetName();
                }
            }

            // user may be disabled or glogouted, check status again
            if (!TSessionUtils::ValidCookieStatus(*sessionUser.Status)) {
                TLog::Debug() << "User in sessionid is invalid: status=" << TSessionUtils::StatusName(sessionUser.Status)
                              << ", comment=" << sessionUser.Comment
                              << ". uid=" << currentUid;

                // user_status == SessionStatus::INVALID || mda::DISABLED
                if (!multisession && sess.Uid() == currentUid) {
                    TString domain = sess.Domsuff();
                    NAuth::TSessionUtils::KeyspaceToDomain(domain);

                    if (Blackbox_.IsMdaDomain(domain)) {
                        return GetErrorSessionId(ESessionStatus::NOAUTH, sess, "OK");
                    }
                }

                if (!showFullInfo) {
                    continue;
                }
            } else { // valid uid, add to ticket
                NUtils::AppendSeparated(validUids, ',', currentUid);

                if (currentUid == uidsList.at(defaultIdx)) {
                    validDefaultUid = currentUid;
                }

                ui64 uid = TUtils::ToUInt(currentUid, TStrings::UID);
                userSigner.AddUid(uid);
            }

            // for user not in cookie we always have INVALID status so it is safe to say it is not lite
            sessionUser.Uid = TUidHelper::Result(profile, sess.IsLite(idx), profile->PddDomItem());

            baseResult.FillResults(sessionUser, profile);

            std::unique_ptr<TAuthChunk> authChunk = GetAuthChunk(sess, idx, profile->Get(havePwdIdx)->AsBoolean());

            sessionUser.Auth = std::move(authChunk);
        }

        // checking for killed session as late as possible, but before renew and ses_update event
        if (IsSessionKilled(sess, authidStr)) {
            return GetErrorSessionId(ESessionStatus::INVALID, sess, "session logged out");
        }

        if (!validUids.empty()) {
            Blackbox_.StatboxLogger()->LogSesCheck(validUids,
                                                   validDefaultUid,
                                                   Host_,
                                                   authidStr,
                                                   Request_.GetArg(TStrings::STATBOX_ID),
                                                   Request_.GetArg(TStrings::YANDEXUID),
                                                   UserIp_,
                                                   userPort,
                                                   Request_.GetConsumerFormattedName(),
                                                   Request_.GetRemoteAddr(),
                                                   consumer.GetClientId());
        }

        // update session status again - when checking users with external sessions status may change
        status = TSessionUtils::Session2UserStatus(sess.IsValid());
        sessionResult->Status = status;
        sessionResult->DefaultId = defaultIdx;
        sessionResult->AllowMoreUsers = sess.UserCount() < Blackbox_.MultisessionUserLimit();

        // it is safe to update ts every time, we need this for resign, and no harm in other case
        if (haveExternalAuth) { // update external auth too
            sess.SetExtUpdateTs(IntToString<10>(std::time(nullptr)));
        }

        // if we have NEED_RESET cookie we may want to resign it
        // to fix Safari bug with duplicate cookies we need to resign valid cookies too (if force_resign argument given)
        bool needResign = TUtils::GetBoolArg(Request_, TStrings::RESIGN);
        bool forceResign = TUtils::GetBoolArg(Request_, TStrings::FORCE_RESIGN);

        // check if we need new sessionid
        if (needResign && (sessionResult->Status == ESessionStatus::NEED_RESET || forceResign)) {
            sessionResult->NewSession = GetRenewSession(*sessionResult, sess, domain);
        }

        // check if we need new sessguard
        if (needResign && newSessguardAllowed) { // needResign is closed with grants, so this is actually a permission check
            std::vector<TStringBuf> guardHosts = GetGuardHostsArg();
            if (!guardHosts.empty() && (sessionResult->Status == ESessionStatus::NEED_RESET || forceResign || HasAnotherGuardHost(guardHosts, hostGuardspace, domain))) {
                sessionResult->NewSessguards = GetSessguardCookies(guardHosts, sess.AuthId(), domain);
            }
        }

        const TString& resignForDomains = Request_.GetArg(TStrings::RESIGN_FOR_DOMAINS);
        if (!resignForDomains.empty()) {
            sessionResult->ResignedCookies = GetResignedCookies(*sessionResult, sess, resignForDomains);
        }

        if (!validUids.empty() && userSigner.Enabled()) {
            userSigner.SetDefaultUid(
                validDefaultUid.empty() ? 0 : TUtils::ToUInt(validDefaultUid, TStrings::UID));
            userSigner.SetEntryPoint(consumer.GetClientId());
            userSigner.SetEnv(Blackbox_.UserTicketEnv());
            userSigner.AddScope(SCOPE_BB_SESSIONID);
            if (hasValidSessguard) {
                userSigner.AddScope(SCOPE_BB_SESSGUARD);
            }

            TUserTicketCache::TCacheHolder cache;
            if (Blackbox_.UserTicketCacheForSessionid()) {
                cache = Blackbox_.UserTicketCacheForSessionid()->GetCache();
            }

            NCache::EStatus status;
            TString userTicket = cache.GetValue(userSigner, status);
            if (userTicket.empty()) {
                userTicket = userSigner.SerializeV3(
                    *Blackbox_.TvmPrivateKeys().GetKey(),
                    time(nullptr) + Blackbox_.UserTicketTtl() + Blackbox_.UserTicketCacheTtl());
                cache.PutValue(std::move(userSigner), TString(userTicket));
            }

            sessionResult->UserTicket = std::move(userTicket);
        }

        return sessionResult;
    }

    std::unique_ptr<TSessionResult>
    TSessionidProcessor::GetErrorSessionId(ESessionStatus status,
                                           NAuth::TSession& sess,
                                           const TString& comment) const {
        std::unique_ptr<TSessionResult> sessionResult = std::make_unique<TSessionResult>();

        if (status == ESessionStatus::NOAUTH) {
            sessionResult->Age = sess.Age();
            sessionResult->Expires = sess.GetExpireTime();
        }

        sessionResult->Status = status;
        sessionResult->Comment = (sess.LastErr() == NAuth::NSessionCodes::OK) ? comment : sess.LastErrAsString();

        if (TUtils::GetBoolArg(Request_, TStrings::AUTHID)) {
            if (status != ESessionStatus::INVALID) {
                sessionResult->AuthId = ParseAuthId(sess);
            } else {
                sessionResult->AuthId = std::make_unique<TAuthIdChunk>();
            }
        }

        TLog::Debug() << "Sessionid is invalid: status=" << TSessionUtils::StatusName(status)
                      << ", comment=" << comment
                      << ". Body: " << LogableSessionId_ << "...";

        return sessionResult;
    }

    std::unique_ptr<TSessionResult> TSessionidProcessor::GetErrorSessguard(NAuth::TSession& sess,
                                                                           const TString& guardspace) const {
        return GetErrorSessionId(guardspace == "guard_passport" ? ESessionStatus::INVALID : ESessionStatus::WRONG_GUARD,
                                 sess, "Sessguard is not valid");
    }

    // parsing two sessionid cookies auth info
    std::unique_ptr<TAuthChunk>
    TSessionidProcessor::GetAuthChunk(const NAuth::TSession& sess, unsigned idx, bool havePwd) {
        long pwdAge = -1;

        if (sess.PasswordCheckDelta(idx) >= 0) {
            pwdAge = time(nullptr) - (sess.AuthIdTimestamp() + sess.PasswordCheckDelta(idx));
            if (pwdAge < 0) {
                pwdAge = 0;
            }
        }

        return std::make_unique<TAuthChunk>(pwdAge, sess.SocialId(idx), havePwd, sess.IsScholar(idx));
    }

    namespace {
        TString GetRenewComment(const NAuth::TSession& sess, const TString& domain, const TString& authid) {
            // format comment like "aid=...;ttl=0"
            return NUtils::CreateStr(
                AID_COMMENT,
                authid,
                ";",
                TTL_COMMENT,
                sess.Ttl(),
                ";",
                HOST_EQUAL,
                domain);
        }
    }

    std::unique_ptr<TNewSessionChunk>
    TSessionidProcessor::GetRenewSession(const TSessionResult& sessResult, NAuth::TSession& sess, const TString& domain) const {
        // If we do renew the cookie we need to make a "ses_update" record in the auth.log
        if (Blackbox_.AuthLogger()) {
            TString comment = GetRenewComment(sess, domain, sess.AuthId());

            const TString& useragent = Request_.GetArg(TStrings::USERAGENT);
            const TString& yandexuid = Request_.GetArg(TStrings::YANDEXUID);

            // we need to write ses_update event for each good uid in the session
            for (const TSessionResult::TUidUserPair& pair : sessResult.Users) {
                if (pair.second && TSessionUtils::ValidCookieStatus(*pair.second->Status)) {
                    const TString& uid = pair.second->Uid ? pair.second->Uid->Uid : TStrings::EMPTY;
                    const TString& login = pair.second->Account ? pair.second->Account->Login : TStrings::EMPTY;

                    Blackbox_.AuthLogger()->Write(
                        uid,                  // uid
                        login,                // login
                        TStrings::EMPTY,      // sid
                        WEB_AUTH_TYPE,        // auth type
                        TAuthLog::UPDATESESS, // "flag"
                        comment,              // comment
                        SAFE_USER,            // attribs
                        false,                // captcha entered
                        UserIp_,              // user IP
                        TStrings::EMPTY,      // proxy IP
                        yandexuid,            // yandexuid cookie
                        TStrings::EMPTY,      // referer
                        TStrings::EMPTY,      // retpath
                        useragent);           // user-agent
                }
            }
        }

        return std::make_unique<TNewSessionChunk>(sess, domain, true);
    }

    std::unique_ptr<TResignedCookiesChunk> TSessionidProcessor::GetResignedCookies(const TSessionResult& sessResult,
                                                                                   NAuth::TSession& sess,
                                                                                   TStringBuf resignForDomain) const {
        std::unique_ptr<TResignedCookiesChunk> res = std::make_unique<TResignedCookiesChunk>();

        for (const TString& domain : NUtils::NormalizeListValue(resignForDomain, ",")) {
            NAuth::TKeyRing* keyring = Blackbox_.SessionSigner().TryToFindRingByName(domain);
            if (!keyring) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "failed to get keys for domain: " << InvalidValue(domain);
            }

            TString domainForCookie = keyring->KSpace();
            NAuth::TSessionUtils::KeyspaceToDomain(domainForCookie);

            sess.SetDomain(domainForCookie); // for keyspace on resigning
            if (!domainForCookie.empty() && '.' != domainForCookie.front()) {
                domainForCookie.insert(domainForCookie.cbegin(), '.'); // for domain in browser
            }

            res->Cookies.emplace(domain, GetRenewSession(sessResult, sess, domainForCookie));
        }

        return res;
    }
}
