#include "user_ticket.h"

#include "userinfo.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/base_result_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/partitions_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/test_pin_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/user_ticket_result.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/common/private_key.h>
#include <passport/infra/libs/cpp/tvm/common/public_keys.h>

namespace NPassport::NBb {
    TUserTicketProcessor::TUserTicketProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TUserTicketProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::UserTicket);

        if (Request_.GetHeader(TStrings::X_YA_SERVICE_TICKET).empty()) {
            checker.Add("method=user_ticket allowed only with header '" + TStrings::X_YA_SERVICE_TICKET + "'");
        }

        TPartitionsHelper::CheckGrants(Blackbox_.PartitionsSettings(), checker);
        TBaseResultHelper::CheckGrants(
            Blackbox_.DbFieldSettings(),
            Blackbox_.AttributeSettings(),
            TConsumer::ERank::HasCred,
            checker);

        TTestPinHelper::CheckGrants(checker);

        return checker;
    }

    std::unique_ptr<TUserTicketResult> TUserTicketProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& ut = TUtils::GetCheckedArg(Request_, TStrings::USER_TICKET);
        NTvmAuth::TCheckedUserTicket ticket = Blackbox_.TvmClient().CheckUserTicket(ut);
        if (!ticket) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "UserTicket is invalid: " << NTvmAuth::StatusToString(ticket.GetStatus())
                << ": " << InvalidValue(NTvmAuth::NUtils::RemoveTicketSignature(ut));
        }

        const std::vector<TString> uids = MakeUidList(ticket, ut);
        const std::optional<TString> kid = GetUidOfKid();

        TUserInfoProcessor userinfo(Blackbox_, Request_);

        TUserInfoProcessor::TKidStatus kidStatus;
        auto userinfoResult = userinfo.ProcessUserInfoByUidsImpl(uids, kid, &kidStatus, "user_ticket");

        return TranslateResult(
            std::move(userinfoResult),
            uids,
            kid,
            kidStatus,
            consumer);
    }

    std::vector<TString> TUserTicketProcessor::MakeUidList(const NTvmAuth::TCheckedUserTicket& t,
                                                           const TString& ut) const {
        // Sanitizing
        const TString& uid = TUtils::GetUIntArg(Request_, TStrings::UID);
        if (Request_.HasArg(TStrings::MULTI) && !uid.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Params 'multi' and 'uid' conflict with each other";
        }

        // One special uid
        if (!uid.empty()) {
            ui64 u = TUtils::ToUInt(uid, TStrings::UID);
            if (std::find(t.GetUids().begin(), t.GetUids().end(), u) == t.GetUids().end()) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "'uid' is not found in UserTicket: "
                    << InvalidValue(uid)
                    << "; UserTicket: " << InvalidValue(NTvmAuth::NUtils::RemoveTicketSignature(ut));
            }
            return {uid};
        }

        // All uids
        if (TUtils::GetBoolArg(Request_, TStrings::MULTI)) {
            std::vector<TString> uids;
            uids.reserve(t.GetUids().size());
            for (NTvmAuth::TUid u : t.GetUids()) {
                uids.push_back(IntToString<10>(u));
            }
            return uids;
        }

        // Default uid
        return {IntToString<10>(t.GetDefaultUid())};
    }

    std::optional<TString> TUserTicketProcessor::GetUidOfKid() const {
        if (!Request_.HasArg(TStrings::GET_USER_TICKET_FOR_KID)) {
            return {};
        }

        if (Request_.HasArg(TStrings::MULTI)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Params '" << TStrings::MULTI << "' and '" << TStrings::GET_USER_TICKET_FOR_KID
                << "' conflict with each other";
        }

        return TUtils::GetUIntArg(Request_, TStrings::GET_USER_TICKET_FOR_KID, true);
    }

    static TString PrintUids(const TUserTicketResult& result) {
        TString res;
        res.reserve(10 * result.Results.size());

        res.push_back('[');
        for (const auto& [uid, r] : result.Results) {
            NUtils::AppendSeparated(res, ',', uid);
        }
        res.push_back(']');

        return res;
    }

    std::unique_ptr<TUserTicketResult>
    TUserTicketProcessor::TranslateResult(TUserInfoResult userinfo,
                                          const std::vector<TString>& uids,
                                          const std::optional<TString>& kid,
                                          const TUserInfoProcessor::TKidStatus& status,
                                          const TConsumer& consumer) const {
        std::unique_ptr res = std::make_unique<TUserTicketResult>();

        res->Results = std::move(userinfo->Results);

        if (!kid) {
            return res;
        }

        if (res->Results.size() != 1) {
            TLog::Error() << "UserTicketProcessor: processing 'for_kid' could be only for one user: "
                          << uids.at(0)
                          << "; got info for users: " << PrintUids(*res);
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Internal error: only one user from db was expected, got "
                << res->Results.size();
        }

        bool userWasFound = res->Results[0].second &&
                            res->Results[0].second->Uid &&
                            res->Results[0].second->Uid->Uid;
        if (!userWasFound) {
            return res;
        }

        TUserTicketResult::TForKid forKid;
        if (status.Error) {
            forKid.Error = status.Error;

            TLog::Debug()
                << "UserTicketProcessor: failed to get kid: user " << res->Results[0].first
                << ", kid " << *kid
                << ": " << status.Error;
        } else {
            forKid.UserTicket = GenerateUserTicketForKid(*kid, consumer);
        }

        res->ForKid = std::move(forKid);

        return res;
    }

    static const TString SCOPE_BB_KID = "bb:kid";

    TString TUserTicketProcessor::GenerateUserTicketForKid(const TString& uidStr,
                                                           const TConsumer& consumer) const {
        const ui64 uid = TUtils::ToUInt(uidStr, TStrings::GET_USER_TICKET_FOR_KID);

        NTicketSigner::TUserSigner userSigner;

        userSigner.AddUid(uid);
        userSigner.SetDefaultUid(uid);
        userSigner.SetEntryPoint(consumer.GetClientId());
        userSigner.SetEnv(Blackbox_.UserTicketEnv());
        userSigner.AddScope(SCOPE_BB_KID);

        TUserTicketCache::TCacheHolder cache;
        if (Blackbox_.UserTicketCacheForUserTicket()) {
            cache = Blackbox_.UserTicketCacheForUserTicket()->GetCache();
        }

        NCache::EStatus status;
        TString res = cache.GetValue(userSigner, status);
        if (res) {
            return res;
        }

        res = userSigner.SerializeV3(
            *Blackbox_.TvmPrivateKeys().GetKey(),
            time(nullptr) + Blackbox_.UserTicketTtl() + Blackbox_.UserTicketCacheTtl());
        cache.PutValue(std::move(userSigner), TString(res));

        return res;
    }
}
