#include "userinfo.h"

#include "family_info.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/base_result_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/test_pin_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/uid_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/login_type.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/account_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/attributes_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/dbfields_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/emails_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/karma_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/uid_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/user_info_result.h>

#include <passport/infra/libs/cpp/auth_core/public_id.h>
#include <passport/infra/libs/cpp/auth_core/public_id_encryptor.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TUserInfoProcessor::TUserInfoProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TUserInfoProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::UserInfo);

        TPartitionsHelper::CheckGrants(Blackbox_.PartitionsSettings(), checker);
        TBaseResultHelper::CheckGrants(
            Blackbox_.DbFieldSettings(),
            Blackbox_.AttributeSettings(),
            TConsumer::ERank::NoCred,
            checker);

        TTestPinHelper::CheckGrants(checker);

        if (Request_.GetArg(TStrings::UID).empty()) {
            checker.CheckHasArgAllowed(TStrings::FIND_BY_PHONE_ALIAS, TBlackboxFlags::FindByPhoneAlias);

            if (Request_.HasArg(TStrings::ALLOW_SCHOLAR) &&
                !consumer.IsAllowed(TBlackboxFlags::ScholarLogin) &&
                !consumer.IsAllowed(TBlackboxFlags::ScholarSession))
            {
                checker.Add("no grants for arg 'allow_scholar'");
            }
        }

        return checker;
    }

    std::unique_ptr<TBulkUserInfoResult>
    TUserInfoProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        TUtils::CheckUserIpArg(Request_);
        TUtils::GetUserPortArg(Request_); // not used now, but let it be valid for future

        TString uid = Request_.GetArg(TStrings::UID);
        if (uid.empty()) {
            std::unique_ptr<TBulkUserInfoResult> bulkResult = std::make_unique<TBulkUserInfoResult>();
            std::unique_ptr<TUserInfoResult> res = ProcessUserInfoImpl();
            if (res && res->Uid) {
                uid = res->Uid->Uid;
            }
            bulkResult->Results.push_back({uid, std::move(res)});
            return bulkResult;
        }

        return ProcessUserInfoByUids(uid);
    }

    std::unique_ptr<TBulkUserInfoResult>
    TUserInfoProcessor::ProcessUserInfoByUids(const TString& uidArg) {
        std::vector<TString> uids = NUtils::NormalizeListValue(uidArg, ",");

        if (uids.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid uid value: " << InvalidValue(uidArg);
        }

        if (Blackbox_.MultiuserLimit() && uids.size() > Blackbox_.MultiuserLimit()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid uid value: too many uids in request: " << uids.size()
                << ", allowed: " << Blackbox_.MultiuserLimit();
        }

        if (Blackbox_.MultiuserWarnLimit() && uids.size() > Blackbox_.MultiuserWarnLimit()) {
            TLog::Debug() << "Multiuser limit warning: too many uids in request:"
                          << uids.size() << ", allowed: " << Blackbox_.MultiuserWarnLimit()
                          << ". Consumer: " << Request_.GetConsumerFormattedName()
                          << ", remote addr " << Request_.GetRemoteAddr();
        }

        // sanitize all uids, throw on error
        for (const TString& uid : uids) {
            TUtils::ToUInt(uid, TStrings::UID);
            TUtils::ThrowOnLeadingZeroInNumber(uid, TStrings::UID);
        }

        return ProcessUserInfoByUidsImpl(uids);
    }

    std::unique_ptr<TBulkUserInfoResult>
    TUserInfoProcessor::ProcessUserInfoByUidsImpl(const std::vector<TString>& uids,
                                                  const std::optional<TString>& kid,
                                                  TKidStatus* kidStatus,
                                                  TStringBuf method) {
        TPartitionsHelper::ParsePartitionArg(
            Blackbox_.PartitionsSettings(),
            Request_,
            TPartitionsHelper::TSettings{
                .Method = method,
                .ForbidNonDefault = true,
            });

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId());

        TBaseResultHelper baseResult(conv, Blackbox_, Request_);
        TTestPinHelper pinHelper(fetcher, Request_);

        if (kid && kidStatus) {
            if (uids.size() != 1) {
                TLog::Error() << "UserInfoProcessor: single user was expected, got "
                              << uids.size() << " users";
                throw TBlackboxError(TBlackboxError::EType::Unknown)
                    << "Internal error: single user was expected, got "
                    << uids.size() << " users";
            }

            fetcher.AddFamilyInfo();
            fetcher.AddAlias(TAlias::PORTAL_LOGIN);
            fetcher.AddAlias(TAlias::KIDDISH);

            std::vector<TString> allUids = uids;
            allUids.push_back(*kid);
            fetcher.FetchByUids(allUids);
        } else {
            fetcher.FetchByUids(uids);
        }

        std::unique_ptr<TBulkUserInfoResult> bulkResult = std::make_unique<TBulkUserInfoResult>();
        for (const TString& uid : uids) {
            const TDbProfile* profile = fetcher.ProfileByUid(uid);

            std::unique_ptr<TUserInfoResult> result = std::make_unique<TUserInfoResult>();

            if (profile) {
                result->Uid = TUidHelper::Result(profile, false, profile->PddDomItem());
                baseResult.FillResults(*result, profile);
                result->PinStatus = pinHelper.Result(profile);
            } else {
                result->Uid = TUidHelper::Result(nullptr, false, PddDomain_);
                result->Karma = std::make_unique<TKarmaChunk>();
            }
            bulkResult->Results.push_back({uid, std::move(result)});
        }

        if (kid && kidStatus) {
            FillKidStatus(fetcher, uids[0], *kid, *kidStatus);
        }

        return bulkResult;
    }

    std::unique_ptr<TUserInfoResult>
    TUserInfoProcessor::ProcessUserInfoImpl() {
        if (!InitByAuthorizationType()) {
            return GetErrorResult();
        }

        TPartitionsHelper::ParsePartitionArg(Blackbox_.PartitionsSettings(), Request_, PartitionsHelperSettings_);

        TDbFetcher fetcher = Blackbox_.CreateDbFetcher();
        TDbFieldsConverter conv(fetcher, Blackbox_.Hosts(), Blackbox_.MailHostId(), Sids_, ForceSid2_);
        TUtils::CheckFindByPhoneAliasArg(Request_, fetcher);

        TBaseResultHelper baseResult(conv, Blackbox_, Request_);
        TTestPinHelper pinHelper(fetcher, Request_);

        const TDbProfile* profile = nullptr;
        std::unique_ptr<TUserInfoResult> result = std::make_unique<TUserInfoResult>();

        if (PublicId_.empty()) {
            if (!Suid_.empty()) {
                fetcher.FetchBySuid(Suid_, Sid_);
            } else {
                fetcher.FetchByLogin(Login_, PddDomain_.Id(), Sids_, PddDomain_.DefaultUid(), AllowScholar_);
            }
            profile = fetcher.NextProfile();
        } else { // public_id search
            NAuth::TPublicId id = Blackbox_.PublicIdEncryptor()->Decrypt(PublicId_);
            if (id.Status() == NAuth::TPublicId::VALID && id.Uid() != 0) {
                // it is generated public_id
                // don't forget to check version!
                int versionAttr = fetcher.AddAttr(TAttr::ACCOUNT_PUBLIC_ID_VERSION);

                fetcher.FetchByUid(IntToString<10>(id.Uid()));
                profile = fetcher.NextProfile();

                if (profile) {
                    const TString& dbVersion = profile->Get(versionAttr)->Value;
                    ui8 version;

                    if (dbVersion.empty() || !TryIntFromString<10>(dbVersion, version)) {
                        TLog::Error() << "Bad PublicId version for user " << profile->Uid()
                                      << " version=" << dbVersion << " taking 0 instead";
                        version = 0;
                    }
                    if (version != id.Version()) {
                        TLog::Debug() << "PublicId version mismatch for uid " << profile->Uid()
                                      << ", version in db: " << (int)version << ", version in public_id: " << (int)id.Version();
                        profile = nullptr;
                    }
                }
            } else {
                // probably it is custom user-defined public_id
                fetcher.FetchByAlias(TUtils::NormalizeLogin(PublicId_), {TAlias::PUBLICID, TAlias::OLDPUBLICID}, TStrings::EMPTY);
                profile = fetcher.NextProfile();
            }
        }

        if (profile) {
            result->Uid = TUidHelper::Result(profile, false, profile->PddDomItem());
            baseResult.FillResults(*result, profile);
            result->PinStatus = pinHelper.Result(profile);
        } else {
            // when searching by login we could have domain_id set up for alt domain instead of pdd
            // in this case domain_name will be empty
            // if so, we need to clean up domain id to show hosted="0" correctly
            if (PddDomain_.AsciiName().empty() && PddDomain_.UtfName().empty()) {
                PddDomain_.ClearId();
            }
            result->Uid = TUidHelper::Result(nullptr, false, PddDomain_);
            result->Karma = std::make_unique<TKarmaChunk>();
        }

        return result;
    }

    std::unique_ptr<TUserInfoResult> TUserInfoProcessor::GetErrorResult() {
        std::unique_ptr<TUserInfoResult> result = std::make_unique<TUserInfoResult>();
        result->Uid = std::make_unique<TUidChunk>();
        result->Karma = std::make_unique<TKarmaChunk>();
        return result;
    }

    bool TUserInfoProcessor::InitLoginSid() {
        Login_ = TUtils::GetCheckedArg(Request_, TStrings::LOGIN);

        if (Login_.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Empty login argument";
        }

        switch (TUtils::SanitizeLogin(Login_)) {
            case ESanitizeStatus::Ok:
                break;
            case ESanitizeStatus::TooLong:
                return false;
            case ESanitizeStatus::InvalidChars:
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Invalid characters in login: " << InvalidValue(Login_);
        }

        if (Login_.find('@') == Login_.size() - 1) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Empty domain part in login: " << InvalidValue(Login_);
        }
        Sid_.assign(Request_.GetArg(TStrings::SID)); // is sanitized later by DbFields::Fetch()
        // Do some (login,sid) magic: we may need to strip off @domain.tld part,
        // substitute actual sids value for a macro and decide whether this request
        // goes to Yandex users or mail-4-domains (PDD) users. In the latter case
        // we also learn a few things about the domain in question.
        ELoginType loginType =
            Blackbox_.TranslateLoginSids(Login_, Sid_, PddDomain_, Request_.GetArg(TStrings::COUNTRY));

        // Reject "alien' domains
        if (loginType == ELoginType::Alien) {
            return false;
        }

        if (!Sid_.empty()) {
            Sids_ = NUtils::ToVector(Sid_, ',');
        }
        ForceSid2_ = loginType == ELoginType::Yandex_Force_Sid2 ||
                     loginType == ELoginType::Hosted;
        // Next, for Yandex (as opposed to PDD) logins replace '.' with '-'; internally
        // this function checks to see that login is not an e-mail and for e-mails
        // does nothing.
        if (loginType != ELoginType::Hosted && Sid_ != TStrings::FEDERAL_SID) {
            TUtils::DotsToHyphens(Login_, false);
        }

        AllowScholar_ = TUtils::GetBoolArg(Request_, TStrings::ALLOW_SCHOLAR);

        PartitionsHelperSettings_ = TPartitionsHelper::TSettings{
            .Method = "userinfo with identification by login",
            .ForbidArray = true,
            .ForbidNonDefault = true,
        };

        return true;
    }

    bool TUserInfoProcessor::InitByAuthorizationType() {
        // How is the user identified? Accept three methods: 1) by suid [and sid]; 2) by login [and sid]; 3) by public_id
        // we decided to minimize confusion and don't allow PUBLIC_ID with LOGIN or SUID in one request
        // i.e. there should be no priorities here, you give only one id for search (except the UID which overrides everything else)
        // LOGIN + SUID are allowed for historical reasons but we hope to change this later
        if (Request_.HasArg(TStrings::PUBLIC_ID)) {
            if (Request_.HasArg(TStrings::LOGIN)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Can't search both by login AND public_id, please choose only one key";
            }
            if (Request_.HasArg(TStrings::SUID)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Can't search both by suid AND public_id, please choose only one key";
            }
        }

        if (Request_.HasArg(TStrings::SUID)) { // suid or suid-sid search
            // get suid/sid args, throw on insanity
            Suid_.assign(TUtils::GetUIntArg(Request_, TStrings::SUID));
            Sid_.assign(TUtils::GetUIntArg(Request_, TStrings::SID));
            Sids_.assign(1, Sid_);

            PartitionsHelperSettings_ = TPartitionsHelper::TSettings{
                .Method = "userinfo with identification by suid",
                .ForbidNonDefault = true,
            };

            return true;
        }

        if (Request_.HasArg(TStrings::LOGIN)) { // login or login-sid search
            return InitLoginSid();
        }

        if (Request_.HasArg(TStrings::PUBLIC_ID)) { // public_id search
            if (!Blackbox_.PublicIdEncryptor()) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "public_id is not supported in this environment";
            }
            PublicId_ = Request_.GetArg(TStrings::PUBLIC_ID);

            PartitionsHelperSettings_ = TPartitionsHelper::TSettings{
                .Method = "userinfo with identification by public id",
                .ForbidNonDefault = true,
            };

            return true;
        }

        throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Don't know what to search. Please, specify one of: uid, login, suid, public_id";
    }

    void TUserInfoProcessor::FillKidStatus(const TDbFetcher& fetcher,
                                           const TString& uid,
                                           const TString& kid,
                                           TUserInfoProcessor::TKidStatus& kidStatus) {
        const TDbProfile* profile = fetcher.ProfileByUid(uid);
        if (!profile) {
            kidStatus.Error = "User was not found";
            return;
        }

        if (profile->IsKid()) {
            kidStatus.Error = "Kid cannot get ticket for another kid";
            return;
        }

        if (!profile->FamilyInfo()) {
            kidStatus.Error = "User does not belong to any family";
            return;
        }

        const TDbProfile* kidProfile = fetcher.ProfileByUid(kid);
        if (!kidProfile) {
            kidStatus.Error = "Kid was not found";
            return;
        }

        if (!kidProfile->IsKid()) {
            kidStatus.Error = "'Kid' from request is not kid actually";
            return;
        }

        if (!kidProfile->FamilyInfo()) {
            kidStatus.Error = "'Kid' does not belong to any family";
            return;
        }

        if (kidProfile->FamilyInfo()->FamilyId != profile->FamilyInfo()->FamilyId) {
            kidStatus.Error = NUtils::CreateStr(
                "User belongs to family '",
                TFamilyInfoProcessor::FromInternalFamilyId(profile->FamilyInfo()->FamilyId),
                "', kid belongs to family '",
                TFamilyInfoProcessor::FromInternalFamilyId(kidProfile->FamilyInfo()->FamilyId),
                "'");
            return;
        }
    }
}
