#include "yakey_backup.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/out_tokens.h>
#include <passport/infra/daemons/blackbox/src/output/table_result.h>

namespace NPassport::NBb {
    TYakeyBackupProcessor::TYakeyBackupProcessor(const TBlackboxImpl& impl, const NCommon::TRequest& request)
        : Blackbox_(impl)
        , Request_(request)
    {
    }

    TGrantsChecker TYakeyBackupProcessor::CheckGrants(const TConsumer& consumer, bool throwOnError) {
        TGrantsChecker checker(Request_, consumer, throwOnError);

        checker.CheckMethodAllowed(TBlackboxMethods::YakeyBackup);

        return checker;
    }

    static const TString GET_YAKEY_BACKUP_QUERY = "SELECT phone_number,backup,UNIX_TIMESTAMP(updated),device_name FROM yakey_backups WHERE phone_number=";
    static const TString GET_YAKEY_BACKUP_META_QUERY = "SELECT phone_number,device_name,UNIX_TIMESTAMP(updated) FROM yakey_backups WHERE phone_number=";
    static const TTableResult::TStringVector YAKEY_BACKUP_COLUMNS = {
        "phone_number",
        "backup",
        "updated",
        "device_name",
    };
    static const TTableResult::TStringVector YAKEY_BACKUP_META_COLUMNS = {
        "phone_number",
        "device_name",
        "updated",
    };

    std::unique_ptr<TTableResult> TYakeyBackupProcessor::Process(const TConsumer& consumer) {
        CheckGrants(consumer);

        const TString& phoneNumber = TUtils::GetUIntArg(Request_, TStrings::PHONE_NUMBER, true);
        bool isMeta = TUtils::GetBoolArg(Request_, TStrings::META);

        std::unique_ptr<TTableResult> result = std::make_unique<TTableResult>(
            TOutTokens::YAKEY_BACKUPS,
            isMeta ? YAKEY_BACKUP_META_COLUMNS
                   : YAKEY_BACKUP_COLUMNS);
        unsigned nCols = result->Columns.size();

        TString query(isMeta ? GET_YAKEY_BACKUP_META_QUERY : GET_YAKEY_BACKUP_QUERY);
        query.append(phoneNumber);

        std::unique_ptr<NDbPool::TResult> res;
        try {
            NDbPool::TBlockingHandle sqlh(Blackbox_.CentralDb());
            res = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception in yakeyBackup: %s", e.what());
            throw TDbpoolError("dbpool exception in yakeyBackup", e.what());
        }

        time_t timeNow = time(nullptr);
        for (const NDbPool::TRow& row : res->Table()) {
            TTableResult::TStringVector strRow;
            if (row.size() < 3) {
                continue; // ignore incomplete records
            }

            strRow.reserve(nCols);
            for (unsigned i = 0; i < nCols && i < (unsigned)row.size(); ++i) {
                strRow.push_back(row[i].IsNull() ? "" : row[i].AsString());
            }

            // check backup expiration time
            const TString& lastUpdate = strRow[2];

            if (timeNow - IntFromString<time_t, 10>(lastUpdate) > Blackbox_.YakeyBackupExpireTime()) {
                continue;
            }

            result->Rows.push_back(std::move(strRow));
        }

        return result;
    }
}
