#include "anonymiser.h"

#include "db_types.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <algorithm>

namespace NPassport::NBb {
    static const TString RUS_LETTERS("абвгдеёжзийклмнопрстуфхцчшщъыьэюяАБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ");

    void TAnonymiser::AddAttribute(const TString& name, const TString& value) {
        AttrMap_.insert(TFieldsMap::value_type(name, value));
    }

    void TAnonymiser::SetEmailDomain(const TString& prefix, const TString& body) {
        EmailDomPrefix_ = prefix;
        EmailDomBody_ = body;
    }

    void TAnonymiser::SetNickname(const TString& ru, const TString& en) {
        NicknameRu_ = NicknameRuDots_ = ru;
        NicknameEn_ = NicknameEnDots_ = en;

        std::replace(NicknameRuDots_.begin(), NicknameRuDots_.vend(), ' ', '.');
        std::replace(NicknameEnDots_.begin(), NicknameEnDots_.vend(), ' ', '.');
    }

    bool TAnonymiser::MapEmail(TString& email, const TString& login, unsigned& num) const {
        if (EmailDomBody_.empty()) {
            return false;
        }

        TString::size_type pos = email.rfind('.');
        TString suffix = (pos != TString::npos) ? email.substr(pos + 1) : "ru";

        TString out = NUtils::CreateStr(
            login.empty() ? TStringBuf(email).substr(0, email.find('@')) : TStringBuf(login),
            "@",
            EmailDomPrefix_,
            num++,
            EmailDomBody_,
            suffix);
        email = std::move(out);

        return true;
    }

    bool TAnonymiser::MapAttribute(const TString& name, TString& value) const {
        // map nickname/display_name
        if ((!NicknameRu_.empty() || !NicknameEn_.empty()) && name == TAttr::ACCOUNT_DISPLAY_NAME && !value.empty()) {
            MapNickname(value);
            return true;
        }

        // map generic attribute
        TFieldsMap::const_iterator p = AttrMap_.find(name);

        if (p != AttrMap_.end()) {
            value = p->second;
            return true;
        }

        return false;
    }

    bool TAnonymiser::MapPhone(TString& phoneNumber) {
        size_t len = phoneNumber.size();
        if (len < 4) {
            return false;
        }

        TString maskedPhone;
        maskedPhone.reserve(len);

        // take first digit, zeroes and 2 last digits
        maskedPhone.push_back(phoneNumber[0]);
        maskedPhone.append(len - 3, '0');
        maskedPhone.append(phoneNumber, len - 2, TString::npos);

        phoneNumber = std::move(maskedPhone);

        return true;
    }

    void TAnonymiser::MapNickname(TString& value) const {
        TString prefix;
        if (value[0] == 'p' && value[1] == ':') {
            prefix = "p:";
        } else if (value[0] == 's' && value[1] == ':') {
            size_t pos = value.find(':', 2);
            if (pos != TString::npos) {
                pos = value.find(':', pos + 1);
            }

            prefix = TString(value, 0, pos) + ":";
        }

        TString newVal;
        bool haveSpace = (value.find(' ') != TString::npos);

        if (value.find_first_of(RUS_LETTERS) == TString::npos) {
            newVal = haveSpace ? NicknameEn_ : NicknameEnDots_;
        } else {
            newVal = haveSpace ? NicknameRu_ : NicknameRuDots_;
        }

        value = prefix + newVal;
    }
}
