#include "db_fetcher.h"

#include "anonymiser.h"
#include "db_types.h"
#include "exception.h"
#include "experiment.h"
#include "shards_map.h"
#include "strings.h"
#include "utils.h"
#include "ya_domains.h"

#include <passport/infra/daemons/blackbox/src/domain/domain_fetcher.h>

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/string/join.h>

#include <set>

// uncomment for debugging
//#define LOG_QUERIES

namespace NPassport::NBb {
    TDbFetcher::TDbFetcher(NDbPool::TDbPool& dbcentral,
                           const TRangedShardsMap& dbshard,
                           const TYandexDomains* yaDomains,
                           const IDomainFetcher& pddDomains,
                           const TTotpEncryptor* totpEncryptor,
                           const TSyntheticAttributes& synthAttrs,
                           const TAnonymiser* anonymizer)
        : DbCentral_(dbcentral)
        , DbSharded_(dbshard)
        , Anonymizer_(anonymizer)
        , YandexDomains_(yaDomains)
        , HostedDomains_(pddDomains)
        , TotpEncryptor_(totpEncryptor)
        , SyntheticAttributes_(synthAttrs)
        , Profiles_(1)
    {
        Current_ = Profiles_.end();
        DefaultProfile_ = &Profiles_.front();
    }

    TDbIndex TDbFetcher::AddAttr(const TString& attr) {
        TDbIndex id = DefaultProfile_->AddAttr(attr);

        if (TAttr::IsSynthetic(attr)) {
            SyntAttrs_.insert(std::make_pair(attr, SyntheticAttributes_.GetSyntheticFunc(*this, attr)));
        }

        return id;
    }

    TDbIndex TDbFetcher::AddAlias(const TString& alias) {
        return DefaultProfile_->AddAlias(alias);
    }

    TDbIndex TDbFetcher::AddSuid2() {
        return DefaultProfile_->AddSuid2();
    }

    void TDbFetcher::AddExtendedPhoneAttr(const TString& attr) {
        DefaultProfile_->AddExtendedPhoneAttr(attr);

        if (TPhoneAttr::IsSynthetic(attr)) {
            SyntExtendedPhones_.reserve(TExtendedAttrType::MAX_SYN_EXTENDED_COUNT);
            SyntExtendedPhones_.insert(std::make_pair(attr, SyntheticAttributes_.GetSyntheticExtPhoneFunc(*this, attr)));
        }
    }

    void TDbFetcher::AddExtendedEmailAttr(const TString& attr) {
        DefaultProfile_->AddExtendedEmailAttr(attr);

        if (TEmailAttr::IsSynthetic(attr)) {
            SyntExtendedEmails_.reserve(TExtendedAttrType::MAX_SYN_EXTENDED_COUNT);
            SyntExtendedEmails_.insert(std::make_pair(attr, SyntheticAttributes_.GetSyntheticExtEmailFunc(*this, attr)));
        }
    }

    void TDbFetcher::AddExtendedWebauthnAttr(const TString& attr) {
        DefaultProfile_->AddExtendedWebauthnAttr(attr);

        if (TWebauthnAttr::IsSynthetic(attr)) {
            SyntExtendedWebauthn_.reserve(TExtendedAttrType::MAX_SYN_EXTENDED_COUNT);
            SyntExtendedWebauthn_.insert(std::make_pair(attr, SyntheticAttributes_.GetSyntheticExtWebauthnFunc(*this, attr)));
        }
    }

    void TDbFetcher::AddPhoneOperations() {
        DefaultProfile_->AddPhoneOperations();
    }

    void TDbFetcher::AddPhoneBindings(EPhoneBindingsType type) {
        DefaultProfile_->AddPhoneBindings(type);
    }

    void TDbFetcher::SetPhoneAliasMode(EPhoneAliasMode mode) {
        PhoneAliasMode_ = mode;
    }

    void TDbFetcher::AddFamilyInfo() {
        DefaultProfile_->AddFamilyInfo();
    }

    void TDbFetcher::ResetProfiles() {
        Current_ = Profiles_.end();
    }

    void TDbFetcher::ResetCurrentProfile() {
        if (Current_ != Profiles_.end()) {
            Current_->Uid_.clear();
        }
    }

    const TDbProfile* TDbFetcher::NextProfile() {
        // take next (and roll over the end)
        if (Current_ == Profiles_.end()) {
            Current_ = Profiles_.begin();
        } else {
            ++Current_;
        }

        // skip empty uids (i.e. not found in db)
        while (Current_ != Profiles_.end() && Current_->Uid().empty()) {
            ++Current_;
        }

        if (Current_ != Profiles_.end()) {
            return &(*Current_);
        }
        return nullptr;
    }

    const TDbProfile* TDbFetcher::ProfileByUid(const TString& uid) const {
        TProfiles::const_iterator result = std::find_if(
            Profiles_.begin(), Profiles_.end(), [&uid](const TDbProfile& p) { return p.Uid() == uid; });
        if (result == Profiles_.end()) {
            return nullptr;
        }
        return &(*result);
    }

    TDbProfile* TDbFetcher::ProfileByUidImpl(const TString& uid) {
        TProfiles::iterator result = std::find_if(
            Profiles_.begin(), Profiles_.end(), [&uid](const TDbProfile& p) { return p.Uid() == uid; });
        if (result == Profiles_.end()) {
            return nullptr;
        }
        return &(*result);
    }

    void TDbFetcher::SetAttrsBoundary(unsigned boundary) {
        AttrsQueryBoundary_ = boundary;
    }

    // return all non-synthetic attrs as comma-separated string
    TString TDbFetcher::GetAttrsCondition(const TDbProfile::TAttrs& attrs, unsigned boundary) {
        // Here we got some ad-hoc optimizations of db attribute filter
        // More details and experimental results you can see in PASSP-22674
        // In short: when selecting data from blackbox we used to do "WHERE type IN (....)" filter
        // and it turns to be very slow when attribute count increases
        // so, for requests that contain > 50 attributes we just select all (ignore filter completely)
        // for other requests we do 'WHERE type < N OR type IN (...)' choosing N so that 'IN' has not more than 10 elements
        const unsigned typeInLimit = 10;
        size_t totalCount = 0;
        std::vector<int> attrsInt;
        attrsInt.reserve(attrs.size());

        for (const auto& [attrType, attrValue] : attrs) {
            if (attrType.empty() || TAttr::IsSynthetic(attrType)) {
                continue;
            }

            ++totalCount;
            unsigned type = IntFromString<int, 10>(attrType);
            if (type >= boundary) {
                attrsInt.push_back(type);
            }
        }

        if (totalCount > 50) { // to many attributes to select, select all without filtering
            return {};
        }

        if (attrsInt.empty()) { // all requested attributes are < boundary
            return NUtils::CreateStr(" AND type < ", boundary);
        }

        std::sort(attrsInt.begin(), attrsInt.end(), [](int l, int r) { return l > r; });
        if (attrsInt.size() > typeInLimit) { // raise boundary to leave only 10 types for 'type IN'
            attrsInt.erase(attrsInt.begin() + typeInLimit, attrsInt.end());
            boundary = attrsInt.back();
        }

        TString result = NUtils::CreateStrExt(100, " AND (type < ", boundary, " OR type IN (");
        for (int type : attrsInt) {
            NUtils::Append(result, type, ",");
        }
        result.back() = ')';
        result.push_back(')');

        return result;
    }

    TString TDbFetcher::GetAliasTypes(const std::vector<TString>& sids, bool pdd, bool allowScholar) const {
        if (sids.empty()) {
            if (pdd) {
                DefaultProfile_->AddAlias("7");
                DefaultProfile_->AddAlias("9");
                return "7,9";
            }
            DefaultProfile_->AddAlias("1");
            DefaultProfile_->AddAlias("5");
            if (allowScholar) {
                DefaultProfile_->AddAlias("23");
            }
            return allowScholar ? "1,5,23" : "1,5";
        }
        std::vector<bool> types(26, false);
        for (const TString& sid : sids) {
            if (sid == "2") {
                if (pdd) {
                    types[7] = true;
                    types[9] = true;
                } else {
                    types[1] = true;
                    types[2] = true;
                }
            } else if (sid == "16") {
                types[1] = true;
                types[2] = true;
                types[3] = true;
            } else if (sid == "33") {
                types[5] = true;
            } else if (sid == "58") {
                types[6] = true;
            } else if (sid == "61") {
                types[9] = true;
            } else if (sid == "65") {
                if (PhoneAliasMode_ != EPhoneAliasMode::ForceOff) {
                    types[11] = true;
                }
            } else if (sid == "68") {
                types[10] = true;
            } else if (sid == "105") {
                types[8] = true;
            } else if (sid == "669") {
                types[13] = true;
            } else if (sid == TStrings::MAILISH_SID) {
                types[12] = true;
            } else if (sid == TStrings::UBER_SID) {
                types[16] = true;
            } else if (sid == TStrings::NEOPHONISH_SID) {
                types[21] = true;
            } else if (sid == TStrings::FEDERAL_SID) {
                types[24] = true;
            } else if (sid == TStrings::BANK_SID) {
                types[25] = true;
            }
        }

        TString res;
        res.reserve(3 * types.size());
        for (size_t i = 0; i < types.size(); ++i) {
            if (types[i]) {
                TString alias = IntToString<10>(i);
                NUtils::AppendSeparated(res, ',', alias);
                DefaultProfile_->AddAlias(alias);
            }
        }

        return res;
    }

    // find a profile searching by login in given aliases in priority order
    // we should remember aliases we checked to prevent finding them by lower priority alias
    // i.e. if we have account with 2 aliases:  1:foo, 2:bar
    // and searching for login='foo' aliases=2,1 then
    // at first run  we don't find 2:foo, at second we find 1:foo but should ignore it since it's mail is different
    // Moreover, since we don't know actually if account has this sid subscription,
    // we need to check all matching accounts and later check in attributes which one actually is found
    std::vector<TDbFetcher::TProfiles::iterator> TDbFetcher::FindProfiles(const TString& login,
                                                                          const std::vector<TString>& aliases) {
        std::vector<TProfiles::iterator> result;
        result.reserve(aliases.size());
        std::set<TString> matching;
        std::set<TString> nonMatching;
        for (const TString& alias : aliases) {
            for (TProfiles::iterator p = Profiles_.begin(); p != Profiles_.end(); ++p) {
                // ignore empty profiles or those we already know don't match
                if (p->Uid_.empty() || nonMatching.find(p->Uid_) != nonMatching.end()) {
                    continue;
                }

                TDbProfile::EMatchStatus st = p->MatchesLogin(login, alias);
                if (st == TDbProfile::Match && matching.find(p->Uid_) == matching.end()) { // found new account
                    result.push_back(p);
                    matching.insert(p->Uid_);
                } else if (st == TDbProfile::NoMatch) {
                    nonMatching.insert(p->Uid_);
                }
            }
        }
        return result;
    }

    std::vector<TDbFetcher::TProfileSidPair> TDbFetcher::SelectProfiles(const TString& login,
                                                                        const std::vector<TString>& sids,
                                                                        bool pdd) {
        std::vector<TProfileSidPair> result;
        if (sids.empty()) { // search by default alias group
            static const std::vector<TString> PDD_ALIASES = {TAlias::ALT_DOMAIN_LOGIN, TAlias::PDD_MASTER_LOGIN};
            static const std::vector<TString> OTHER_ALIASES = {TAlias::PORTAL_LOGIN, TAlias::LITE_LOGIN, TAlias::SCHOLAR};

            const std::vector<TString>& aliases = pdd ? PDD_ALIASES
                                                      : OTHER_ALIASES;

            std::vector<TProfiles::iterator> sidResult = FindProfiles(login, aliases);
            // since there's no sid to check subscription, we'll select the first account anyway
            if (!sidResult.empty()) {
                result.push_back(std::make_pair(sidResult.front(), TStrings::EMPTY));
            }
            return result;
        }

        // look for each sid in given order and check if it matches by corresponding aliases
        result.reserve(3 * sids.size());
        for (const TString& sid : sids) {
            const std::vector<TString>* aliases = nullptr;
            if (sid == "2") {
                if (pdd) {
                    static const std::vector<TString> VEC_ = {TAlias::ALT_DOMAIN_LOGIN, TAlias::PDD_MASTER_LOGIN};
                    aliases = &VEC_;
                } else {
                    static const std::vector<TString> VEC_ = {TAlias::MAIL_LOGIN, TAlias::PORTAL_LOGIN};
                    aliases = &VEC_;
                }
            } else if (sid == "16") {
                static const std::vector<TString> VEC_ = {TAlias::NAROD_MAIL_LOGIN, TAlias::MAIL_LOGIN, TAlias::PORTAL_LOGIN};
                aliases = &VEC_;
            } else if (sid == "33") {
                static const std::vector<TString> VEC_ = {TAlias::LITE_LOGIN};
                aliases = &VEC_;
            } else if (sid == "58") {
                static const std::vector<TString> VEC_ = {TAlias::SOCIAL_LOGIN};
                aliases = &VEC_;
            } else if (sid == "61") {
                static const std::vector<TString> VEC_ = {TAlias::ALT_DOMAIN_LOGIN};
                aliases = &VEC_;
            } else if (sid == "65") {
                static const std::vector<TString> VEC_ = {TAlias::PHONE_NUMBER};
                aliases = &VEC_;
            } else if (sid == "68") {
                static const std::vector<TString> VEC_ = {TAlias::PHONY_LOGIN};
                aliases = &VEC_;
            } else if (sid == "105") {
                static const std::vector<TString> VEC_ = {TAlias::PDD_ALIAS_LOGIN};
                aliases = &VEC_;
            } else if (sid == "669") {
                static const std::vector<TString> VEC_ = {TAlias::YANDEXOID_LOGIN};
                aliases = &VEC_;
            } else if (sid == TStrings::MAILISH_SID) {
                static const std::vector<TString> VEC_ = {TAlias::MAILISH_LOGIN};
                aliases = &VEC_;
            } else if (sid == TStrings::UBER_SID) {
                static const std::vector<TString> VEC_ = {TAlias::UBER_ID};
                aliases = &VEC_;
            } else if (sid == TStrings::NEOPHONISH_SID) {
                static const std::vector<TString> VEC_ = {TAlias::NEOPHONISH};
                aliases = &VEC_;
            } else if (sid == TStrings::FEDERAL_SID) {
                static const std::vector<TString> VEC_ = {TAlias::FEDERAL};
                aliases = &VEC_;
            } else if (sid == TStrings::BANK_SID) {
                static const std::vector<TString> VEC_ = {TAlias::BANK_PHONE_NUMBER};
                aliases = &VEC_;
            } else {
                continue;
            }

            if (aliases == nullptr) {
                throw yexception() << "Aliases must not be NULL";
            }

            for (TProfiles::iterator sidIt : FindProfiles(login, *aliases)) {
                result.push_back(std::make_pair(sidIt, sid));
            }
        }

        return result;
    }

    // here we select profile by alias in order of given alias types
    // for independent aliases (i.e. which does not default to another one)
    // DON'T use this for mail/narod/narodmail aliases, it will not work correctly!
    TDbFetcher::TProfiles::iterator TDbFetcher::SelectProfile(const TString& alias,
                                                              const std::vector<TString>& types) {
        for (const TString& t : types) {
            for (TProfiles::iterator p = Profiles_.begin(); p != Profiles_.end(); ++p) {
                if (p->Uid_.empty()) {
                    continue;
                }
                if (p->MatchesLogin(alias, t) == TDbProfile::Match) {
                    return p;
                }
            }
        }

        return Profiles_.end();
    }

    void TDbFetcher::FetchByUid(const TString& uid) {
        return FetchByUids(std::vector<TString>(1, uid));
    }

    void TDbFetcher::FetchByUids(const std::vector<TString>& uids) {
        if (uids.empty()) {
            return;
        }
        // First, prepare profiles to be filled up
        int count = uids.size() - Profiles_.size();

        DefaultProfile_->FillDefaultAttrValues();

        while (count-- > 0) { // prepare n copies of default_profile
            Profiles_.push_back(*DefaultProfile_);
        }

        TString uidsStr;
        uidsStr.reserve(10 * uids.size());

        // fill the uids
        TProfiles::iterator itProfile = Profiles_.begin();
        for (const TString& uid : uids) {
            if (uid.empty()) {
                continue;
            }

            itProfile++->Uid_.assign(uid);
            NUtils::AppendSeparated(uidsStr, ',', uid);
        }

        NDbPool::TNonBlockingHandle centralSqlh;
        if (DefaultProfile_->NeedAliases()) {
            TString select = "al.uid,al.type,al.value";
            TString from = "aliases al";
            TString where = (uids.size() > 1) ? "al.uid IN (" + uidsStr + ")" : "al.uid=" + uidsStr;

            AddMoreTablesToQuery(3, select, from);

            TString query = NUtils::CreateStr(
                "SELECT ", select, " FROM ", from, " WHERE ", where, " ORDER BY al.uid");

#ifdef LOG_QUERIES
            TLog::Error("QUERY (central): '%s'", query.c_str());
#endif
            centralSqlh = NDbPool::TNonBlockingHandle(DbCentral_);
            centralSqlh.SendQuery(query);
        }

        FetchRawAttrs();

        if (centralSqlh) {
            try {
                NDbPool::TNonBlockingHandle sqlh(std::move(centralSqlh));
                ReadCentralQuery(*sqlh.WaitResult(), false);
            } catch (const NDbPool::TException& e) {
                TLog::Debug("BlackBox: dbpool exception querying central : '%s'", e.what());
                throw TDbpoolError("dbpool exception in dbfield(s) fetch", e.what());
            }
        }

        DoAttrsPostprocessing();

        // erase uids not found in db
        for (TDbProfile& profile : Profiles_) {
            if (profile.Empty_) {
                profile.Uid_.clear();
            }
        }
    }

    void TDbFetcher::FetchBySuid(const TString& suid, const TString& sid) {
        if (sid != "2") {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid sid, allowed sids: 2; got " << InvalidValue(sid);
        }

        // First, find user and fetch basic data from aliases
        TString select = "al.uid,al.type,al.value";
        TString from = "suid" + sid + " s LEFT JOIN aliases al USING (uid)";
        TString where = "s.suid=" + suid; // + " AND al.type IN (" + aliases + ")";

        AddMoreTablesToQuery(3, select, from);
        DefaultProfile_->Sid_ = sid;

        DefaultProfile_->FillDefaultAttrValues();

        {
            NDbPool::TBlockingHandle sqlh(DbCentral_);

            TString query = NUtils::CreateStr(
                "SELECT ", select, " FROM ", from, " WHERE ", where, " ORDER BY al.uid, al.type");

            ReadCentralQuery(*DoCentralDbQuery(sqlh, query), true);
        }

        FetchAttrs();

        for (TDbProfile& profile : Profiles_) {
            if (profile.Empty_) {
                profile.Uid_.clear();
            }
        }
    }

    namespace {
        bool HasMailSid(const std::vector<TString>& sidArray) {
            for (const TString& sid : sidArray) {
                if (sid == TStrings::MAIL_SID || sid == TStrings::MAIL_NAROD_SID) {
                    return true;
                }
            }

            return false;
        }
    }

    void TDbFetcher::FetchByLogin(const TString& login, const TString& domid, const std::vector<TString>& sidArray, const TString& defUid, bool allowScholar) {
        TString realLogin;
        if (domid.empty()) {
            realLogin.assign(login);
        } else {
            realLogin.assign(domid).push_back('/');
            realLogin.append(login, 0, login.find('@'));
        }

        // should recheck in 'for' the subscription by suid2 & mail_host_id
        TDbIndex mailSuid = -1;
        if (HasMailSid(sidArray)) {
            mailSuid = AddSuid2();
        }

        DefaultProfile_->FillDefaultAttrValues();

        DoCentralDbQuery(realLogin, GetAliasTypes(sidArray, !domid.empty(), allowScholar), defUid);

        // filter by alias, then fetch all data
        std::vector<TProfileSidPair> candidates = SelectProfiles(login, sidArray, !domid.empty());

        // Since we don't have all info in aliases table, i.e. if account has no mail alias
        // it can mean that it has no mail subscription or that its mail alias does not differ from passport one
        // so we have a list of candidates here, usually only one, but in some bad cases here can be 2-4 accounts
        // and we need to check in attributes that the candidate really has the subscription
        // Note that in candidate

        // for PDD if profile not found, look for default uid profile
        if (candidates.empty() && !defUid.empty()) {
            TProfiles::iterator prof = std::find_if(Profiles_.begin(), Profiles_.end(), [&defUid](const TDbProfile& p) { return p.Uid() == defUid; });
            if (prof != Profiles_.end()) {
                prof->CatchAll_ = true;
                candidates.push_back(std::make_pair(prof, TStrings::EMPTY));
            }
        }

        // filter out all profiles that are not candidates for our search
        std::set<TString> candSet;
        for (const TProfileSidPair& pair : candidates) {
            if (pair.first->Uid_.empty()) {
                continue;
            }

            if (pair.second == TStrings::MAIL_SID || pair.second == TStrings::MAIL_NAROD_SID) {
                // for mail/narodmail sid - add MailHostId and MailStatus attributes to check
                if (mailSuid < 0 || !pair.first->Get(mailSuid)->Exists) {
                    continue;
                }

                AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID);
                pair.first->AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID);
                AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS);
                pair.first->AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS);
            } else if (pair.second == TStrings::SID_65 && PhoneAliasMode_ == EPhoneAliasMode::ByAttribute) {
                // for phone alias sid - add Account_EnableSearchByPhoneAlias to check
                AddAttr(TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS);
                pair.first->AddAttr(TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS);
            }
            candSet.insert(pair.first->Uid_);
        }

        for (TDbProfile& profile : Profiles_) {
            if (candSet.find(profile.Uid_) == candSet.end()) { // clear those not in candidate set
                profile.Uid_.clear();
            }
        }

        FetchRawAttrs();

        // Now we need to select one profile to return as result and then clear all others
        // We do this by searching in order and checking whicn subscription it actually has
        TProfiles::iterator prof = Profiles_.end();

        // TODO: rewrite this cycle
        for (const TProfileSidPair& pair : candidates) {
            TDbIndex a = -1;
            if (pair.first->Uid_.empty()) {
                continue;
            }

            if (pair.second == TStrings::MAIL_SID || pair.second == TStrings::MAIL_NAROD_SID) {
                a = pair.first->AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS);
                if (pair.first->Get(a)->Value == TMailStatus::ACTIVE) { // found active mail subscription
                    prof = pair.first;
                    prof->Sid_ = pair.second;
                    break;
                }
                if (pair.first->Get(a)->Value == TMailStatus::FROZEN) { // found frozen subscription, ignore
                    continue;
                }
                a = pair.first->AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID);
            } else if (pair.second == TStrings::SID_65 && PhoneAliasMode_ == EPhoneAliasMode::ByAttribute) {
                a = pair.first->AddAttr(TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS);
                if (pair.first->Get(a)->AsBoolean()) { // check if search enabled by attribute
                    prof = pair.first;
                    prof->Sid_ = pair.second;
                    break;
                }
                continue;

            } else { // no sid to check or some other sid, found account
                prof = pair.first;
                prof->Sid_ = pair.second;
                break;
            }

            if (a >= 0 && pair.first->Get(a)->Exists) {
                prof = pair.first;
                prof->Sid_ = pair.second;
                break;
            }
        }

        // actually clear all others
        for (TProfiles::iterator it = Profiles_.begin(); it != Profiles_.end(); ++it) {
            if (it != prof) {
                it->Uid_.clear();
            }
        }

        DoAttrsPostprocessing();
    }

    void TDbFetcher::FetchByAlias(const TString& alias, const std::vector<TString>& types, const TString& defUid) {
        DefaultProfile_->FillDefaultAttrValues();

        for (const TString& t : types) {
            DefaultProfile_->AddAlias(t);
        }

        DoCentralDbQuery(alias, JoinSeq(",", types), defUid);

        // Here we need to select one profile even if db returned many
        TProfiles::iterator prof = SelectProfile(alias, types);
        for (TProfiles::iterator it = Profiles_.begin(); it != Profiles_.end(); ++it) {
            if (it != prof) {
                it->Uid_.clear();
            }
        }

        FetchAttrs();
    }

    void TDbFetcher::FetchAliasesByFamilyId(const TString& familyId, bool getPlace) {
        // First, find user and fetch basic data from aliases
        TString select = "al.uid,al.type,al.value";
        TString from = "aliases al ";

        // We need it in any way
        DefaultProfile_->NeedFamilyInfo_ = true;
        AddMoreTablesToQuery(3, select, from, getPlace);

        DefaultProfile_->FillDefaultAttrValues();

        NDbPool::TBlockingHandle sqlh(DbCentral_);

        TString where = NUtils::CreateStr(
            "al.uid IN (SELECT uid FROM family_members WHERE family_id=",
            sqlh.EscapeQueryParam(familyId),
            ")");

        TString query = NUtils::CreateStr(
            "SELECT ", select, " FROM ", from, " WHERE ", where);

        ReadCentralQuery(*DoCentralDbQuery(sqlh, query), true);

        for (TDbProfile& profile : Profiles_) {
            if (profile.Empty_) {
                profile.Uid_.clear();
            }
        }
    }

    void TDbFetcher::FetchAttrs() {
        FetchRawAttrs();
        DoAttrsPostprocessing();
    }

    // fetch a number of possible accounts by alias of given types
    void TDbFetcher::DoCentralDbQuery(const TString& alias, const TString& types, const TString& defUid) {
        if (alias.empty() || types.empty()) {
            return;
        }

        NDbPool::TBlockingHandle sqlh(DbCentral_);

        TString select = "al.uid,al.type,al.value";
        TString from = "(SELECT 1) d LEFT JOIN aliases a0 ON a0.type IN (" + types + ") AND a0.value='" + sqlh.EscapeQueryParam(alias) + "' LEFT JOIN aliases al ON al.uid=";

        if (defUid.empty()) {
            from.append("a0.uid");
        } else {
            from.append("IFNULL(a0.uid,").append(defUid).push_back(')');
        }

        AddMoreTablesToQuery(3, select, from);

        TString query = NUtils::CreateStr("SELECT ", select, " FROM ", from, " ORDER BY al.uid");

        ReadCentralQuery(*DoCentralDbQuery(sqlh, query), true);
    }

    // Consider we have accounts/uids already fetched in our profiles list, fetch attrs by uids
    void TDbFetcher::FetchRawAttrs() {
        std::vector<TString> uidList;
        uidList.reserve(Profiles_.size());

        for (const TDbProfile& profile : Profiles_) {
            if (profile.Uid().empty()) {
                continue;
            }

            uidList.push_back(profile.Uid());
        }

        // check if we have extended attributes to ask
        bool needPhoneAttrs = DefaultProfile_->ExtendedPhones_.find(TStrings::EMPTY) != DefaultProfile_->ExtendedPhones_.end();
        bool needEmailAttrs = DefaultProfile_->ExtendedEmails_.find(TStrings::EMPTY) != DefaultProfile_->ExtendedEmails_.end();
        bool needWebauthnAttrs = DefaultProfile_->ExtendedWebauthn_.find(TStrings::EMPTY) != DefaultProfile_->ExtendedWebauthn_.end();

        if (uidList.empty()) {
            return;
        }

        DoShardedDbQuery(uidList, needPhoneAttrs, needEmailAttrs, needWebauthnAttrs);
    }

    std::unique_ptr<NDbPool::TResult> TDbFetcher::DoCentralDbQuery(NDbPool::TBlockingHandle& db, const TString& query) {
#ifdef LOG_QUERIES
        TLog::Error("QUERY (central): '%s'", query.c_str());
#endif
        try {
            return db.Query(query);
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: dbpool exception in the central db query '%s' : '%s'", query.c_str(), e.what());
            throw TDbpoolError("dbpool exception in dbfield(s) fetch", e.what());
        }
    }

    // Find the accounts
    // if add_profiles is false, consider all profiles are ready and uids filled, need to fill other data
    // if add_profiles is true, add profiles to the list each time new uid is found
    void TDbFetcher::ReadCentralQuery(NDbPool::TResult& result, bool addProfiles) {
        std::unique_ptr<TDbProfile> emptyProfile;

        // a small optimization to minimize searching in the profile list
        // since result is sorted by uid we search for a profile only when uid changes
        TDbProfile* currentProfile = nullptr;

        if (!addProfiles) {
            // find first non-empty uid
            TProfiles::iterator p = Profiles_.begin();
            while (p != Profiles_.end() && p->Uid_.empty()) {
                ++p;
            }

            if (p == Profiles_.end()) {
                throw yexception() << "BlackBox internal error: no initialized DbProfiles found";
            }
            currentProfile = &(*p);
        } else {
            emptyProfile = std::make_unique<TDbProfile>(*DefaultProfile_);
            currentProfile = &Profiles_.front();
        }

        for (const NDbPool::TRow& row : result.Table()) {
            const TString uid = row[0].IsNull() ? TStrings::EMPTY : row[0].AsString();

            if (uid.empty()) { // a stub for user that is not found
                continue;
            }

            if (uid != currentProfile->Uid_) {
                TDbProfile* newProfile = ProfileByUidImpl(uid);
                if (newProfile) {
                    currentProfile = newProfile;
                } else {
                    if (addProfiles) { // no new profile, should find empty or add a new one
                        TProfiles::iterator p = Profiles_.begin();
                        while (p != Profiles_.end() && !p->Uid_.empty()) {
                            ++p;
                        }

                        if (p == Profiles_.end()) { // add new one
                            Profiles_.push_back(*emptyProfile);
                            currentProfile = &(Profiles_.back());
                        } else {
                            currentProfile = &(*p);
                        }
                        currentProfile->Uid_ = uid;
                    } else { // not allowed to add profiles and uid unknown
                        throw yexception() << "BlackBox internal error: db request returned unexpected uid=" << uid;
                    }
                }
            }

            TString type = row[1].IsNull() ? TStrings::EMPTY : row[1].AsString();
            TString value = row[2].IsNull() ? TStrings::EMPTY : row[2].AsString();
            TExperiment::Get().RunAliasCheck();

            if (value.empty()) { // ignore empty aliases, should not have them in db
                TLog::Warning("Got empty alias value from db: uid=%s, type=%s", uid.c_str(), type.c_str());
                continue;
            }

            if (type == TAlias::PDD_MASTER_LOGIN || type == TAlias::PDD_ALIAS_LOGIN) {
                TString::size_type pos = value.find('/');
                if (pos != TString::npos) {
                    TString domid = value.substr(0, pos);

                    TDomainFetcher::TResult domRes;
                    if ((domRes = HostedDomains_.FindById(domid)) ||
                        (domRes = HostedDomains_.FindInDb(domid, true))) {
                        if (type == TAlias::PDD_MASTER_LOGIN) {
                            currentProfile->DomItem_ = *domRes.Domain;
                            currentProfile->DomCache_ = std::move(domRes.Cache);
                        }
                        value.erase(0, pos + 1);
                        value.push_back('@');
                        value.append(domRes.Domain->UtfName());
                    } else {
                        TLog::Warning("Got alias with unknown domain_id: '%s', uid=%s, domain_id=%s", value.c_str(), uid.c_str(), domid.c_str());
                        continue; // pdd with unknown domain, skip
                    }
                } else {
                    continue; // malformed alias, skip
                }
            }

            if (type == TAlias::ALT_DOMAIN_LOGIN) {
                TString::size_type pos = value.find('/');
                if (pos != TString::npos) {
                    TString domid = value.substr(0, pos);
                    currentProfile->AltDomain_ = YandexDomains_->AltDomain(domid);

                    if (!currentProfile->AltDomain_.empty()) {
                        value.erase(0, pos + 1);
                        value.push_back('@');
                        value.append(currentProfile->AltDomain_);
                        currentProfile->AltDomainId_ = domid;
                    } else {
                        TLog::Warning("Got alias with unknown alt_domain_id: '%s', uid=%s, alt_domain_id=%s", value.c_str(), uid.c_str(), domid.c_str());
                        continue; // unknown alt domain, skip
                    }
                } else {
                    continue; // malformed alias, skip
                }
            }

            currentProfile->Empty_ = false;

            // fill aliases
            if (type == TAlias::PDD_ALIAS_LOGIN) {
                // dont store PDD aliases if not asked
                if (currentProfile->NeedPddAliases_) {
                    currentProfile->AliasesPdd_.insert(value);
                }
                continue; // don't read suid and family info for each pdd alias, it should be too many of them
            }
            if (type == TAlias::OLDPUBLICID) {
                // this is very rare alias and maximum 1 is expected so we always fill it
                currentProfile->AliasesOldPublic_.insert(value);
            } else {
                using TAliasIter = TDbProfile::TAliases::iterator;
                TAliasIter res = currentProfile->Aliases_.find(type);

                if (res != currentProfile->Aliases_.end() && !res->second.Exists) { // empty alias, fill it
                    res->second.Exists = true;
                    res->second.Value = std::move(value);
                }
            }

            ReadSuid(row, *currentProfile);
            ReadFamilyInfo(row, *currentProfile);
            ReadWebauthnCredentialIds(row, *currentProfile);
        }
    }

    void TDbFetcher::ReadShardQuery(NDbPool::TResult& result) {
        // a small optimization to minimize searching in the profile list
        // since result is sorted by uid we search for a profile only when uid changes
        TDbProfile* currentProfile;
        // find first non-empty uid
        TProfiles::iterator p = Profiles_.begin();
        while (p != Profiles_.end() && p->Uid_.empty()) {
            ++p;
        }

        if (p == Profiles_.end()) { // couldn't happen, we should have at least 1 uid to search for
            TLog::Error("BlackBox internal error: DbProfiles list inconsistency");
            return;
        }
        currentProfile = &(*p);

        unsigned anonymizerNumber = 1;

        for (const NDbPool::TRow& row : result.Table()) {
            TString uid = row[0].IsNull() ? TStrings::EMPTY : row[0].AsString();

            if (uid != currentProfile->Uid_) {
                TDbProfile* newProfile = ProfileByUidImpl(uid);
                if (!newProfile) {
                    TLog::Error("BlackBox internal error: db request returned unknown uid=%s", uid.c_str());
                    continue;
                }
                currentProfile = newProfile;
                anonymizerNumber = 1; // reset the numbering for new user
            }

            TString type = row[1].IsNull() ? TStrings::EMPTY : row[1].AsString();
            TString value = row[2].IsNull() ? TStrings::EMPTY : row[2].AsString();
            TString entityType = row.size() > 3 && !row[3].IsNull() ? row[3].AsString() : TStrings::EMPTY;
            TExperiment::Get().RunFullAttrsCheck();

            if (entityType.empty() || entityType == TExtendedAttrType::NONE) {
                // this is plain simple attribute from attributes table

                // anonymize attribute if needed
                if (Anonymizer_) {
                    Anonymizer_->MapAttribute(type, value);
                }

                TDbProfile::TAttrs::iterator it = currentProfile->Attrs_.find(type);

                if (it != currentProfile->Attrs_.end()) {
                    it->second.Exists = true;
                    it->second.Value = std::move(value);
                } else {
                    it = currentProfile->Attrs_.insert(std::make_pair(type, TDbValue(TStrings::EMPTY))).first;
                    it->second.Value = std::move(value);
                }
            } else {
                // this is extended attribute
                TString entityId = row[4].IsNull() ? TStrings::EMPTY : row[4].AsString();

                if (entityType == TExtendedAttrType::PHONE_OPERATION) {
                    // this is actually a phone operation returned as extended attribute
                    currentProfile->PhoneOperations_.insert(std::make_pair(entityId, value));
                    continue;
                }

                if (entityType == TExtendedAttrType::PHONE_BINDINGS) {
                    // this is actually a phone binding returned as extended attribute
                    std::vector<TString> splittedValue = NUtils::ToVector(value, ',');
                    if (splittedValue.size() != 3) {
                        TLog::Error("BlackBox internal error: unexpected response from phone_bindings table");
                        throw yexception() << "Broken phone_bindings processing";
                    }

                    // - if 'unbound' found and we need only current - skip
                    // - if 'current' found and we need only unbound - skip
                    TString& bound = splittedValue[1];
                    if (bound == TStrings::ZERO) {
                        if (DefaultProfile_->PhoneBindingsType_ == EPhoneBindingsType::Current) {
                            continue;
                        }
                    } else if (DefaultProfile_->PhoneBindingsType_ == EPhoneBindingsType::Unbound) {
                        continue;
                    }
                    TString& number = splittedValue[0];
                    TString& flags = splittedValue[2];

                    TPhoneBindingsChunk::TPhoneBindingsItem pbItem;
                    pbItem.Type = bound == TStrings::ZERO ? TStrings::UNBOUND : TStrings::CURRENT;
                    pbItem.Number = std::move(number);
                    pbItem.PhoneId = std::move(entityId);
                    pbItem.Uid = std::move(uid);
                    pbItem.Bound = std::move(bound);
                    pbItem.Flags = std::move(flags);

                    currentProfile->PhoneBindings_.push_back(std::move(pbItem));
                    continue;
                }

                TDbProfile::TExtendedEntities* entities;
                if (TExtendedAttrType::PHONE == entityType) {
                    entities = &currentProfile->ExtendedPhones_;
                    if (type == TPhoneAttr::NUMBER || type == TPhoneAttr::SECURED) {
                        TExperiment::Get().RunMainAttrsCheck();
                    }
                } else if (TExtendedAttrType::EMAIL == entityType) {
                    entities = &currentProfile->ExtendedEmails_;
                    if (type == TEmailAttr::ADDRESS || type == TEmailAttr::IS_UNSAFE) {
                        TExperiment::Get().RunMainAttrsCheck();
                    }
                } else if (TExtendedAttrType::WEBAUTHN == entityType) {
                    entities = &currentProfile->ExtendedWebauthn_;
                    if (type == TWebauthnAttr::PUBLIC_KEY) {
                        TExperiment::Get().RunMainAttrsCheck();
                    }
                } else {
                    TLog::Error("BlackBox internal error: unknown extended attribute entity_type value");
                    continue;
                }

                if (entities->empty()) { // shouldn't happen since we asked only for types we have in the map
                    TLog::Error("BlackBox internal error: entities must not be empty");
                    continue;
                }

                TDbProfile::TExtendedEntities::iterator curEntity = entities->find(entityId);
                if (curEntity == entities->end()) {
                    curEntity = entities->insert({entityId, (*entities)[TStrings::EMPTY]}).first;
                }

                // anonymize extended attribute if needed
                if (Anonymizer_) {
                    if (entityType == TExtendedAttrType::EMAIL && type == TEmailAttr::ADDRESS) {
                        // map email address
                        // we try to mimic DbFieldsConverter::mailLoginField() logic,
                        // i.e. try to determine sid=2 login to use as login part for fake emails

                        // yes, we'll construct this login one time for each email,
                        // but this is only for mimino so we don't care for performance
                        TString login;

                        if (currentProfile->Pdd()) {
                            login = currentProfile->GetAlias(TAlias::PDD_MASTER_LOGIN);
                        } else {
                            login = currentProfile->GetAlias(TAlias::MAIL_LOGIN);
                            if (login.empty()) {
                                login = currentProfile->GetAlias(TAlias::PORTAL_LOGIN);
                                if (login.empty()) {
                                    login = currentProfile->GetAlias(TAlias::ALT_DOMAIN_LOGIN);
                                }
                            }
                        }
                        // if it is an email (portal or alt), cut off domain part
                        size_t atPos = login.find('@');
                        if (atPos != TString::npos) {
                            login.resize(atPos);
                        }

                        Anonymizer_->MapEmail(value, login, anonymizerNumber);
                    } else if (entityType == TExtendedAttrType::PHONE && type == TPhoneAttr::NUMBER) {
                        // map phone number
                        TAnonymiser::MapPhone(value);
                    }
                }

                TDbValue& val = curEntity->second[type];
                val.Exists = true;
                val.Value = std::move(value);
            }
        }
    }

    static const TString ATTRS_QUERY = "SELECT uid,type,value FROM attributes WHERE uid IN (";
    static const TString ATTRS_QUERY2 = "(SELECT uid,type,value,0 AS entity_type,0 AS entity_id FROM attributes WHERE uid IN (";
    static const TString UNION = " UNION ALL ";
    static const TString EXT_ATTRS_QUERY = "(SELECT uid,type,value,entity_type,entity_id FROM extended_attributes WHERE uid IN (";
    static const TString PHONE_BINDINGS_QUERY =
        "(SELECT uid, 0 as type, CONCAT_WS(',',number,unix_timestamp(bound),flags) AS value,"
        " 102 AS entity_type, phone_id as entity_id FROM phone_bindings WHERE uid IN (";
    static const TString ENTITY_TYPE_COND = " AND entity_type IN (";

    void TDbFetcher::DoShardedDbQuery(const std::vector<TString>& uidsList,
                                      bool needPhoneAttrs, bool needEmailAttrs, bool needWebauthnAttrs) {
        std::vector<TString> uids = DbSharded_.SplitUids(uidsList);
        const bool haveExtended = needPhoneAttrs ||
                                  needEmailAttrs ||
                                  needWebauthnAttrs ||
                                  DefaultProfile_->NeedPhoneOperations_ ||
                                  DefaultProfile_->PhoneBindingsType_ != EPhoneBindingsType::None;

        TString attrsCond = GetAttrsCondition(DefaultProfile_->Attrs(), AttrsQueryBoundary_);

        std::vector<NDbPool::TNonBlockingHandle> handles;
        handles.reserve(DbSharded_.GetShardCount());
        for (TShardRanges::TShard s = 0; s < DbSharded_.GetShardCount(); ++s) {
            if (uids[s].empty()) {
                continue;
            }

            TString query;
            // reserve memory for (attr query + phone_operations) * approximate sub-query size
            query.reserve(2 * (100 + uids[s].size() + attrsCond.size()));

            // select from attributes where uid in (...) and type in (...)
            NUtils::Append(query, haveExtended ? ATTRS_QUERY2 : ATTRS_QUERY, uids[s], ')', attrsCond);
            if (haveExtended) {
                query.push_back(')');
            }

            // select from extended_attributes where uid in (...) and entity_type IN (...)
            if (needPhoneAttrs || needEmailAttrs || needWebauthnAttrs) {
                NUtils::AppendSeparated(query, UNION, EXT_ATTRS_QUERY);
                NUtils::Append(query, uids[s], ')');

                if (needPhoneAttrs && needEmailAttrs && needWebauthnAttrs) { // if we need all - no filter by entity_type
                    query.push_back(')');
                } else { // we need some extended attrs
                    TString entityTypes;
                    if (needPhoneAttrs) {
                        NUtils::AppendSeparated(entityTypes, ',', TExtendedAttrType::PHONE);
                    }
                    if (needEmailAttrs) {
                        NUtils::AppendSeparated(entityTypes, ',', TExtendedAttrType::EMAIL);
                    }
                    if (needWebauthnAttrs) {
                        NUtils::AppendSeparated(entityTypes, ',', TExtendedAttrType::WEBAUTHN);
                    }
                    NUtils::Append(query, ENTITY_TYPE_COND, entityTypes, "))");
                }
            }

            if (DefaultProfile_->NeedPhoneOperations_) {
                if (!query.empty()) {
                    query.append(UNION);
                }
                TQueries::AppendPhoneOperationsField(query, uids[s]);
            }

            if (DefaultProfile_->PhoneBindingsType_ != EPhoneBindingsType::None) {
                NUtils::AppendSeparated(query, UNION, PHONE_BINDINGS_QUERY).append(uids[s]).append("))");
            }

            NDbPool::TNonBlockingHandle sqlh(DbSharded_.GetPool(s));

#ifdef LOG_QUERIES
            TLog::Error("QUERY (shard #%d): '%s'", s + 1, query.c_str());
#endif
            sqlh.SendQuery(query);
            handles.push_back(std::move(sqlh));
        }

        for (NDbPool::TNonBlockingHandle& sqlh : handles) {
            ReadShardQuery(*TUtils::WaitResult(sqlh, "dbpool exception in dbfield(s) fetch"));
        }
    }

    void TDbFetcher::AddMoreTablesToQuery(unsigned col, TString& select, TString& from, bool getPlace) {
        col = AddQuerySuids(col, select, from);
        col = AddQueryFamilyInfo(col, select, from, getPlace);
        col = AddQueryWebauthnCredentials(col, select, from);
    }

    // add suids tables to the query, basing on what suids were needed, col is next column index
    // i.e. add "SELECT ... , sN.suid FROM ... LEFT JOIN suidN sN USING (uid) ...
    unsigned TDbFetcher::AddQuerySuids(unsigned col, TString& select, TString& from) {
        if (DefaultProfile_->Suid2_.Index == -1) {
            return col;
        }

        // only sid=2 is supported now
        NUtils::Append(select, ",s2.suid");
        NUtils::Append(from, " LEFT JOIN suid2 s2 ON s2.uid=al.uid");

        Suid2Col_ = col++;
        return col;
    }

    unsigned TDbFetcher::AddQueryFamilyInfo(unsigned col, TString& select, TString& from, bool getPlace) {
        if (!DefaultProfile_->NeedFamilyInfo_) {
            return col;
        }

        NUtils::Append(select, ",fi.family_id,fi.admin_uid,fi.meta");
        NUtils::Append(from, " LEFT JOIN family_members fm ON al.uid=fm.uid LEFT JOIN family_info fi ON fm.family_id=fi.family_id");

        FamilyCol_ = TFamilyColumns();
        FamilyCol_->FamilyId = col++;
        FamilyCol_->AdminUid = col++;
        FamilyCol_->Meta = col++;

        if (getPlace) {
            NUtils::Append(select, ",fm.place");
            FamilyCol_->PlaceCol = col++;
        }

        return col;
    }

    unsigned TDbFetcher::AddQueryWebauthnCredentials(unsigned col, TString& select, TString& from) {
        if (DefaultProfile_->ExtendedWebauthnAttrs().empty()) {
            return col;
        }

        NUtils::Append(select, ",wc.credential_id");
        NUtils::Append(from, " LEFT JOIN webauthn_credentials wc ON wc.uid=al.uid");

        WebauthnCredentialIdCol_ = col++;
        return col;
    }

    void TDbFetcher::ReadSuid(const NDbPool::TRow& row, TDbProfile& profile) const {
        if (!Suid2Col_) {
            return;
        }

        const NDbPool::TValue& suid = row[Suid2Col_];
        if (suid.IsNull()) {
            return;
        }

        profile.Suid2_.Value = row[Suid2Col_].AsString();
        profile.Suid2_.Exists = true;
    }

    void TDbFetcher::ReadFamilyInfo(const NDbPool::TRow& row, TDbProfile& profile) const {
        if (!FamilyCol_) {
            return;
        }

        const NDbPool::TValue& familyId = row[FamilyCol_->FamilyId];
        if (familyId.IsNull()) {
            return;
        }

        TFamilyInfo res;

        res.FamilyId = familyId.AsString();
        res.AdminUid = row[FamilyCol_->AdminUid].AsString();
        Y_UNUSED(FamilyCol_->Meta);
        res.Place = row[FamilyCol_->PlaceCol].AsString();

        profile.FamilyInfo_ = std::move(res);
    }

    void TDbFetcher::ReadWebauthnCredentialIds(const NDbPool::TRow& row, TDbProfile& profile) const {
        if (!WebauthnCredentialIdCol_) {
            return;
        }

        const NDbPool::TValue& credentialId = row[WebauthnCredentialIdCol_];
        if (credentialId.IsNull()) {
            return;
        }

        profile.WebauthnCredentialIds_.insert(credentialId.AsString());
    }

    void TDbFetcher::DoAttrsPostprocessing() {
        for (TDbProfile& profile : Profiles_) {
            if (profile.Empty_ || profile.Uid_.empty()) {
                continue; // skip empty profiles
            }

            profile.ComputeCorrectDefaultPhoneId();

            FillSyntheticAttrs(profile);

            FillSyntheticAttrsForEntity(profile.ExtendedPhones_, SyntExtendedPhones_, profile);
            FillSyntheticAttrsForEntity(profile.ExtendedEmails_, SyntExtendedEmails_, profile);
            FillSyntheticAttrsForEntity(profile.ExtendedWebauthn_, SyntExtendedWebauthn_, profile);

            profile.FilterWebauthnCredentials();
        }
    }

    void TDbFetcher::FillSyntheticAttrs(TDbProfile& profile) const {
        for (const auto& [type, func] : SyntAttrs_) {
            TDbProfile::TAttrs::iterator dbvalue = profile.Attrs_.find(type);
            if (dbvalue == profile.Attrs_.end()) {
                continue;
            }
            TString value = func(&profile);
            dbvalue->second.Exists = !value.empty();
            dbvalue->second.Value = std::move(value);
        }
    }

    void TDbFetcher::FillSyntheticAttrsForEntity(TDbProfile::TExtendedEntities& profileEntities,
                                                 const TSyntheticAttributes::TSyntheticExtEntity& synEntities,
                                                 TDbProfile& profile) {
        for (auto& [id, attrs] : profileEntities) {
            if (id.empty()) {
                continue;
            }

            for (const auto& [type, func] : synEntities) {
                TDbProfile::TAttrs::iterator itAttr = attrs.find(type);
                if (itAttr == attrs.end()) {
                    continue;
                }

                TDbValue& dbValue = itAttr->second;
                TString value = func(&profile, id);
                dbValue.Exists = !value.empty();
                dbValue.Value = std::move(value);
            }
        }
    }
}
