#pragma once

#include "db_profile.h"
#include "synthetic_attrs.h"

#include <passport/infra/daemons/blackbox/src/domain/idomain_fetcher.h>
#include <passport/infra/daemons/blackbox/src/output/phone_bindings_chunk.h>

#include <util/generic/string.h>

#include <list>
#include <map>
#include <set>
#include <vector>

namespace NPassport::NDbPool {
    class TDbPool;
    class TBlockingHandle;
    class TResult;
    struct TRow;
}

namespace NPassport::NBb {
    class TAnonymiser;
    class TDbValue;
    class TDomainFetcher;
    class TRangedShardsMap;
    class TTotpEncryptor;
    class TYandexDomains;

    class TDbFetcher: TNonCopyable {
    public:
        enum class EPhoneAliasMode {
            ByAttribute, // default mode
            ForceOn,
            ForceOff
        };

        using TProfiles = std::list<TDbProfile>;

        TDbFetcher(NDbPool::TDbPool& dbcentral,
                   const TRangedShardsMap& dbshard,
                   const TYandexDomains* yaDomains,
                   const IDomainFetcher& pddDomains,
                   const TTotpEncryptor* totpEncryptor,
                   const TSyntheticAttributes& synthAttrs,
                   const TAnonymiser* anonymizer = nullptr);

        TDbIndex AddAttr(const TString& attr);
        TDbIndex AddAlias(const TString& alias);
        TDbIndex AddSuid2();
        void AddExtendedPhoneAttr(const TString& attr);
        void AddExtendedEmailAttr(const TString& attr);
        void AddExtendedWebauthnAttr(const TString& attr);
        void AddPhoneOperations();
        void AddPhoneBindings(EPhoneBindingsType type);
        void SetPhoneAliasMode(EPhoneAliasMode mode);
        void AddFamilyInfo();

        // utilities for iteration over profiles
        void ResetProfiles();
        void ResetCurrentProfile();
        const TDbProfile* NextProfile();
        const TDbProfile* ProfileByUid(const TString& uid) const;

        void SetAttrsBoundary(unsigned boundary);

        void FetchByUid(const TString& uid);
        void FetchByUids(const std::vector<TString>& uids);
        void FetchBySuid(const TString& suid, const TString& sid);

        void FetchByLogin(const TString& login, const TString& domid, const std::vector<TString>& sidArray, const TString& defUid, bool allowScholar = false);

        void FetchByAlias(const TString& alias, const std::vector<TString>& types, const TString& defUid);

        void FetchAliasesByFamilyId(const TString& familyId, bool getPlace);
        void FetchAttrs();

    private:
        using TProfileSidPair = std::pair<TProfiles::iterator, TString>;

        TDbProfile* ProfileByUidImpl(const TString& uid);

    protected:
        static TString GetAttrsCondition(const TDbProfile::TAttrs& attrs, unsigned boundary = 30);

    private:
        TString GetAliasTypes(const std::vector<TString>& sids, bool pdd, bool allowScholar = false) const;
        std::vector<TProfiles::iterator> FindProfiles(const TString& login,
                                                      const std::vector<TString>& aliases);
        std::vector<TProfileSidPair> SelectProfiles(const TString& login,
                                                    const std::vector<TString>& sids,
                                                    bool pdd);

        TProfiles::iterator SelectProfile(const TString& alias,
                                          const std::vector<TString>& types);

        void DoCentralDbQuery(const TString& alias, const TString& types, const TString& defUid);

        // these methods fill up existing profiles_ with data
        void FetchRawAttrs();
        void FetchEmails();

        void ReadCentralQuery(NDbPool::TResult& result, bool addProfiles);
        static std::unique_ptr<NDbPool::TResult> DoCentralDbQuery(NDbPool::TBlockingHandle& db, const TString& query);
        void ReadShardQuery(NDbPool::TResult& result);
        void DoShardedDbQuery(const std::vector<TString>& uidsList,
                              bool needPhoneAttrs,
                              bool needEmailAttrs,
                              bool needWebauthnAttrs);

        void AddMoreTablesToQuery(unsigned col, TString& select, TString& from, bool getPlace = false);
        unsigned AddQuerySuids(unsigned col, TString& select, TString& from);
        unsigned AddQueryFamilyInfo(unsigned col, TString& select, TString& from, bool getPlace);
        unsigned AddQueryWebauthnCredentials(unsigned col, TString& select, TString& from);

        void ReadSuid(const NDbPool::TRow& row, TDbProfile& profile) const;
        void ReadFamilyInfo(const NDbPool::TRow& row, TDbProfile& profile) const;
        void ReadWebauthnCredentialIds(const NDbPool::TRow& row, TDbProfile& profile) const;

        void DoAttrsPostprocessing();
        void FillSyntheticAttrs(TDbProfile& profile) const;
        static void FillSyntheticAttrsForEntity(TDbProfile::TExtendedEntities& profileEntities,
                                                const TSyntheticAttributes::TSyntheticExtEntity& synEntities,
                                                TDbProfile& profile);

    protected:
        EPhoneAliasMode PhoneAliasMode_ = EPhoneAliasMode::ByAttribute;
        NDbPool::TDbPool& DbCentral_;
        const TRangedShardsMap& DbSharded_;

        friend TSyntheticAttributes;
        const TAnonymiser* Anonymizer_;
        const TYandexDomains* YandexDomains_;
        const IDomainFetcher& HostedDomains_;
        const TTotpEncryptor* TotpEncryptor_;
        const TSyntheticAttributes& SyntheticAttributes_;

        // all profiles read from db
        TProfiles Profiles_;
        // just for convenience, to fill only one profile with attrs/aliases etc
        TDbProfile* DefaultProfile_;
        TProfiles::iterator Current_;

        unsigned AttrsQueryBoundary_ = 30;
        unsigned Suid2Col_ = 0;
        unsigned WebauthnCredentialIdCol_ = 0;
        struct TFamilyColumns {
            unsigned FamilyId = 0;
            unsigned AdminUid = 0;
            unsigned Meta = 0;
            unsigned PlaceCol = 0;
        };
        std::optional<TFamilyColumns> FamilyCol_;

        // map: synthetic attr type -> synthesis function
        std::map<TString, TSyntheticAttributes::TAttrFuncType> SyntAttrs_;
        TSyntheticAttributes::TSyntheticExtEntity SyntExtendedPhones_;
        TSyntheticAttributes::TSyntheticExtEntity SyntExtendedEmails_;
        TSyntheticAttributes::TSyntheticExtEntity SyntExtendedWebauthn_;
    };
}
