#include "db_profile.h"

#include "db_types.h"
#include "strings.h"
#include "utils.h"

#include <passport/infra/daemons/blackbox/src/output/phone_bindings_chunk.h>

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <algorithm>
#include <functional>

namespace NPassport::NBb {
    const TString NULL_STR = "<NULL>";

    TDbValue::TDbValue(const TString& val)
        : Value(val)
        , Exists(true)
    {
    }

    bool TDbValue::AsBoolean() const {
        return AsBoolean(Value);
    }

    bool TDbValue::AsBoolean(TStringBuf value) {
        return !value.empty() && value[0] != '0';
    }

    time_t TDbValue::AsTime() const {
        time_t t = 0;
        TryIntFromString<10>(TStringBuf(Value), t);
        return t;
    }

    const TString& TDbValue::AsString() const {
        return Exists ? Value : NULL_STR;
    }

    TDbProfile::TDbProfile() {
        Index_.reserve(50);
    }

    TDbProfile::TDbProfile(const TDbProfile& d)
        : Empty_(d.Empty_)
        , Uid_(d.Uid_)
        , Sid_(d.Sid_)
        , DomCache_(d.DomCache_)
        , DomItem_(d.DomItem_)
        , AltDomainId_(d.AltDomainId_)
        , AltDomain_(d.AltDomain_)
        , CatchAll_(d.CatchAll_)
        , Aliases_(d.Aliases_)
        , NeedPddAliases_(d.NeedPddAliases_)
        , AliasesPdd_(d.AliasesPdd_)
        , AliasesOldPublic_(d.AliasesOldPublic_)
        , Attrs_(d.Attrs_)
        , Suid2_(d.Suid2_)
        , DefaultPhoneId_(d.DefaultPhoneId_)
        , ExtendedPhones_(d.ExtendedPhones_)
        , ExtendedEmails_(d.ExtendedEmails_)
        , ExtendedWebauthn_(d.ExtendedWebauthn_)
        , NeedPhoneOperations_(d.NeedPhoneOperations_)
        , PhoneOperations_(d.PhoneOperations_)
        , PhoneBindingsType_(d.PhoneBindingsType_)
        , PhoneBindings_(d.PhoneBindings_)
        , NeedFamilyInfo_(d.NeedFamilyInfo_)
    {
        Index_.resize(d.Index_.size());
        UpdateIndex();
    }

    const TDomainList& TDbProfile::DomainList() const {
        if (!DomCache_) {
            // Should be added in DbFetcher
            throw yexception() << "DbProfile: missing domain cache for pdd user";
        }
        return DomCache_->List;
    }

    bool TDbProfile::HasAlias(const TString& alias) const {
        if (alias == TAlias::PDD_ALIAS_LOGIN) {
            return !AliasesPdd_.empty();
        }
        if (alias == TAlias::OLDPUBLICID) {
            return !AliasesOldPublic_.empty();
        }

        TAliases::const_iterator p = Aliases_.find(alias);

        return p != Aliases_.end() && !p->second.Value.empty();
    }

    const TString& TDbProfile::GetAlias(const TString& alias) const {
        // we can't return multi-alias here, only unique ones
        if (alias == TAlias::PDD_ALIAS_LOGIN || alias == TAlias::OLDPUBLICID) {
            return TStrings::EMPTY;
        }

        TAliases::const_iterator p = Aliases_.find(alias);

        return (p != Aliases_.end()) ? p->second.Value : TStrings::EMPTY;
    }

    TDbIndex TDbProfile::AddAlias(const TString& alias) {
        if (alias == TAlias::PDD_ALIAS_LOGIN) {
            NeedPddAliases_ = true;
            return -1;
        }
        if (alias == TAlias::OLDPUBLICID) {
            return -1;
        }
        return AddValue(alias, Aliases_)->Index;
    }

    TDbIndex TDbProfile::AddAttr(const TString& attr) {
        return AddValue(attr, Attrs_)->Index;
    }

    TDbIndex TDbProfile::AddSuid2() {
        if (-1 == Suid2_.Index) {
            Suid2_.Index = Index_.size();
            Index_.push_back(&Suid2_);
        }
        return Suid2_.Index;
    }

    void TDbProfile::AddExtendedPhoneAttr(const TString& attr) {
        ExtendedPhones_[TStrings::EMPTY][attr];
    }

    void TDbProfile::AddExtendedEmailAttr(const TString& attr) {
        ExtendedEmails_[TStrings::EMPTY][attr];
    }

    void TDbProfile::AddExtendedWebauthnAttr(const TString& attr) {
        ExtendedWebauthn_[TStrings::EMPTY][attr];
    }

    void TDbProfile::AddPhoneBindings(EPhoneBindingsType type) {
        PhoneBindingsType_ = type;
    }

    void TDbProfile::AddFamilyInfo() {
        NeedFamilyInfo_ = true;
    }

    const TDbValue* TDbProfile::Get(TDbIndex i) const {
        if (i < 0 || (size_t)i >= Index_.size()) {
            throw yexception() << "Incorrect hidden index";
        }
        return Index_[i];
    }

    //  assign default value for attribute, if this attribute was registered
    void TDbProfile::SetAttrValue(const TString& attr, const TString& value) {
        TAttrs::iterator p = Attrs_.find(attr);
        if (p != Attrs_.end()) {
            TDbValue& val = p->second;
            val.Exists = true;
            val.Value.assign(value);
        }
    }

    static const TString DEFAULT_LANG = "ru";
    static const TString DEFAULT_COUNTRY = "ru";
    static const TString DEFAULT_TIMEZONE = "Europe/Moscow";

    void TDbProfile::FillDefaultAttrValues() {
        SetAttrValue(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME, TStrings::ONE);
        SetAttrValue(TAttr::REVOKER_TOKENS, TStrings::ONE);
        SetAttrValue(TAttr::REVOKER_APP_PASSWORDS, TStrings::ONE);
        SetAttrValue(TAttr::REVOKER_WEB_SESSIONS, TStrings::ONE);
        SetAttrValue(TAttr::KARMA_VALUE, TStrings::ZERO);
        SetAttrValue(TAttr::KARMA_ACTIVATION_DATETIME, TStrings::ZERO);
        SetAttrValue(TAttr::PASSWORD_UPDATE_DATETIME, TStrings::ZERO);
        SetAttrValue(TAttr::PERSON_COUNTRY, DEFAULT_COUNTRY);
        SetAttrValue(TAttr::PERSON_TIMEZONE, DEFAULT_TIMEZONE);
        SetAttrValue(TAttr::PERSON_LANGUAGE, DEFAULT_LANG);
        SetAttrValue(TAttr::ACCOUNT_PUBLIC_ID_VERSION, TStrings::ZERO);
    }

    void TDbProfile::FilterWebauthnCredentials() {
        for (auto it = ExtendedWebauthn_.begin(); it != ExtendedWebauthn_.end();) {
            if (it->first.empty()) {
                ++it;
                continue;
            }
            TDbProfile::TAttrs::const_iterator credId = it->second.find(TWebauthnAttr::EXTERNAL_ID);
            if (credId == it->second.end() ||
                credId->second.Value.empty() ||
                !WebauthnCredentialIds_.contains(credId->second.Value))
            {
                TLog::Warning("Got broken webauthn credential from db: uid=%s, entity_id=%s", Uid_.c_str(), it->first.c_str());
                it = ExtendedWebauthn_.erase(it); // erase entities with no id or unknown id
            } else {
                ++it;
            }
        }
    }

    void TDbProfile::ComputeCorrectDefaultPhoneId() {
        // If default phone id is set explicitly in attribute - check if it is bound and then use it
        TDbProfile::TAttrs::const_iterator itDefaultId = Attrs_.find(TAttr::PHONES_DEFAULT);
        if (itDefaultId != Attrs_.end()) {
            const TString& defaultId = itDefaultId->second.Value;
            if (!defaultId.empty()) {
                TDbProfile::TExtendedEntities::const_iterator itDefault = ExtendedPhones_.find(defaultId);
                if (itDefault != ExtendedPhones_.end()) {
                    const TDbProfile::TAttrs& phoneAttrs = itDefault->second;
                    TDbProfile::TAttrs::const_iterator itBoundAttr = phoneAttrs.find(TPhoneAttr::BOUND);

                    if (itBoundAttr != phoneAttrs.end() && !itBoundAttr->second.Value.empty()) {
                        DefaultPhoneId_ = defaultId;
                        return; // explicit default phone is found and it is bound
                    }
                }
            }
        }

        // Max entity id
        ui64 defaultIdInt = 0;
        for (const auto& [id, phoneAttrs] : ExtendedPhones_) {
            if (id.empty()) {
                continue; // skip dummy entry
            }
            TDbProfile::TAttrs::const_iterator itBoundAttr = phoneAttrs.find(TPhoneAttr::BOUND);
            if (itBoundAttr == phoneAttrs.end() || itBoundAttr->second.Value.empty()) {
                continue; // skip unbound entiries
            }
            ui64 curId = IntFromString<ui64, 10>(id);
            if (curId > defaultIdInt) {
                defaultIdInt = curId;
            }
        }

        if (defaultIdInt) {
            DefaultPhoneId_ = IntToString<10>(defaultIdInt);
        }
    }

    TDbValue* TDbProfile::AddValue(const TString& val, std::map<TString, TDbValue>& cont) {
        TDbValue& dbvalue = cont[val];
        if (-1 == dbvalue.Index) {
            dbvalue.Index = Index_.size();
            Index_.push_back(&dbvalue);
        }
        return &dbvalue;
    }

    template <typename Type>
    void TDbProfile::UpdateIndex(Type& cont) {
        for (auto& typeValuePair : cont) {
            if (-1 != typeValuePair.second.Index) {
                Index_[typeValuePair.second.Index] = &(typeValuePair.second);
            }
        }
    }

    // when doing a copy of DbProfile we first copy all data maps (attrs, aliases, suids)
    // and then need to rebuild index, storing references to this exact class values
    // here we assume index_ is resized correctly and then collect references according to their indices
    // in the original class
    void TDbProfile::UpdateIndex() {
        UpdateIndex(Attrs_);
        UpdateIndex(Aliases_);

        if (-1 != Suid2_.Index) {
            Index_[Suid2_.Index] = &Suid2_;
        }
    }

    namespace {
        void PrintElem(TString* out, const std::pair<TString, TDbValue>& e) {
            out->push_back('\t');
            *out += "'" + e.first + "' : '";
            *out += e.second.AsString() + "'\n";
        }

        void PrintIndexedElem(TString* out, const std::pair<TString, TDbValue>& e) {
            out->push_back('\t');
            *out += "'" + e.first + "' : '";
            *out += e.second.AsString() + "' (index=" + IntToString<10>(e.second.Index) + ")\n";
        }

        void PrintSameElem(TString* out, const TString& type, const TString& e) {
            out->push_back('\t');
            *out += "'" + type + "' : '";
            if (e.empty()) {
                *out += "<NULL>";
            } else {
                *out += e;
            }
            *out += "'\n";
        }

        void PrintPair(TString* out, const std::pair<TString, TString>& e) {
            out->push_back('\t');
            *out += "'" + e.first + "' : '" + e.second + "'\n";
        }

        void PrintExtended(TString* out, const TString& type, const TDbProfile::TExtendedEntities& e) {
            *out += "type=" + type + ":\n";

            using namespace std::placeholders;
            for (const auto& [id, attrs] : e) {
                *out += "id=" + id + " {\n";
                std::for_each(attrs.begin(), attrs.end(), [out](const auto& attr) { return PrintElem(out, attr); });
                *out += "}\n";
            }
        }
    }

    TString TDbProfile::ToString() const {
        TString res;
        using namespace std::placeholders;

        res += "DbProfile for uid '" + Uid_ + "':\n";
        res += "Sid = '" + Sid_ + "', pdd domain_id='" + PddDomId() + "', alternative domain id='" + AltDomainId_ + "':\n";
        res += "Aliases:\n";
        std::for_each(Aliases_.begin(), Aliases_.end(), [&res](const auto& alias) { return PrintIndexedElem(&res, alias); });
        std::for_each(AliasesPdd_.begin(), AliasesPdd_.end(), [&res](const auto& alias) { return PrintSameElem(&res, TAlias::PDD_ALIAS_LOGIN, alias); });
        std::for_each(AliasesOldPublic_.begin(), AliasesOldPublic_.end(), [&res](const auto& alias) { return PrintSameElem(&res, TAlias::OLDPUBLICID, alias); });
        res += "Attrs:\n";
        std::for_each(Attrs_.begin(), Attrs_.end(), [&res](const auto& attr) { return PrintIndexedElem(&res, attr); });
        res += "Suid2:" + Suid2_.AsString() + "\n";
        res += "Extended phone attrs:\n";
        PrintExtended(&res, TExtendedAttrType::PHONE, ExtendedPhones_);
        res += "Extended email attrs:\n";
        PrintExtended(&res, TExtendedAttrType::EMAIL, ExtendedEmails_);
        res += "Extended webauthn attrs:\n";
        PrintExtended(&res, TExtendedAttrType::WEBAUTHN, ExtendedWebauthn_);
        res += "Phone operations:\n";
        std::for_each(PhoneOperations_.begin(), PhoneOperations_.end(), [&res](const auto& op) { return PrintPair(&res, op); });
        res += "Index:\n";
        for (unsigned i = 0; i < Index_.size(); ++i) {
            res += "\t " + IntToString<10>(i) + " : " + (Index_[i] ? "'" + Index_[i]->AsString() + "'" : "-") + '\n';
        }
        return res;
    }

    bool TDbProfile::IsKid() const {
        return HasAlias(TAlias::KIDDISH) && !HasAlias(TAlias::PORTAL_LOGIN);
    }

    bool TDbProfile::Pdd() const {
        if (HasAlias(TAlias::PORTAL_LOGIN) ||
            HasAlias(TAlias::LITE_LOGIN) ||
            HasAlias(TAlias::SOCIAL_LOGIN) ||
            HasAlias(TAlias::ALT_DOMAIN_LOGIN))
        {
            return false; // these are portal accounts even if pdd alias set
        }

        return HasAlias(TAlias::PDD_MASTER_LOGIN);
    }

    TDbProfile::EMatchStatus TDbProfile::MatchesLogin(const TString& login, const TString& alias) const {
        if (alias == TAlias::PDD_ALIAS_LOGIN) {
            if (AliasesPdd_.empty()) {
                return NoAlias;
            }
            if (AliasesPdd_.find(login) == AliasesPdd_.end()) {
                return NoMatch;
            }
            return Match;
        }

        if (alias == TAlias::OLDPUBLICID) {
            if (AliasesOldPublic_.empty()) {
                return NoAlias;
            }
            if (AliasesOldPublic_.find(login) == AliasesOldPublic_.end()) {
                return NoMatch;
            }
            return Match;
        }
        // get all values of given alias
        TAliases::const_iterator it = Aliases_.find(alias);
        if (it == Aliases_.end()) {
            return NoAlias;
        }

        if (it->second.Exists) {
            if (it->second.Value == login) {
                return Match;
            }
            return NoMatch;
        }

        return NoAlias;
    }

}
