#pragma once

#include <passport/infra/daemons/blackbox/src/domain/domain_list.h>
#include <passport/infra/daemons/blackbox/src/output/phone_bindings_chunk.h>

#include <util/generic/string.h>

#include <map>
#include <set>
#include <vector>

namespace NPassport::NBb {
    using TDbIndex = int;

    class TDbValue {
    public:
        TDbValue() = default;
        explicit TDbValue(const TString& val);

        // treat attribute as boolean flag
        bool AsBoolean() const;
        static bool AsBoolean(TStringBuf value);
        // treat as time, doesn't check format!
        time_t AsTime() const;

        const TString& AsString() const;

        TString Value;
        TDbIndex Index = -1;
        bool Exists = false;
    };

    inline bool operator==(const TDbValue& l, const TDbValue& r) {
        return l.Exists == r.Exists && l.Value == r.Value;
    }

    struct TFamilyInfo {
        TString FamilyId;
        TString AdminUid;
        TString Place;
    };

    class TDbProfile {
    public:
        // map: alias type -> alias value
        using TAliases = std::map<TString, TDbValue>;
        // set: alias value
        using TMultiAlias = std::set<TString>;
        // map: attr type -> attr value
        using TAttrs = std::map<TString, TDbValue>;

        // map: extended entity id -> entity attributes
        using TExtendedEntities = std::map<TString, TAttrs>;

        // map: phone operation id -> operation details
        using TPhoneOperations = std::map<TString, TString>;

        TDbProfile();
        TDbProfile(const TDbProfile& d);

        TDbProfile& operator=(const TDbProfile& d) = delete;

        const TString& Uid() const {
            return Uid_;
        }
        const TString& Sid() const {
            return Sid_;
        }

        bool Pdd() const;
        bool CatchAll() const {
            return CatchAll_;
        }
        const TString& PddDomId() const {
            return DomItem_.Id();
        }
        const TString& PddDomain() const {
            return DomItem_.AsciiName();
        }
        const TString& PddDomainUtf() const {
            return DomItem_.UtfName();
        }
        const TDomain& PddDomItem() const {
            return DomItem_;
        }
        const TDomainList& DomainList() const;

        const TString& AltDomId() const {
            return AltDomainId_;
        }
        const TString& AltDomain() const {
            return AltDomain_;
        }

        const TAliases& Aliases() const {
            return Aliases_;
        }
        bool NeedAliases() const {
            return !Aliases_.empty() || NeedPddAliases_;
        }
        const TMultiAlias& AliasesPdd() const {
            return AliasesPdd_;
        }
        const TMultiAlias& AliasesOldPublic() const {
            return AliasesOldPublic_;
        }
        bool HasAlias(const TString& alias) const;
        const TString& GetAlias(const TString& alias) const;

        const TAttrs& Attrs() const {
            return Attrs_;
        }
        const TDbValue& Suid2() const {
            return Suid2_;
        }
        const TExtendedEntities& ExtendedPhoneAttrs() const {
            return ExtendedPhones_;
        }
        const TExtendedEntities& ExtendedEmailAttrs() const {
            return ExtendedEmails_;
        }
        const TExtendedEntities& ExtendedWebauthnAttrs() const {
            return ExtendedWebauthn_;
        }
        const TPhoneOperations& PhoneOperations() const {
            return PhoneOperations_;
        }
        const TPhoneBindingsChunk::TPhoneBindings& PhoneBindings() const {
            return PhoneBindings_;
        }
        const std::optional<TFamilyInfo>& FamilyInfo() const {
            return FamilyInfo_;
        }

        const TDbValue* Get(TDbIndex i) const;

        // checks if the profile matches the login by alias
        enum EMatchStatus {
            Match,
            NoMatch,
            NoAlias,
        };
        EMatchStatus MatchesLogin(const TString& login, const TString& alias) const;

        // print object for debug purposes
        TString ToString() const;

        const TString& DefaultPhoneId() const {
            return DefaultPhoneId_;
        }

        bool IsKid() const;

    protected:
        friend class TDbFetcher;
        friend class TSyntheticAttributes;

        TDbIndex AddAlias(const TString& alias);
        TDbIndex AddAttr(const TString& attr);
        TDbIndex AddSuid2();
        void AddExtendedPhoneAttr(const TString& attr);
        void AddExtendedEmailAttr(const TString& attr);
        void AddExtendedWebauthnAttr(const TString& attr);
        void AddPhoneOperations() {
            NeedPhoneOperations_ = true;
        }
        void AddPhoneBindings(EPhoneBindingsType type);
        void AddFamilyInfo();

        void SetAttrValue(const TString& attr, const TString& value);
        void FillDefaultAttrValues();
        void FilterWebauthnCredentials();
        void ComputeCorrectDefaultPhoneId();

    protected:
        TDbValue* AddValue(const TString& val, std::map<TString, TDbValue>& cont);

        template <typename Cont>
        void UpdateIndex(Cont& cont);

        void UpdateIndex();

        bool Empty_ = true;
        TString Uid_;
        TString Sid_;

        TDomainCachePtr DomCache_;
        TDomain DomItem_;
        TString AltDomainId_;
        TString AltDomain_;
        bool CatchAll_ = false;

        TAliases Aliases_;
        bool NeedPddAliases_ = false;
        TMultiAlias AliasesPdd_;
        TMultiAlias AliasesOldPublic_;

        TAttrs Attrs_;
        TDbValue Suid2_;
        std::set<TString> WebauthnCredentialIds_;

        TString DefaultPhoneId_;

        // DbValues are referenced by indices
        std::vector<TDbValue*> Index_;

        // Extended Attributes entities
        TExtendedEntities ExtendedPhones_;
        TExtendedEntities ExtendedEmails_;
        TExtendedEntities ExtendedWebauthn_;

        // phone operations
        bool NeedPhoneOperations_ = false;
        TPhoneOperations PhoneOperations_;

        // phone bindings
        EPhoneBindingsType PhoneBindingsType_ = EPhoneBindingsType::None;
        TPhoneBindingsChunk::TPhoneBindings PhoneBindings_;

        // family info
        bool NeedFamilyInfo_ = false;
        std::optional<TFamilyInfo> FamilyInfo_;
    };
}
