#include "dbfields_converter.h"

#include "db_fetcher.h"
#include "db_types.h"
#include "exception.h"
#include "experiment.h"
#include "hosts_list.h"
#include "strings.h"
#include "utils.h"

#include <passport/infra/libs/cpp/utils/string/format.h>

#include <algorithm>
#include <ctime>

namespace NPassport::NBb {
    const TDbFieldsConverter::TUserinfoToAttrMap& TDbFieldsConverter::UserInfoFields() {
        static const TDbFieldsConverter::TUserinfoToAttrMap USERINFO_FIELDS = {
            {"firstname", TAttr::PERSON_FIRSTNAME},
            {"lastname", TAttr::PERSON_LASTNAME},
            {"display_name", TAttr::ACCOUNT_DISPLAY_NAME},
            {"birth_date", TAttr::PERSON_BIRTHDATE},
            {"city", TAttr::PERSON_CITY},
            {"lang", TAttr::PERSON_LANGUAGE},
            {"country", TAttr::PERSON_COUNTRY},
            {"fio", TAttr::PERSON_FIO},
            {"tz", TAttr::PERSON_TIMEZONE}};
        return USERINFO_FIELDS;
    }

    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    TDbFieldsConverter::TSidsToAttrMap TDbFieldsConverter::SUBSCRIPTIONS;

    const TString TDbFieldsConverter::REGNAME = "subscription.login.8";
    const TString TDbFieldsConverter::NAROD_MAIL_LOGIN = "subscription.login.16";
    const TString TDbFieldsConverter::REGDATE = "userinfo.reg_date.uid";
    const TString TDbFieldsConverter::LOGIN2 = "subscription.login.2";
    const TString TDbFieldsConverter::LOGIN8 = "subscription.login.8";
    const TString TDbFieldsConverter::LOGIN16 = "subscription.login.16";

    const TString TBL_ACCOUNTS = "accounts";
    const TString TBL_ACCOUNT_INFO = "account_info";
    const TString TBL_USERINFO = "userinfo";
    const TString TBL_SUBSCRIPTION = "subscription";
    const TString TBL_HOSTS = "hosts";
    const TString TBL_USERINFO_SAFE = "userinfo_safe";
    const TString TBL_PASSWORD_QUALITY = "password_quality";
    const TString TBL_USERPHONES = "userphones";
    const TString TBL_ADTARGET = "adtarget";

    const TString FLD_LOGIN = "login";
    const TString FLD_ENA = "ena";
    const TString FLD_GLOGOUT = "glogout";

    const TString FLD_SEX = "sex";
    const TString FLD_REGDATE = "reg_date";

    const TString FLD_SUID = "suid";
    const TString FLD_LOGIN_RULE = "login_rule";
    const TString FLD_HOST_ID = "host_id";
    const TString FLD_BORN_DATE = "born_date";

    const TString FLD_NICKNAME = "nickname";

    const TString FLD_HINTQ = "hintq";
    const TString FLD_HINTA = "hinta";

    const TString FLD_VERSION = "version";
    const TString FLD_QUALITY = "quality";

    const TString FLD_DB_ID = "db_id";
    const TString FLD_PRIO = "prio";

    const TString FLD_NUMBER = "number";
    const TString FLD_CONFIRMED = "confirmed";

    const TString FLD_UID = "uid";

    using namespace std::placeholders;

    // empty-string const field
    TString
    TDbFieldsConverter::EmptyField(const TDbProfile*) {
        return TStrings::EMPTY;
    }

    // predefined const field
    TString
    TDbFieldsConverter::ConstantField(const TDbProfile*, const TString& value) {
        return value;
    }

    // field value if exists, empty string otherwise
    TString
    TDbFieldsConverter::SimpleField(const TDbProfile* profile, TDbIndex field) {
        return profile->Get(field)->Value;
    }

    // given flag value if field exists, empty otherwise
    TString
    TDbFieldsConverter::CheckValueField(const TDbProfile* profile, TDbIndex field, const TString& val) {
        return profile->Get(field)->Exists ? val : TStrings::EMPTY;
    }

    // given flag value if field exists, empty otherwise
    TString
    TDbFieldsConverter::ConditionalUid(const TDbProfile* profile, TDbIndex condition) {
        return profile->Get(condition)->Exists ? profile->Uid() : TStrings::EMPTY;
    }

    // split field in 2 parts by sep and return first or last part
    TString
    TDbFieldsConverter::SplitPairField(const TDbProfile* profile, TDbIndex field, char sep, bool first) {
        const TDbValue* val = profile->Get(field);

        TString::size_type pos = val->Value.find(sep);
        if (pos == TString::npos) { // if empty or not found
            return TStrings::EMPTY;
        }

        return first ? val->Value.substr(0, pos) : val->Value.substr(pos + 1);
    }

    // given flag value if galatasaray alias exists, empty otherwise
    TString
    TDbFieldsConverter::GalatasarayField(const TDbProfile* profile, TDbIndex alias, const TString& val) {
        return profile->Get(alias)->Exists && profile->AltDomId() == TStrings::ONE ? val : TStrings::EMPTY;
    }

    TString TDbFieldsConverter::CheckAliasPddField(const TDbProfile* profile, const TString& val) {
        return profile->AliasesPdd().empty()
                   ? TStrings::EMPTY
                   : val;
    }

    // check if account has active mail subscription, i.e. has mail suid + mail host OR active mail status
    bool TDbFieldsConverter::HasMailSubscription(const TDbProfile* profile,
                                                 TDbIndex mailSuid,
                                                 TDbIndex mailHost,
                                                 TDbIndex mailStatus,
                                                 const bool ignoreFrozen) {
        // mail suid is required for mail subscription
        if (!profile->Get(mailSuid)->Exists) {
            return false;
        }

        const TString& status = profile->Get(mailStatus)->Value;

        // if mail is frozen, show subscription only when frozen is ignored
        if (status == TMailStatus::FROZEN) {
            return ignoreFrozen;
        }

        // not frozen, check old attribute that turns on mail subscription
        if (profile->Get(mailHost)->Exists) {
            return true;
        }

        // no old mail attribute, has subscription only if status is active
        return status == TMailStatus::ACTIVE;
    }

    // find given phone attributes, if any, or return nullptr
    const TDbProfile::TAttrs*
    TDbFieldsConverter::GetPhoneAttrs(const TDbProfile* profile, const TString& phoneid) {
        const TDbProfile::TExtendedEntities& entities = profile->ExtendedPhoneAttrs();
        TDbProfile::TExtendedEntities::const_iterator p = entities.find(phoneid);

        return p == entities.end() ? nullptr : &(p->second);
    }

    // const Attrs* getPhoneAttrs(const TString& phone_id)
    // return val if phone exists (new by phone_id or legacy old one)
    TString
    TDbFieldsConverter::CheckPhoneField(const TDbProfile* profile, TDbIndex phoneId, const TString& val) {
        if (profile->Get(phoneId)->Exists) {
            // this account was transferred to extended attributes, search there
            return GetPhoneAttrs(profile, profile->Get(phoneId)->Value) ? val : TStrings::EMPTY;
        }
        return TStrings::EMPTY;
    }

    // last 4 digits of phone number
    TString
    TDbFieldsConverter::PhoneTailField(const TDbProfile* profile, TDbIndex phoneId) {
        const TDbValue* val = nullptr;
        if (profile->Get(phoneId)->Exists) {
            // this account was transferred to extended attributes, search there
            const TDbProfile::TAttrs* attrs = GetPhoneAttrs(profile, profile->Get(phoneId)->Value);
            if (!attrs) {
                return TStrings::EMPTY;
            }
            TDbProfile::TAttrs::const_iterator p = attrs->find(TPhoneAttr::NUMBER);
            if (p != attrs->end()) {
                val = &(p->second);
            }
        }

        if (val && val->Exists) {
            size_t len = val->Value.size();
            if (len > 4) { // cut off 4 last digits
                return val->Value.substr(len - 4);
            }
        }

        return TStrings::EMPTY;
    }

    // return subscription.login.36 either basing on external attrs or from legacy phone attr
    TString
    TDbFieldsConverter::PhoneLoginField(const TDbProfile* profile, TDbIndex phoneNum, TDbIndex phoneId, TDbIndex normLogin) {
        if (profile->Get(phoneId)->Exists) {
            // this account was transferred to extended attributes, search there
            if (!GetPhoneAttrs(profile, profile->Get(phoneId)->Value)) {
                return TStrings::EMPTY;
            }
        } else {
            if (!profile->Get(phoneNum)->Exists) {
                return TStrings::EMPTY;
            }
        }
        // either phone attributes set exists for this user secure phone id, or old attribute exists
        return profile->Get(normLogin)->Value;
    }

    // return phone confirmation date, either from external attrs or from old attribute
    TString
    TDbFieldsConverter::PhoneBornDate(const TDbProfile* profile, TDbIndex phoneId) {
        if (profile->Get(phoneId)->Exists) {
            // this account was transferred to extended attributes, search there
            const TDbProfile::TAttrs* attrs = GetPhoneAttrs(profile, profile->Get(phoneId)->Value);
            if (!attrs) {
                return TStrings::EMPTY;
            }
            TDbProfile::TAttrs::const_iterator p = attrs->find(TPhoneAttr::CONFIRMED);
            if (p != attrs->end()) {
                return NUtils::FormatTimestamp(p->second.Value);
            }
        }

        return TStrings::EMPTY;
    }

    // return phone number, either from extended attributes or from legacy attribute
    TString
    TDbFieldsConverter::PhoneNumberField(const TDbProfile* profile, TDbIndex phoneId) {
        const TDbValue* val = nullptr;
        if (profile->Get(phoneId)->Exists) {
            // this account was transferred to extended attributes, search there
            const TDbProfile::TAttrs* attrs = GetPhoneAttrs(profile, profile->Get(phoneId)->Value);
            if (!attrs) {
                return TStrings::EMPTY;
            }
            TDbProfile::TAttrs::const_iterator p = attrs->find(TPhoneAttr::NUMBER);
            if (p != attrs->end()) {
                val = &(p->second);
            }
        }

        if (val && val->Exists) {
            return val->Value;
        }

        return TStrings::EMPTY;
    }

    // return user defined login if any, else - portal login
    TString
    TDbFieldsConverter::UserDefinedLoginField(const TDbProfile* profile,
                                              TDbIndex user,
                                              TDbIndex normLogin) {
        if (profile->Get(user)->Exists) {
            TString login = profile->Get(user)->Value;

            if (profile->Pdd()) {
                login.push_back('@');
                login.append(profile->PddDomainUtf());
            }

            return login;
        }
        return profile->Get(normLogin)->Value;
    }

    // return mail login value
    TString
    TDbFieldsConverter::MailLoginField(const TDbProfile* profile,
                                       TDbIndex mailSuid,
                                       TDbIndex mailHostId,
                                       TDbIndex mailStatus,
                                       TDbIndex mail,
                                       TDbIndex portal,
                                       TDbIndex pdd,
                                       TDbIndex alt,
                                       const bool ignoreFrozen) {
        if (!HasMailSubscription(profile, mailSuid, mailHostId, mailStatus, ignoreFrozen)) {
            return TStrings::EMPTY;
        }

        const TDbValue* mailLogin = profile->Get(mail);
        if (mailLogin->Exists) {
            return mailLogin->Value;
        }
        const TDbValue* login = profile->Get(portal);
        if (login->Exists) {
            return login->Value;
        }
        const TDbValue* altLogin = profile->Get(alt);
        if (altLogin->Exists) {
            return altLogin->Value;
        }
        const TDbValue* pddLogin = profile->Get(pdd);
        if (pddLogin->Exists) {
            return pddLogin->Value;
        }
        return TStrings::EMPTY;
    }

    // return narodmail login value
    TString
    TDbFieldsConverter::NarodMailLoginField(const TDbProfile* profile,
                                            TDbIndex mailSuid,
                                            TDbIndex mailHostId,
                                            TDbIndex mailStatus,
                                            TDbIndex narod,
                                            TDbIndex mail,
                                            TDbIndex portal,
                                            const bool ignoreFrozen) {
        if (!HasMailSubscription(profile, mailSuid, mailHostId, mailStatus, ignoreFrozen)) {
            return TStrings::EMPTY;
        }

        const TDbValue* narodLogin = profile->Get(narod);
        if (narodLogin->Exists) {
            return narodLogin->Value;
        }
        const TDbValue* mailLogin = profile->Get(mail);
        if (mailLogin->Exists) {
            return mailLogin->Value;
        }
        const TDbValue* login = profile->Get(portal);
        if (login->Exists) {
            return login->Value;
        }
        return TStrings::EMPTY;
    }

    // return portal login field if sid_attr exists
    TString
    TDbFieldsConverter::SidLoginField(const TDbProfile* profile, TDbIndex sidAttr, TDbIndex normLogin) {
        if (profile->Get(sidAttr)->Exists) {
            return profile->Get(normLogin)->Value;
        }
        return TStrings::EMPTY;
    }

    // return galatasaray login field if alias exists
    TString
    TDbFieldsConverter::GalatasarayLoginField(const TDbProfile* profile, TDbIndex alias) {
        if (profile->Get(alias)->Exists && profile->AltDomId() == TStrings::ONE) {
            return SplitPairField(profile, alias, '@', true);
        }
        return TStrings::EMPTY;
    }

    // value of accounts.ena (inverse of disabled)
    TString
    TDbFieldsConverter::EnaField(const TDbProfile* profile, TDbIndex disabled) {
        return profile->Get(disabled)->AsBoolean() ? TStrings::ZERO : TStrings::ONE;
    }

    // value of gender field in old format
    TString
    TDbFieldsConverter::GenderField(const TDbProfile* profile, TDbIndex index) {
        const TDbValue* val = profile->Get(index);
        return val->Exists ? (val->Value == "m" ? TStrings::ONE : TStrings::TWO) : TStrings::ZERO;
    }

    // value of regdate in old (MySql date) format
    TString
    TDbFieldsConverter::RegdateField(const TDbProfile* profile, TDbIndex index) {
        return NUtils::FormatTimestamp(profile->Get(index)->Value);
    }

    TString
    TDbFieldsConverter::MailLoginRuleField(const TDbProfile* profile,
                                           TDbIndex mailSuid,
                                           TDbIndex mailHostId,
                                           TDbIndex mailStatus,
                                           TDbIndex attr,
                                           const bool ignoreFrozen) {
        const TDbValue* rule = profile->Get(attr);
        if (HasMailSubscription(profile, mailSuid, mailHostId, mailStatus, ignoreFrozen)) {
            return rule->Exists ? rule->Value : TStrings::ONE;
        }
        return TStrings::EMPTY;
    }

    TString
    TDbFieldsConverter::MailHostIdField(const TDbProfile* profile,
                                        TDbIndex mailSuid,
                                        TDbIndex mailHostId,
                                        TDbIndex mailStatus,
                                        const TString& hostIdValue,
                                        const bool ignoreFrozen) {
        if (HasMailSubscription(profile, mailSuid, mailHostId, mailStatus, ignoreFrozen)) {
            return hostIdValue;
        }
        return TStrings::EMPTY;
    }

    TString
    TDbFieldsConverter::MailSuidField(const TDbProfile* profile,
                                      TDbIndex mailSuid,
                                      TDbIndex mailHostId,
                                      TDbIndex mailStatus,
                                      const bool ignoreFrozen) {
        if (HasMailSubscription(profile, mailSuid, mailHostId, mailStatus, ignoreFrozen)) {
            return profile->Get(mailSuid)->Value;
        }
        return TStrings::EMPTY;
    }

    // value of passport login rule, restored from 2 new attributes
    TString
    TDbFieldsConverter::PassportLoginRuleField(const TDbProfile* profile, TDbIndex changeReason) {
        unsigned char res = 1;
        const bool changeRequired = profile->Get(changeReason)->AsBoolean();
        if (changeRequired) {
            res += 4;
        }
        return IntToString<10>((int)res);
    }

    // value of lite login rule, "1" if lite with no portal login, else "3"
    TString
    TDbFieldsConverter::LiteLoginRuleField(const TDbProfile* profile, TDbIndex liteAlias, TDbIndex portalAlias) {
        if (!profile->Get(liteAlias)->Exists) {
            return TStrings::EMPTY;
        }

        return profile->Get(portalAlias)->Exists ? "3" : "1";
    }

    TString
    TDbFieldsConverter::ComplexField(const TDbProfile* profile,
                                     const TDbFieldsResult& sid2field,
                                     const TComplexDbField& field) {
        if (profile->Sid().empty()) {
            return TStrings::EMPTY;
        }
        // This shit is used to emulate subscription.<field>.- which is used by mail
        // For correct mail delivery we need to return <field> from sid 2, NOT from sid where user was found
        // But do it only if the profile found on a requested sid and has requested field set up
        for (const auto& [sid, func] : field) {
            if (sid == profile->Sid()) {             // check that we found profile on this sid
                if (!func.Value(profile).empty()) {  // and we have this field
                    return sid2field.Value(profile); // then return something different from sid 2
                }
            }
        }
        return TStrings::EMPTY;
    }

    TString
    TDbFieldsConverter::HostsField(const TDbProfile* profile,
                                   const THostsList& hosts,
                                   const TString& field,
                                   const TString& sid,
                                   TDbIndex mailSuid,
                                   TDbIndex mailHostId,
                                   TDbIndex mailStatus,
                                   const TString& mailHostIdValue,
                                   const bool ignoreFrozen) {
        THostsEntry host;

        // if mail sid or - and account found at mail sid, or none sid given
        if (sid == TStrings::MAIL_SID || sid == TStrings::MAIL_NAROD_SID || profile->Pdd() || sid == "-") {
            if (!HasMailSubscription(profile, mailSuid, mailHostId, mailStatus, ignoreFrozen)) {
                return TStrings::EMPTY;
            }
            if (!hosts.Find(mailHostIdValue, profile->Pdd(), host)) {
                return TStrings::EMPTY;
            }
        } else {
            return TStrings::EMPTY;
        }

        // here, host data is found and filled up, return field that was asked for
        if (field == FLD_DB_ID) {
            return host.DbId;
        }
        if (field == FLD_PRIO) {
            return host.Prio;
        }

        return TStrings::EMPTY;
    }

    TDbFieldsConverter::TDbFieldsConverter(TDbFetcher& fetcher, const THostsList& hosts, const TString& mailHostId)
        : Fetcher_(fetcher)
        , Hosts_(hosts)
        , MailHostId_(mailHostId)
    {
    }

    TDbFieldsConverter::TDbFieldsConverter(TDbFetcher& fetcher,
                                           const THostsList& hosts,
                                           const TString& mailHostId,
                                           const std::vector<TString>& sids,
                                           bool forceSid2)
        : ForceSid2_(forceSid2)
        , Fetcher_(fetcher)
        , Hosts_(hosts)
        , MailHostId_(mailHostId)
        , Sids_(sids)
    {
    }

    void TDbFieldsConverter::AddDbFields(const std::vector<TString>& dbfields,
                                         std::vector<TDbFieldsResult*>& results) {
        results.reserve(dbfields.size());
        for (const TString& field : dbfields) {
            results.push_back(Add(field));
        }
    }

    static void ThrowOnUnsupportedDbField(const TString& val) {
        throw TBlackboxError(TBlackboxError::EType::InvalidParams)
            << "Illegal dbfield: " << InvalidValue(val);
    }

    TDbFieldsResult*
    TDbFieldsConverter::Add(const TString& value) {
        TString::size_type pos = value.find('.');
        if (TString::npos == pos) {
            ThrowOnUnsupportedDbField(value);
        }
        TString::size_type pos2 = value.find('.', pos + 1);
        if (TString::npos == pos2) {
            ThrowOnUnsupportedDbField(value);
        }
        // check if already seen this value
        TValueMapType::iterator it = Values_.find(value);
        if (Values_.end() != it) {
            return &it->second;
        }

        TString table = value.substr(0, pos);
        TString field = value.substr(pos + 1, pos2 - pos - 1);
        TString type = value.substr(pos2 + 1);

        TFuncType func = GetFunc(table, field, type, value);
        return &(Values_.insert(std::make_pair(value, TDbFieldsResult(std::move(func)))).first->second);
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetFunc(const TString& table, const TString& field, const TString& type, const TString& value) const {
        if (table == TBL_ACCOUNTS) {
            if (type != TStrings::UID) {
                ThrowOnUnsupportedDbField(value);
            }
            return GetAccountsFunc(field);
        }
        if (table == TBL_USERINFO || table == TBL_ACCOUNT_INFO) {
            if (type != TStrings::UID) {
                ThrowOnUnsupportedDbField(value);
            }
            return GetUserInfoFunc(field);
        }
        if (table == TBL_SUBSCRIPTION) {
            return GetSubscriptionFunc(field, type, value);
        }
        if (table == TBL_USERINFO_SAFE) {
            if (type != TStrings::UID) {
                ThrowOnUnsupportedDbField(value);
            }
            return GetUserinfoSafeFunc(field);
        }
        if (table == TBL_PASSWORD_QUALITY) {
            if (type != TStrings::UID) {
                ThrowOnUnsupportedDbField(value);
            }
            return GetPasswordQualityFunc(field);
        }
        if (table == TBL_HOSTS) {
            return GetHostsFunc(field, type);
        }
        if (table == TBL_USERPHONES) {
            if (type != TStrings::UID) {
                ThrowOnUnsupportedDbField(value);
            }
            return GetUserphonesFunc(field);
        }
        if (table == TBL_ADTARGET) {
            if (type != TStrings::UID) {
                ThrowOnUnsupportedDbField(value);
            }
            return GetAdtargetFunc(field);
        }

        ThrowOnUnsupportedDbField(value);
        return {};
    }

    TDbFetcher& TDbFieldsConverter::Fetcher() {
        return Fetcher_;
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetAccountsFunc(const TString& field) const {
        if (field == FLD_LOGIN) {
            return [normalized = Fetcher_.AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN)](
                       const TDbProfile* profile) {
                return SimpleField(profile, normalized);
            };
        }
        if (field == FLD_ENA) {
            return [disabled = Fetcher_.AddAttr(TAttr::ACCOUNT_DISABLED)](
                       const TDbProfile* profile) {
                return EnaField(profile, disabled);
            };
        }
        if (field == FLD_GLOGOUT) {
            return [glogout = Fetcher_.AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME)](
                       const TDbProfile* profile) {
                return SimpleField(profile, glogout);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetUserInfoFunc(const TString& field) const {
        if (field == FLD_SEX) {
            return [gender = Fetcher_.AddAttr(TAttr::PERSON_GENDER)](
                       const TDbProfile* profile) {
                return GenderField(profile, gender);
            };
        }
        if (field == FLD_REGDATE) {
            return [regdate = Fetcher_.AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME)](
                       const TDbProfile* profile) {
                return RegdateField(profile, regdate);
            };
        }
        auto it = UserInfoFields().find(field);
        if (it != UserInfoFields().end()) {
            return [attrIdx = Fetcher_.AddAttr(it->second)](
                       const TDbProfile* profile) {
                return SimpleField(profile, attrIdx);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionFuncSid(const TString& field, const TString& sid) const {
        if (field == FLD_SUID) {
            return GetSubscriptionSuidFunc(sid);
        }
        if (field == FLD_LOGIN) {
            return GetSubscriptionLoginFunc(sid);
        }
        if (field == FLD_LOGIN_RULE) {
            return GetSubscriptionLoginRuleFunc(sid);
        }
        if (field == FLD_HOST_ID) {
            return GetSubscriptionHostIdFunc(sid);
        }
        if (field == FLD_BORN_DATE) {
            return GetSubscriptionBornDateFunc(sid);
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionFunc(const TString& field, const TString& sid, const TString& value) const {
        if (sid == "-") {
            if (ForceSid2_) {
                return GetSubscriptionFuncSid(field, TStrings::MAIL_SID);
            }
            return GetSubscriptionFuncDash(field, value);
        }
        return GetSubscriptionFuncSid(field, sid);
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionFuncDash(const TString& field, const TString& value) const {
        if (Sids_.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Illegal '-' dbfield suffix: not a (login,sid) query: " << InvalidValue(value);
        }

        TComplexDbField results;
        results.reserve(Sids_.size());
        for (const TString& sid : Sids_) {
            TFuncType func = GetSubscriptionFuncSid(field, sid);
            results.push_back(std::make_pair(sid, TDbFieldsResult(std::move(func))));
        }

        TFuncType sid2func = GetSubscriptionFuncSid(field, TStrings::MAIL_SID);
        return [func = TDbFieldsResult(std::move(sid2func)),
                results = std::move(results)](
                   const TDbProfile* profile) {
            return ComplexField(profile, func, results);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionSuidFunc(const TString& sid) const {
        if (sid == TStrings::MAIL_SID || sid == TStrings::MAIL_NAROD_SID) {
            return [mailSuid = Fetcher_.AddSuid2(),
                    mailHostId = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID),
                    mailStatus = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS),
                    this](
                       const TDbProfile* profile) {
                return MailSuidField(profile, mailSuid, mailHostId, mailStatus, ForceShowMailSubscription_);
            };
        }
        if (sid == TStrings::PASSPORT_SID) {
            return [](const TDbProfile* profile) {
                return ConstantField(profile, TStrings::ONE);
            };
        }
        if (sid == "33") {
            return [lite = Fetcher_.AddAlias(TAlias::LITE_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, lite, TStrings::ONE);
            };
        }
        if (sid == "36") {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
            return [secure = Fetcher_.AddAttr(TAttr::PHONES_SECURE)](
                       const TDbProfile* profile) {
                return CheckPhoneField(profile, secure, TStrings::ONE);
            };
        }
        if (sid == "42") {
            return [subscription = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_WWWDGT_MODE)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, subscription, TStrings::ONE);
            };
        }
        if (sid == "58") {
            return [social = Fetcher_.AddAlias(TAlias::SOCIAL_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, social, TStrings::ONE);
            };
        }
        if (sid == "61") {
            return [altDomain = Fetcher_.AddAlias(TAlias::ALT_DOMAIN_LOGIN)](
                       const TDbProfile* profile) {
                return GalatasarayField(profile, altDomain, TStrings::ONE);
            };
        }
        if (sid == "65") {
            return [phone = Fetcher_.AddAlias(TAlias::PHONE_NUMBER)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, phone, TStrings::ONE);
            };
        }
        if (sid == "68") {
            return [phony = Fetcher_.AddAlias(TAlias::PHONY_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, phony, TStrings::ONE);
            };
        }
        if (sid == "89") {
            return [phone = Fetcher_.AddAttr(TAttr::PERSON_CONTACT_PHONE)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, phone, TStrings::ONE);
            };
        }
        if (sid == "105") {
            return [](const TDbProfile* profile) {
                return CheckAliasPddField(profile, TStrings::ONE);
            };
        }
        if (sid == "669") {
            return [login = Fetcher_.AddAlias(TAlias::YANDEXOID_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, login, TStrings::ONE);
            };
        }

        TSidsToAttrMap::iterator it = SUBSCRIPTIONS.find(sid);
        if (it != SUBSCRIPTIONS.end()) {
            return [attrIdx = Fetcher_.AddAttr(it->second)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, attrIdx, TStrings::ONE);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionLoginFunc(const TString& sid) const {
        if (sid == TStrings::MAIL_SID) {
            return [mailSuid = Fetcher_.AddSuid2(),
                    mailHostId = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID),
                    mailStatus = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS),
                    mailAlias = Fetcher_.AddAlias(TAlias::MAIL_LOGIN),
                    portalAlias = Fetcher_.AddAlias(TAlias::PORTAL_LOGIN),
                    pddAlias = Fetcher_.AddAlias(TAlias::PDD_MASTER_LOGIN),
                    altAlias = Fetcher_.AddAlias(TAlias::ALT_DOMAIN_LOGIN),
                    this](
                       const TDbProfile* profile) {
                return MailLoginField(profile, mailSuid, mailHostId, mailStatus, mailAlias, portalAlias, pddAlias, altAlias, ForceShowMailSubscription_);
            };
        }
        if (sid == TStrings::PASSPORT_SID) {
            return [userLogin = Fetcher_.AddAttr(TAttr::ACCOUNT_USER_DEFINED_LOGIN),
                    normLogin = Fetcher_.AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN)](
                       const TDbProfile* profile) {
                return UserDefinedLoginField(profile, userLogin, normLogin);
            };
        }
        if (sid == TStrings::MAIL_NAROD_SID) {
            return [mailSuid = Fetcher_.AddSuid2(),
                    mailHostId = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID),
                    mailStatus = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS),
                    narodmailAlias = Fetcher_.AddAlias(TAlias::NAROD_MAIL_LOGIN),
                    mailAlias = Fetcher_.AddAlias(TAlias::MAIL_LOGIN),
                    portalAlias = Fetcher_.AddAlias(TAlias::PORTAL_LOGIN),
                    this](
                       const TDbProfile* profile) {
                return NarodMailLoginField(profile, mailSuid, mailHostId, mailStatus, narodmailAlias, mailAlias, portalAlias, ForceShowMailSubscription_);
            };
        }
        if (sid == "33") {
            return [lite = Fetcher_.AddAlias(TAlias::LITE_LOGIN)](
                       const TDbProfile* profile) {
                return SimpleField(profile, lite);
            };
        }
        if (sid == "36") {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);

            return [phoneNum = Fetcher_.AddAttr(TAttr::PHONE_NUMBER),
                    phoneId = Fetcher_.AddAttr(TAttr::PHONES_SECURE),
                    normLogin = Fetcher_.AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN)](
                       const TDbProfile* profile) {
                return PhoneLoginField(profile, phoneNum, phoneId, normLogin);
            };
        }
        if (sid == "58") {
            return [social = Fetcher_.AddAlias(TAlias::SOCIAL_LOGIN)](
                       const TDbProfile* profile) {
                return SimpleField(profile, social);
            };
        }
        if (sid == "61") {
            return [altDomain = Fetcher_.AddAlias(TAlias::ALT_DOMAIN_LOGIN)](
                       const TDbProfile* profile) {
                return GalatasarayLoginField(profile, altDomain);
            };
        }
        if (sid == "65") {
            return [phone = Fetcher_.AddAlias(TAlias::PHONE_NUMBER)](
                       const TDbProfile* profile) {
                return SimpleField(profile, phone);
            };
        }
        if (sid == "68") {
            return [phony = Fetcher_.AddAlias(TAlias::PHONY_LOGIN)](
                       const TDbProfile* profile) {
                return SimpleField(profile, phony);
            };
        }
        if (sid == "89") {
            return [phone = Fetcher_.AddAttr(TAttr::PERSON_CONTACT_PHONE)](
                       const TDbProfile* profile) {
                return SimpleField(profile, phone);
            };
        }
        if (sid == "669") {
            return [login = Fetcher_.AddAlias(TAlias::YANDEXOID_LOGIN)](
                       const TDbProfile* profile) {
                return SimpleField(profile, login);
            };
        }
        TDbIndex attr = -1;
        if (sid == "42") {
            attr = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_WWWDGT_MODE);
        } else {
            TSidsToAttrMap::iterator it = SUBSCRIPTIONS.find(sid);
            if (it != SUBSCRIPTIONS.end()) {
                attr = Fetcher_.AddAttr(it->second);
            }
        }

        if (attr >= 0) {
            return [attr,
                    normLogin = Fetcher_.AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN)](
                       const TDbProfile* profile) {
                return SidLoginField(profile, attr, normLogin);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionLoginRuleFunc(const TString& sid) const {
        if (sid == TStrings::MAIL_SID || sid == TStrings::MAIL_NAROD_SID) {
            return [mailSuid = Fetcher_.AddSuid2(),
                    mailHostId = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID),
                    mailStatus = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS),
                    mailRule = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_LOGIN_RULE),
                    this](
                       const TDbProfile* profile) {
                return MailLoginRuleField(profile, mailSuid, mailHostId, mailStatus, mailRule, ForceShowMailSubscription_);
            };
        }
        if (sid == TStrings::PASSPORT_SID) {
            return [changeReason = Fetcher_.AddAttr(TAttr::PASSWORD_FORCED_CHANGING_REASON)](
                       const TDbProfile* profile) {
                return PassportLoginRuleField(profile, changeReason);
            };
        }
        if (sid == "33") {
            return [liteAlias = Fetcher_.AddAlias(TAlias::LITE_LOGIN),
                    portalAlias = Fetcher_.AddAlias(TAlias::PORTAL_LOGIN)](
                       const TDbProfile* profile) {
                return LiteLoginRuleField(profile, liteAlias, portalAlias);
            };
        }
        if (sid == "36") {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
            return [phone = Fetcher_.AddAttr(TAttr::PHONES_SECURE)](
                       const TDbProfile* profile) {
                return CheckPhoneField(profile, phone, TStrings::ONE);
            };
        }
        if (sid == "42") {
            return [subscription = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_WWWDGT_MODE)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, subscription, TStrings::ONE);
            };
        }
        if (sid == "58") {
            return [social = Fetcher_.AddAlias(TAlias::SOCIAL_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, social, TStrings::ONE);
            };
        }
        if (sid == "61") {
            return [altDomain = Fetcher_.AddAlias(TAlias::ALT_DOMAIN_LOGIN)](
                       const TDbProfile* profile) {
                return GalatasarayField(profile, altDomain, TStrings::ONE);
            };
        }
        if (sid == "65") {
            return [phone = Fetcher_.AddAlias(TAlias::PHONE_NUMBER)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, phone, TStrings::ONE);
            };
        }
        if (sid == "68") {
            return [login = Fetcher_.AddAlias(TAlias::PHONY_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, login, TStrings::ONE);
            };
        }
        if (sid == "89") {
            return [phone = Fetcher_.AddAttr(TAttr::PERSON_CONTACT_PHONE)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, phone, TStrings::ONE);
            };
        }
        if (sid == "105") {
            return [](const TDbProfile* profile) {
                return CheckAliasPddField(profile, TStrings::ONE);
            };
        }
        if (sid == "669") {
            return [login = Fetcher_.AddAlias(TAlias::YANDEXOID_LOGIN)](
                       const TDbProfile* profile) {
                return CheckValueField(profile, login, TStrings::ONE);
            };
        }

        TSidsToAttrMap::iterator it = SUBSCRIPTIONS.find(sid);
        if (it != SUBSCRIPTIONS.end()) {
            return [attrIdx = Fetcher_.AddAttr(it->second)](
                       const TDbProfile* profile) {
                return SimpleField(profile, attrIdx);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionHostIdFunc(const TString& sid) const {
        TString type;
        if (sid == TStrings::MAIL_SID || sid == TStrings::MAIL_NAROD_SID) {
            return [mailSuid = Fetcher_.AddSuid2(),
                    mailHostId = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID),
                    mailStatus = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS),
                    defaultMailHostId = MailHostId_,
                    this](
                       const TDbProfile* profile) {
                return MailHostIdField(profile, mailSuid, mailHostId, mailStatus, defaultMailHostId, ForceShowMailSubscription_);
            };
        }
        if (sid == "36") {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
            return [phone = Fetcher_.AddAttr(TAttr::PHONES_SECURE)](
                       const TDbProfile* profile) {
                return PhoneTailField(profile, phone);
            };
        }
        if (sid == "42") {
            type.assign(TAttr::SUBSCRIPTION_WWWDGT_MODE);
        }

        if (type.empty()) {
            return [](const TDbProfile* profile) {
                return EmptyField(profile);
            };
        }
        return [attrIdx = Fetcher_.AddAttr(type)](
                   const TDbProfile* profile) {
            return SimpleField(profile, attrIdx);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetSubscriptionBornDateFunc(const TString& sid) const {
        if (sid == "36") {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::CONFIRMED);
            return [phone = Fetcher_.AddAttr(TAttr::PHONES_SECURE)](
                       const TDbProfile* profile) {
                return PhoneBornDate(profile, phone);
            };
        }
        if (sid == TStrings::MAIL_SID || sid == TStrings::MAIL_NAROD_SID || sid == "66") {
            return [regdate = Fetcher_.AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME)](
                       const TDbProfile* profile) {
                return RegdateField(profile, regdate);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetUserinfoSafeFunc(const TString& field) const {
        if (field == FLD_HINTQ) {
            return [hintq = Fetcher_.AddAttr(TAttr::HINT_QUESTION_SERIALIZED)](
                       const TDbProfile* profile) {
                return SimpleField(profile, hintq);
            };
        }
        if (field == FLD_HINTA) {
            return [hinta = Fetcher_.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED)](
                       const TDbProfile* profile) {
                return SimpleField(profile, hinta);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetPasswordQualityFunc(const TString& field) const {
        if (field == FLD_VERSION) {
            return [pwdQuality = Fetcher_.AddAttr(TAttr::PASSWORD_QUALITY)](
                       const TDbProfile* profile) {
                return SplitPairField(profile, pwdQuality, ':', true);
            };
        }
        if (field == FLD_QUALITY) {
            return [pwdQuality = Fetcher_.AddAttr(TAttr::PASSWORD_QUALITY)](
                       const TDbProfile* profile) {
                return SplitPairField(profile, pwdQuality, ':', false);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetHostsFunc(const TString& field, const TString& sid) const {
        return [field,
                sid,
                mailSuid = Fetcher_.AddSuid2(),
                mailHostId = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_HOST_ID),
                mailStatus = Fetcher_.AddAttr(TAttr::SUBSCRIPTION_MAIL_STATUS),
                this](
                   const TDbProfile* profile) {
            return HostsField(profile, Hosts_, field, sid, mailSuid, mailHostId, mailStatus, MailHostId_, ForceShowMailSubscription_);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetUserphonesFunc(const TString& field) const {
        if (field == FLD_NUMBER) {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
            return [phone = Fetcher_.AddAttr(TAttr::PHONES_SECURE)](
                       const TDbProfile* profile) {
                return PhoneNumberField(profile, phone);
            };
        }
        if (field == FLD_CONFIRMED) {
            Fetcher_.AddExtendedPhoneAttr(TPhoneAttr::CONFIRMED);
            return [phone = Fetcher_.AddAttr(TAttr::PHONES_SECURE)](
                       const TDbProfile* profile) {
                return PhoneBornDate(profile, phone);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsConverter::TFuncType
    TDbFieldsConverter::GetAdtargetFunc(const TString& field) const {
        if (field == FLD_UID) {
            return [flag = Fetcher_.AddAttr(TAttr::ACCOUNT_AD_CAMPAIGN_PARTICIPANT)](
                       const TDbProfile* profile) {
                return ConditionalUid(profile, flag);
            };
        }
        return [](const TDbProfile* profile) {
            return EmptyField(profile);
        };
    }

    TDbFieldsResult::TDbFieldsResult(TDbFieldsConverter::TFuncType&& func)
        : Func_(std::move(func))
    {
    }

    const TString&
    TDbFieldsResult::Value(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return TStrings::EMPTY;
        }
        if (profile != Profile_) {
            Value_ = Func_(profile);
            Profile_ = profile;
        }
        return Value_;
    }

    void TDbFieldsConverter::InitMaps(TSidsToAttrMap&& sids) {
        SUBSCRIPTIONS = std::move(sids);
    }

}
