#include "exception.h"

#include "strings.h"

#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <map>

namespace NPassport::NBb {
    namespace {
        struct TErrorStatuses {
            TErrorStatuses()
                : Map({
                      {TBlackboxError::EType::Unknown, "UNKNOWN"},
                      {TBlackboxError::EType::InvalidParams, "INVALID_PARAMS"},
                      {TBlackboxError::EType::DbException, "DB_EXCEPTION"},
                      {TBlackboxError::EType::NoDataKeyspace, "NO_DATA_KEYSPACE"},
                      {TBlackboxError::EType::KeyNotFound, "KEY_NOT_FOUND"},
                      {TBlackboxError::EType::KeyspaceEmpty, "KEYSPACE_EMPTY"},
                      {TBlackboxError::EType::KeyspaceFailed, "KEYSPACE_FAILED"},
                      {TBlackboxError::EType::BadSign, "BAD_SIGN"},
                      {TBlackboxError::EType::AccessDenied, "ACCESS_DENIED"},
                  })
            {
                Y_VERIFY(Map.size() == 9, "Forgotten some descriptions");
            }

            const std::map<TBlackboxError::EType, TString> Map;
        } const ERROR_STATUSES;
    }

    const TString&
    TBlackboxError::StatusStr() const {
        return StatusStr(Status_);
    }

    const TString& TBlackboxError::StatusStr(EType status) {
        auto it = ERROR_STATUSES.Map.find(status);
        return ERROR_STATUSES.Map.end() == it ? TStrings::UNKNOWN : it->second;
    }

    TDbpoolError::TDbpoolError(const TStringBuf details, const TStringBuf what)
        : TBlackboxError(TBlackboxError::EType::DbException)
    {
        (*this) << details << ": " << what;
    }

    TExceptionInfo::TExceptionInfo()
        : CurrentTime(TInstant::Now().ToStringLocal())
    {
    }

    TString TExceptionInfo::ToString() const {
        TStringStream res;
        res.Reserve(256);

        TStringBuf delim = ". ";

        res << Msg << delim;
        res << "request_id=" << NUtils::GetThreadLocalRequestId() << delim;

        if (!ExtendedDescription.empty()) {
            res << "ext=" << ExtendedDescription << delim;
        }
        if (!Method.empty()) {
            res << "method=" << Method << delim;
        }
        res << "host=" << Host << delim;
        res << "hostname=" << Hostname << delim;
        res << "current_time=" << CurrentTime;

        return res.Str();
    }
}
