#include "experiment.h"

#include <passport/infra/libs/cpp/utils/crypto/rsa.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <contrib/libs/libsodium/include/sodium/crypto_sign_ed25519.h>
#include <contrib/libs/openssl/include/openssl/bio.h>
#include <contrib/libs/openssl/include/openssl/err.h>
#include <contrib/libs/openssl/include/openssl/pem.h>
#include <contrib/libs/openssl/include/openssl/x509.h>

#include <library/cpp/openssl/holders/evp.h>
#include <library/cpp/openssl/init/init.h>

namespace NPassport::NBb {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    TExperiment TExperiment::ins_;

    namespace {
        struct TErrCleaner {
            TErrCleaner() {
                ERR_clear_error();
            }

            ~TErrCleaner() {
                ERR_clear_error();
            }
        };

        TStringBuf GetOpensslError() {
            return ERR_reason_error_string(ERR_get_error());
        }

        template <typename Func>
        NPassport::NUtils::TEvpKey ParseKey(TStringBuf pem, int expected, Func func) {
            TErrCleaner cleaner;

            std::unique_ptr<BIO, decltype(&BIO_free)> bio(
                BIO_new_mem_buf(pem.data(), pem.size()),
                BIO_free);
            if (!bio) {
                throw std::bad_alloc();
            }

            NPassport::NUtils::TEvpKey res;
            res.Pkey.reset(func(bio.get(), nullptr, nullptr, nullptr));
            if (!res.Pkey) {
                ythrow yexception() << "Failed : " << GetOpensslError();
            }

            const int actual = EVP_PKEY_id(res.Pkey.get());
            if (expected != actual) {
                ythrow yexception()
                    << "Wrong type of key:"
                    << " expected==" << expected << "(" << OBJ_nid2sn(expected) << "),"
                    << " actual==" << actual << "(" << OBJ_nid2sn(actual) << ")";
            }

            return res;
        }

        class IPublicKey {
        public:
            virtual ~IPublicKey() = default;
            virtual bool Verify(TStringBuf data, TStringBuf signature) const = 0;
        };

        class TRsaPublicKey: public IPublicKey {
        public:
            TRsaPublicKey(TStringBuf pem)
                : Key_(NPassport::NUtils::TRsaPublicEvp::FromPem(pem))
            {
            }

            bool Verify(TStringBuf data, TStringBuf signature) const override {
                return Key_.VerifyWithSha256(data, signature).IsSuccess;
            }

        private:
            NPassport::NUtils::TRsaPublicEvp Key_;
        };

        const TRsaPublicKey RSA_KEY(
            "-----BEGIN PUBLIC KEY-----\n"
            "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAznZTGga3nrA5gkhlY4Vx\n"
            "vDaPUoqmnx3YXBNdfeW4gJF5KvzykUTt3A1VPeuNYmWWndFKhIed7ciytCxlIgxo\n"
            "V3yJWIe4A/YznlKHIddUcHXG7wB74NQIrlZ6CrfvAmDSNPHSNpdDCNYFioY9Xq7w\n"
            "G8BLMpB+g6Qil8K+N5rCmdg2UiQPWqCFE4Dh3kb35ozVBohiTxxwDzWQ4uYDP66K\n"
            "kknrpJBPRFWClm6sWhegUbSOBMZAP9KXxuEIsDi2IquwkG2j1WEXYxsozstqgBZ1\n"
            "Ykic67K89IAd8IIwBh/DQa3dZrbY4ixokZMchyf6FGJN3r+gLzxBS0fukGfH760J\n"
            "9QIDAQAB\n"
            "-----END PUBLIC KEY-----\n");

        class TEcdsaPublicKey: public IPublicKey {
        public:
            TEcdsaPublicKey(TStringBuf pem)
                : Key_(ParseKey(pem, EVP_PKEY_EC, PEM_read_bio_PUBKEY))
            {
            }

            bool Verify(TStringBuf data, TStringBuf signature) const override {
                const int res = ECDSA_verify(
                    0,
                    (const unsigned char*)data.data(),
                    data.size(),
                    (const unsigned char*)signature.data(),
                    signature.size(),
                    EVP_PKEY_get0_EC_KEY(Key_.Pkey.get()));

                Y_ENSURE(1 == res || 0 == res,
                         "failed to verify: " << res << ": " << GetOpensslError());
                return 1 == res;
            }

        private:
            NPassport::NUtils::TEvpKey Key_;
        };

        const TEcdsaPublicKey ECDSA_KEY(
            "-----BEGIN PUBLIC KEY-----\n"
            "MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEsdV6eF57E/dr0k0NPugKfhpiy7iD\n"
            "zN9hKd9UdLdSCSOPClT4BSgpBRcOkBMBXOAeY6n66iXEgTR/r+ovgJkSeQ==\n"
            "-----END PUBLIC KEY-----\n");

        class TEddsaSodiumPublicKey: public IPublicKey {
        public:
            TEddsaSodiumPublicKey(TStringBuf key)
                : Key_(key)
            {
            }

            bool Verify(TStringBuf data, TStringBuf signature) const override {
                return 0 == crypto_sign_ed25519_verify_detached(
                                (unsigned char*)signature.data(),
                                (unsigned char*)data.data(),
                                data.size(),
                                (unsigned char*)Key_.data());
            }

        private:
            TString Key_;
        };

        const TEddsaSodiumPublicKey EDDSA_KEY(NUtils::Base64ToBin("DDfMMF8Nj2+yLV+hkhH2i0oqMQZsype8CjD7JTrCzjk="));

        const TString STRING_TO_SIGN = "3:1416923638.5.0.1416922797713:h5RsXw:11.0|70495.-1.0|70496.-1.0|70497.-1.0|70498.-1.0|70499.-1.0|70500.-1.0|70501.-1.400|70502.-1.0|70503.-1.0|70504.-1.0|70505.-1.0|70506.-1.0|71131.876534.BhyfBqqwywQx4e48UIJzIBFBii8";

        const TString RSA_SIGNATURE = "UUu2M/wXAX6Uue8gQzvVb+ksU3xS0K0E5gxgU+VJK/+j5TtnJ5vARFBg/g1a3f2AAPpPE62eBGIle3QLsxbkabtEmOREe8ATw5m+Lrou7I/ouVAfsIjUPIdGEO5ctOifNBLWVmoAig0lN9Xb/3iRauwKbM32pYkKWjRIiXXL6IQ0Q5PeTVery1RkudHInZX19m2cni0qUZAgPFScf70/Luq8GdCk/IBSF2VD/1GrRlZt01KObfVL83piuWs21fX0EC4PuexaScMnRulDjUsl1qQ2AKz1t8RZ93vNesaFnYF/fNNK33+EHDctjLw8v1RHQrhvkmTt4PfYJQ7yZ5DF9A==";
        const TString ECDSA_SIGNATURE = "MEYCIQDwYeBhMvhP+G3PO2JcZpEWnNi/6BJPLxKo5DFBkKR85QIhANDpgGgWiiVU5J7Hic4kqxWCSbuFp1mtBB/WwKarvUK5";
        const TString EDDSA_SIGNATURE = "7T8SoB9v2/oZCxhlV8nZxlS5TCqoiu91YMa82rpmSvGy8WeQtbf/8P4MD55eelRDZY/N+7ySaSi56MSCxddNDQ==";
    }

    static void CheckRsa() {
        try {
            Y_ENSURE(RSA_KEY.Verify(STRING_TO_SIGN, NUtils::Base64ToBin(RSA_SIGNATURE)));
        } catch (...) {
            TLog::Warning() << "Experiment: RSA check failed!";
        }
    }

    static void CheckEcdsa() {
        try {
            Y_ENSURE(ECDSA_KEY.Verify(STRING_TO_SIGN, NUtils::Base64ToBin(ECDSA_SIGNATURE)));
        } catch (...) {
            TLog::Warning() << "Experiment: ECDSA check failed!";
        }
    }

    static void CheckEddsa() {
        try {
            Y_ENSURE(EDDSA_KEY.Verify(STRING_TO_SIGN, NUtils::Base64ToBin(EDDSA_SIGNATURE)));
        } catch (...) {
            TLog::Warning() << "Experiment: EDDSA check failed!";
        }
    }

    bool TExperiment::IsInHalfIpExperiment(const TString& userip) const {
        return std::hash<TString>{}(userip) % 100 < BadauthUseHalfIpv6AddressRate;
    }

    void TExperiment::RunCheck(const TString& method) {
        if (method == "RSA") {
            CheckRsa();
        } else if (method == "ECDSA") {
            CheckEcdsa();
        } else if (method == "EDDSA") {
            CheckEddsa();
        }
    }
}
