#include "hosts_list.h"

#include "strings.h"

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NBb {
    THostsList::THostsList(NDbPool::TDbPool& db)
        : Db_(db)
    {
        try {
            Hosts_.Set(Load(false));
            DomainsHosts_.Set(Load(true));

            TLog::Info("Blackbox: HostsList: loaded hosts list: <%lu> hosts items, <%lu> domains_hosts items",
                       Hosts_.Get()->size(),
                       DomainsHosts_.Get()->size());
        } catch (const std::exception& e) {
            TLog::Warning("Blackbox: HostsList: failed to load hosts/domain_hosts tables at startup. <%s>", e.what());
        }
    }

    THostsList::~THostsList() = default;

    bool THostsList::Find(const TString& hostId, bool pdd, THostsEntry& out) const {
        std::shared_ptr<THostsMap> hosts = pdd ? DomainsHosts_.Get()
                                               : Hosts_.Get();
        if (!hosts) {
            return false;
        }

        THostsMap::const_iterator it = hosts->find(hostId);
        if (it == hosts->end()) {
            return false;
        }

        out = it->second;
        return true;
    }

    static const TString HOSTS_QUERY("SELECT host_id,db_id,prio FROM ");

    std::shared_ptr<THostsList::THostsMap> THostsList::Load(bool pdd) {
        TString query(HOSTS_QUERY);
        query.append(pdd ? "domains_hosts" : "hosts");
        std::unique_ptr<NDbPool::TResult> result;
        try {
            // Create DB handle
            NDbPool::TBlockingHandle sqlh(Db_);
            result = sqlh.Query(query);
        } catch (const NDbPool::TException& e) {
            throw yexception() << "Error: HostsList: dbpool exception in the query: " << e.what()
                               << ", query: <" << query << ">, " << Db_.GetDbInfo();
        }

        std::shared_ptr<THostsMap> hosts = std::make_shared<THostsMap>();

        for (const NDbPool::TRow& row : result->Table()) {
            TString id = row[0].AsString();
            THostsEntry& item = hosts->insert(std::make_pair(id, THostsEntry())).first->second;
            item.DbId = row[1].IsNull() ? TStrings::EMPTY : row[1].AsString();
            item.Prio = row[2].IsNull() ? TStrings::EMPTY : row[2].AsString();
        }

        return hosts;
    }

    void THostsList::Run() {
        try {
            std::shared_ptr<THostsMap> hosts = Load(false);
            std::shared_ptr<THostsMap> domainsHosts = Load(true);

            Hosts_.Set(hosts);
            DomainsHosts_.Set(domainsHosts);

            TLog::Info("Blackbox: HostsList: loaded hosts list: <%lu> hosts items, <%lu> domains_hosts items",
                       hosts->size(),
                       domainsHosts->size());
        } catch (const std::exception& e) {
            TLog::Warning("Blackbox: HostsList: failed to refresh hosts. <%s>", e.what());
        }
    }

}
