#pragma once

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/geobase/lookup.hpp>

#include <util/generic/string.h>

#include <algorithm>

namespace NPassport::NBb {
    class TIpComparator {
    public:
        using TIp = TString;
        virtual ~TIpComparator() = default;
        virtual bool Equal(const TIp& lhs, const TIp& rhs) const = 0;
    };

    class TDirectIpComparator: public TIpComparator {
    public:
        bool Equal(const TIp& lhs, const TIp& rhs) const override {
            return lhs == rhs;
        }
    };

    class TAsIpComparator: public TIpComparator {
    public:
        TAsIpComparator(const NGeobase::TLookup& lookup)
            : Lookup_(lookup)
        {
        }

        bool Equal(const TIp& lhs, const TIp& rhs) const override {
            const auto leftVec = As(lhs);
            const auto rightVec = As(rhs);
            if (leftVec.empty() || rightVec.empty()) {
                return rhs == lhs;
            }

            for (const std::string& leftAs : leftVec) {
                if (rightVec.end() != std::find(rightVec.begin(), rightVec.end(), leftAs)) {
                    return true;
                }
            }

            return false;
        }

    private:
        std::vector<std::string> As(const TIp& ip) const {
            try {
                return Lookup_.GetAsTraitsByIp(ip).AsList;
            } catch (const std::exception& e) {
                TLog::Debug("AsIpComparator::as(%s) exception: %s", ip.c_str(), e.what());
            } catch (...) {
                TLog::Debug("AsIpComparator::as(%s) unknown exception", ip.c_str());
            }
            return {};
        }
        const NGeobase::TLookup& Lookup_;
    };

}
