#include "lcookie_parser.h"

#include "exception.h"
#include "strings.h"
#include "utils.h"

#include <passport/infra/daemons/blackbox/src/output/lcookie_result.h>

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/random.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

#include <array>
#include <vector>

namespace NPassport::NBb {
    static const TString LCOOKIE_KSPACE("cookiel");

    static const unsigned char B62_ENCODE[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

    static std::array<unsigned char, 256> B62Init() noexcept {
        std::array<unsigned char, 256> res{};
        for (size_t i = 0; i < res.size(); ++i) {
            res[i] = 0xff;
        }

        for (size_t i = 0; i < 62; ++i) {
            res[B62_ENCODE[i]] = i;
        }

        return res;
    }
    static const std::array<unsigned char, 256> B62_DECODE = B62Init();

    std::unique_ptr<TLCookieResult> TLCookieParser::Parse(const TString& lcookie, const NAuth::TSessionSigner& sessSigner) {
        TString cookie(lcookie);

        // fix possible issue with spaces in body
        for (char& c : cookie) {
            if (c == '.') {
                break;
            }
            if (c == ' ') {
                c = '+';
            }
        }

        // split cookie value
        std::vector<TStringBuf> fields = NUtils::ToVector<TStringBuf>(cookie, '.');

        if (fields.size() != 5) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Cookie l body has broken format";
        }

        const TStringBuf body = fields[0];
        const TStringBuf timestamp = fields[1];
        const TStringBuf keyId = fields[2];
        const TStringBuf rnd = fields[3];
        const TStringBuf signature = fields[4];

        // find a key
        NAuth::TKeyRing* ring = sessSigner.GetRingByName(LCOOKIE_KSPACE);
        if (!ring) {
            throw TBlackboxError(TBlackboxError::EType::KeyspaceFailed)
                << "Failed to get keyring for keyspace " << LCOOKIE_KSPACE;
        }

        const NAuth::TRandomPtr key = ring->GetRandomById(keyId);
        if (!key) {
            throw TBlackboxError(TBlackboxError::EType::KeyNotFound)
                << "Key not found in keyspace " << LCOOKIE_KSPACE;
        }

        size_t lastdot = cookie.rfind('.');
        TString tosign = cookie.substr(0, lastdot) + key->GetBody();

        // check signature
        if (NUtils::Bin2hex(NUtils::TCrypto::Md5(tosign)) != signature) {
            throw TBlackboxError(TBlackboxError::EType::BadSign) << "Cookie l signature broken";
        }

        if (rnd.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Cookie l body has broken format";
        }
        int version = rnd[0] - '0';

        // decode and unxor body
        TString decoded = NUtils::Base64ToBin(body);
        if (decoded.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Cookie l is malformed: bad base64";
        }
        if (decoded.size() < 24) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Cookie l is malformed: body is too small";
        }

        const TStringBuf keybody = key->GetBody();
        const size_t keylen = keybody.size();

        for (size_t i = 0; i < decoded.size(); ++i) {
            *(decoded.begin() + i) ^= keybody[i % keylen];
        }

        size_t uidOffset = 4 + B62_DECODE[(ui8)decoded[0]];
        size_t uidLength = B62_DECODE[(ui8)decoded[1]];
        size_t loginOffset = 24 + B62_DECODE[(ui8)decoded[2]];
        size_t loginLength = version < 3 ? B62_DECODE[(ui8)decoded[3]] : TString::npos;

        if (uidOffset + uidLength > decoded.size() || loginOffset + loginLength > decoded.size()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Cookie l is malformed: encoded offset+length is to big";
        }

        TString login = decoded.substr(loginOffset, loginLength);

        std::unique_ptr<TLCookieResult> result = std::make_unique<TLCookieResult>();

        result->Uid = decoded.substr(uidOffset, uidLength);
        result->Timestamp = timestamp;
        result->Login = login;
        result->DisplayLogin = TUtils::IsSyntheticLogin(login) ? TStrings::EMPTY : login;

        return result;
    }
}
