#include "login_status.h"

#include "strings.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    const TString TLoginStatus::DEFAULT_RET_COMMENT("unknown (internal error?)");

    const TLoginStatus::TStatusPair TLoginStatus::PAIR_UNKNOWN(
        TLoginStatus::acc_Unknown, TLoginStatus::pwd_Unknown);
    const TLoginStatus::TStatusPair TLoginStatus::PAIR_NOT_FOUND(
        TLoginStatus::acc_Not_Found, TLoginStatus::pwd_Unknown);
    const TLoginStatus::TStatusPair TLoginStatus::PAIR_BAD_PASSWORD(
        TLoginStatus::acc_Valid, TLoginStatus::pwd_Bad);
    const TLoginStatus::TStatusPair TLoginStatus::PAIR_VALID(
        TLoginStatus::acc_Valid, TLoginStatus::pwd_Valid);
    const TLoginStatus::TStatusPair TLoginStatus::PAIR_SECOND_STEP(
        TLoginStatus::acc_Valid, TLoginStatus::pwd_Second_Step);

    TLoginStatus::TLoginStatus(int apiVer)
        : Comment_(DEFAULT_RET_COMMENT)
        , ApiVersion_(apiVer)
    {
    }

    TLoginStatus::TLoginStatus(TStatusPair status, const TString& comment, int apiVer)
        : ApiVersion_(apiVer)
    {
        Assign(status, false, false, false);
        Comment_.assign(comment);
    }

    void TLoginStatus::Assign(TStatusPair status, bool shallRestrict, bool canResist, bool canCaptcha, bool expired) {
        LoginStatus_ = status.first;
        PasswordStatus_ = status.second;

        switch (LoginStatus_) {
            case acc_Unknown:
                LegacyStatus_ = canCaptcha ? ELegacyStatus::SHOW_CAPTCHA : ELegacyStatus::INVALID;
                AuthFlag_ = TAuthLog::BLOCKED;
                Comment_.assign("auth attempt blocked: ").append(RestrictComment_);
                break;
            case acc_Not_Found:
                LegacyStatus_ = ELegacyStatus::INVALID;
                AuthFlag_ = TAuthLog::BAD;
                Comment_.assign("Login not found");
                break;
            case acc_Valid:
                if (PasswordStatus_ == pwd_Valid) {
                    LegacyStatus_ = ELegacyStatus::VALID;
                    AuthFlag_ = TAuthLog::OK;
                    Comment_.assign("OK");
                } else if (PasswordStatus_ == pwd_Second_Step) {
                    LegacyStatus_ = ELegacyStatus::SECOND_STEP_REQUIRED;
                    AuthFlag_ = TAuthLog::SECONDSTEP;
                    Comment_.assign("Second step required");
                } else {
                    LegacyStatus_ = ELegacyStatus::INVALID;
                    AuthFlag_ = TAuthLog::BAD;
                    Comment_.assign("Bad password");
                    if (!ScopeComment_.empty()) {
                        Comment_.append(": ").append(ScopeComment_);
                    }
                }
                break;
            case acc_Disabled:
                if (PasswordStatus_ == pwd_Valid) {
                    LegacyStatus_ = ELegacyStatus::DISABLED;
                    AuthFlag_ = TAuthLog::DISABLED;
                    Comment_.assign("Account disabled");
                } else {
                    LegacyStatus_ = ELegacyStatus::INVALID;
                    AuthFlag_ = TAuthLog::BAD;
                    Comment_.assign("Bad password; account disabled");
                }
                break;
        }

        // Under a bruteforce attack need more processing:
        // - for backwards compatibility we must set status to SHOW_CAPTCHA
        // for those callers that can show captcha
        // - for those that can resist bruteforce set appropriate policy
        if (shallRestrict && canResist) {
            if (canCaptcha) {
                LegacyStatus_ = ELegacyStatus::SHOW_CAPTCHA;
                Policy_ = TLoginStatus::resist_Captcha;
            } else {
                Policy_ = TLoginStatus::resist_Delay;
                Delay_ = 1000;
            }
            NUtils::AppendSeparated(Comment_, "; ", RestrictComment_);
        }

        if (expired) {
            // 'Expired' has lowest priority, return if no other status
            if (PasswordStatus_ == pwd_Valid || PasswordStatus_ == pwd_Second_Step) {
                LegacyStatus_ = ELegacyStatus::EXPIRED;
                Comment_.assign("Password expired");
                Policy_ = TLoginStatus::resist_Expired;
            }
        }
    }

    bool TLoginStatus::operator==(TStatusPair pair) const {
        return pair.first == LoginStatus_ && pair.second == PasswordStatus_;
    }

    void TLoginStatus::FormatRestrictComment(const TString& authlogComment,
                                             const TString& loginlogComment) {
        RestrictComment_.assign("Too many login failures");
        if (!authlogComment.empty()) {
            RestrictComment_.append(": ").append(authlogComment);
        }
        if (!loginlogComment.empty()) {
            RestrictComment_.append(authlogComment.empty() ? ": " : "; ").append(loginlogComment);
        }
    }

    void TLoginStatus::SetScopeComment(const TString& comment) {
        ScopeComment_.assign(comment);
    }

    static const TString UNKNOWN_STATUS("UNKNOWN");
    static const TString NOT_FOUND("NOT_FOUND");
    static const TString DISABLED("DISABLED");
    static const TString BAD_PASSWORD("BAD");
    static const TString VALID("VALID");
    static const TString SECOND_STEP("SECOND_STEP_REQUIRED");
    static const TString INVALID("INVALID");
    static const TString SHOW_CAPTCHA("SHOW_CAPTCHA");
    static const TString EXPIRED("EXPIRED");

    const TString&
    TLoginStatus::AccountStatusName(TLoginStatus::EAccount status) {
        switch (status) {
            case acc_Unknown:
                return UNKNOWN_STATUS;
            case acc_Not_Found:
                return NOT_FOUND;
            case acc_Disabled:
                return DISABLED;
            case acc_Valid:
                return VALID;
        }
    }

    const TString&
    TLoginStatus::PasswordStatusName(TLoginStatus::EPassword status) {
        switch (status) {
            case pwd_Unknown:
                return UNKNOWN_STATUS;
            case pwd_Bad:
                return BAD_PASSWORD;
            case pwd_Valid:
                return VALID;
            case pwd_Second_Step:
                return SECOND_STEP;
        }
    }

    const TString& TLoginStatus::LegacyStatusAsString() const {
        return LegacyStatusName(LegacyStatus_);
    }

    const TString& TLoginStatus::LegacyStatusName(ELegacyStatus status) {
        switch (status) {
            case ELegacyStatus::VALID:
                return VALID;
            case ELegacyStatus::DISABLED:
                return DISABLED;
            case ELegacyStatus::INVALID:
                return INVALID;
            case ELegacyStatus::SHOW_CAPTCHA:
                return SHOW_CAPTCHA;
            case ELegacyStatus::EXPIRED:
                return EXPIRED;
            case ELegacyStatus::SECOND_STEP_REQUIRED:
                return SECOND_STEP;
        }
    }

}
