#pragma once

#include <passport/infra/daemons/blackbox/src/loggers/authlog.h>

#include <util/generic/string.h>

namespace NPassport::NBb {
    class TLoginStatus {
    public:
        enum EAccount {
            acc_Unknown = 0,
            acc_Valid = 1,
            acc_Not_Found = 3,
            acc_Disabled = 4
        };

        enum EPassword {
            pwd_Unknown = 0,
            pwd_Valid = 1,
            pwd_Bad = 2,
            pwd_Second_Step = 4
        };

        using TStatusPair = std::pair<EAccount, EPassword>;

        enum EResistPolicy {
            resist_None = 0,
            resist_Captcha = 1,
            resist_Delay = 2,
            resist_Expired = 3
        };

        // Possible account status
        enum class ELegacyStatus {
            VALID = 0,               // valid, all data present
            DISABLED = 1,            // invalid, no data, account is disabled
            INVALID = 2,             // invalid, all other cases (sign broken, garbage, etc.)
            SHOW_CAPTCHA = 3,        // invalid, shall show CAPTCHA
            EXPIRED = 6,             // password expired
            SECOND_STEP_REQUIRED = 7 // valid, but this is 2FA user and second step required
        };

        TLoginStatus(int apiVer);
        TLoginStatus(TStatusPair status, const TString& comment, int apiVer);

        void FormatRestrictComment(const TString& authLogComment, const TString& loginlogComment);
        void SetScopeComment(const TString& comment);

        void Assign(TStatusPair, bool shallRestrict, bool canResist, bool canCaptcha, bool expired = false);
        bool operator==(TStatusPair) const;

        EAccount AccountStatus() const {
            return LoginStatus_;
        }
        EPassword PasswordStatus() const {
            return PasswordStatus_;
        }
        int GetLegacyStatus() const {
            return static_cast<int>(LegacyStatus_);
        }
        const TString& Comment() const {
            return Comment_;
        }
        TAuthLog::EFlag AuthFlag() const {
            return AuthFlag_;
        }
        EResistPolicy Policy() const {
            return Policy_;
        }
        unsigned long Delay() const {
            return Delay_;
        }
        int ApiVersion() const {
            return ApiVersion_;
        }

        const TString& LegacyStatusAsString() const;

        static const TString& LegacyStatusName(ELegacyStatus);
        static const TString& AccountStatusName(EAccount);
        static const TString& PasswordStatusName(EPassword);

        static const TStatusPair PAIR_UNKNOWN;
        static const TStatusPair PAIR_NOT_FOUND;
        static const TStatusPair PAIR_BAD_PASSWORD;
        static const TStatusPair PAIR_VALID;
        static const TStatusPair PAIR_SECOND_STEP;

    private:
        EAccount LoginStatus_ = acc_Unknown;
        EPassword PasswordStatus_ = pwd_Unknown;
        ELegacyStatus LegacyStatus_ = ELegacyStatus::INVALID;
        TAuthLog::EFlag AuthFlag_ = TAuthLog::BAD;
        EResistPolicy Policy_ = resist_None;
        unsigned long Delay_ = 0; // ms
        TString Comment_;
        TString RestrictComment_;
        TString ScopeComment_;

        const int ApiVersion_;

        static const TString DEFAULT_RET_COMMENT;
    };

}
