#include "partitions.h"

#include "exception.h"

#include <passport/infra/libs/cpp/utils/string/format.h>

namespace NPassport::NBb {
    static ui64 GetDefaultPartition(const TPartitionsSettings::TPartitionIdByName& partitions, const TString& defaultPartition) {
        auto it = partitions.find(defaultPartition);
        Y_ENSURE(it != partitions.end(), "Missing default partition '" << defaultPartition << "'");
        return it->second;
    }

    static TPartitionsSettings::TPartitionNameById ReversePartitionsMap(const TPartitionsSettings::TPartitionIdByName& partitionsByName) {
        TPartitionsSettings::TPartitionNameById partitionsById;
        for (const auto& [name, id] : partitionsByName) {
            bool inserted = partitionsById.emplace(id, name).second;
            Y_ENSURE(inserted, "Duplicated partition id '" << id << "'");
        }

        return partitionsById;
    }

    TPartitionsSettings::TPartitionsSettings(TPartitionIdByName&& partitions, const TString& defaultPartition)
        : PartitionsByName_(std::move(partitions))
        , PartitionsById_(ReversePartitionsMap(PartitionsByName_))
        , DefaultPartition_(GetDefaultPartition(PartitionsByName_, defaultPartition))
    {
    }

    bool TPartitionsSettings::IsDefaultPartition(ui64 id) const {
        return DefaultPartition_ == id;
    }

    bool TPartitionsSettings::IsDefaultPartition(TStringBuf partition) const {
        return IsDefaultPartition(GetPartitionId(partition));
    }

    ui64 TPartitionsSettings::GetDefaultPartitionId() const {
        return DefaultPartition_;
    }

    ui64 TPartitionsSettings::GetPartitionId(TStringBuf partition) const {
        if (partition.empty()) {
            return DefaultPartition_;
        }

        auto it = PartitionsByName_.find(partition);
        if (it == PartitionsByName_.end()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams) << "Unsupported partition '" << partition << "'";
        }

        return it->second;
    }

    const TString& TPartitionsSettings::GetPartitionName(ui64 id) const {
        auto it = PartitionsById_.find(id);
        Y_ENSURE(it != PartitionsById_.end(), "Invalid partition id '" << id << "'");
        return it->second;
    }
}
