#include "passport_wrapper.h"

#include "utils.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>

#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/datetime/base.h>

namespace NPassport::NBb {
    TPassportWrapper::TPassportWrapper(std::shared_ptr<NDbPool::TDbPool> db, const TString& consumer)
        : TotpCheckTimePath_(NUtils::CreateStr("/1/bundle/otp/set_check_time/?consumer=", consumer, "&uid="))
        , PassportDb_(std::move(db))
    {
    }

    TPassportWrapper::~TPassportWrapper() = default;

    bool TPassportWrapper::UpdateTotpCheckTime(const TString& uid, time_t checkTime) {
        TString errMsg;
        if (!PassportDb_->IsOk(&errMsg)) {
            return false; // ignore if pool not working
        }

        if (uid.empty()) {
            return false; // should never happen actually
        }

        TString request = NUtils::CreateStr(TotpCheckTimePath_, uid, "&totp_check_time=", checkTime);

        std::unique_ptr<NDbPool::TResult> result;
        try {
            NDbPool::TBlockingHandle sqlh(*PassportDb_);
            result = sqlh.Query(request);
        } catch (const std::exception& e) {
            TLog::Debug("Passport totp_check_time exception: %s", e.what());
            return false;
        }

        TString body;
        TString err;
        if (!NDbPool::NUtils::FetchBodyFromHttpResult(std::move(result), request, body, err)) {
            TLog::Debug("Passport request failed: %s", err.c_str());
            return false;
        }

        {
            rapidjson::Document doc;
            if (!NJson::TReader::DocumentAsObject(body, doc)) {
                TLog::Error("Passport totp_check_time failed: invalid json response: %s", body.c_str());
                return false;
            }

            try {
                TString status;
                if (!NJson::TReader::MemberAsString(doc, "status", status)) {
                    throw yexception() << "No 'status' in json";
                }

                if (status == "ok") {
                    return true;
                }

                const rapidjson::Value* errs = nullptr;
                if (NJson::TReader::MemberAsArray(doc, "errors", errs) &&
                    errs->Size() > 0 &&
                    (*errs)[0].IsString()) {
                    TLog::Debug("Passport totp_check_time returned error: %s", (*errs)[0].GetString());
                } else {
                    throw yexception() << "Malformed 'errors' in json";
                }

            } catch (const std::exception& e) {
                TLog::Error("Passport totp_check_time json parsing exception: %s", e.what());
                return false;
            }
        }

        return false;
    }
}
