#include "perimeter.h"

#include "exception.h"
#include "strings.h"
#include "utils.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <memory>

namespace NPassport::NBb {
    TPerimeterWrapper::TPerimeterWrapper(std::shared_ptr<NDbPool::TDbPool> db)
        : Path_("/auth")
        , PerimeterDb_(std::move(db))
    {
    }

    TPerimeterWrapper::~TPerimeterWrapper() = default;

    TPerimeterWrapper::EStatus
    TPerimeterWrapper::Login(const TString& login,
                             const TString& password,
                             const TString& ipaddr,
                             const TString& authType,
                             ENetworkKind ipKind,
                             bool isRobot,
                             TString& reqId,
                             TString& msg,
                             std::optional<TString>& allowedSecondSteps,
                             bool& passwordChangeRequired) const {
        TString postBody = NUtils::CreateStr(
            "login=",
            NUtils::Urlencode(login),
            "&password=",
            NUtils::Urlencode(password),
            "&ip=",
            ipaddr,
            "&auth_type=",
            NUtils::Urlencode(authType),
            "&is_ip_internal=",
            ipKind == ENetworkKind::Internal ? "yes" : "no",
            "&is_ip_robot=",
            ipKind == ENetworkKind::Robot ? "yes" : "no",
            "&is_robot=",
            isRobot ? "yes" : "no");

        std::unique_ptr<NDbPool::TResult> result;
        try {
            NDbPool::TQuery httpQuery(Path_);
            httpQuery.SetHttpBody(std::move(postBody));

            result = NDbPool::TBlockingHandle(*PerimeterDb_).Query(std::move(httpQuery));
        } catch (const NDbPool::TException& e) {
            throw TDbpoolError("Perimeter dbpool exception", e.what());
        }

        TString body;
        TString err;
        if (!NDbPool::NUtils::FetchBodyFromHttpResult(std::move(result), Path_, body, err)) {
            TLog::Debug("Perimeter request failed: %s", err.c_str());
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Internal error: unexpected Perimeter response";
        }

        return ParseResponse(body, reqId, msg, allowedSecondSteps, passwordChangeRequired);
    }

    TPerimeterWrapper::EStatus TPerimeterWrapper::ParseResponse(const TString& body, TString& reqId, TString& msg, std::optional<TString>& allowedSecondSteps, bool& passwordChangeRequired) {
        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(body, doc)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Internal error: Perimeter response is not JSON object";
        }

        TString status;
        if (!NJson::TReader::MemberAsString(doc, "status", status) ||
            !NJson::TReader::MemberAsString(doc, "description", msg) ||
            !NJson::TReader::MemberAsString(doc, "request_id", reqId)) {
            TLog::Debug("Unexpected json from Perimeter (base64url): %s", NUtils::Bin2base64url(body).c_str());
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Internal error: invalid Perimeter response contents";
        }

        passwordChangeRequired = false;
        if ((status == "ok" || status == "second_step_required")) {
            NJson::TReader::MemberAsBool(doc, "password_change_required", passwordChangeRequired);
        }

        if (status == "ok") {
            return LoginOk;
        }
        if (status == "second_step_required") {
            const rapidjson::Value* arr;
            if (!NJson::TReader::MemberAsArray(doc, "second_steps", arr)) {
                TLog::Debug("Unexpected json from Perimeter (base64url): %s", NUtils::Bin2base64url(body).c_str());
                throw TBlackboxError(TBlackboxError::EType::Unknown)
                    << "Internal error: invalid Perimeter response contents";
            }

            TString steps;
            for (size_t idx = 0; idx < arr->Size(); ++idx) {
                const auto& v = (*arr)[idx];
                if (!v.IsString()) {
                    TLog::Debug("Unexpected json from Perimeter (base64url): %s", NUtils::Bin2base64url(body).c_str());
                    throw TBlackboxError(TBlackboxError::EType::Unknown)
                        << "Internal error: invalid Perimeter response contents";
                }
                NUtils::AppendSeparated(steps, ',', TStringBuf(v.GetString(), v.GetStringLength()));
            }

            allowedSecondSteps = std::move(steps);
            return SecondStep;
        }
        if (status == "password_invalid") {
            return Rejected;
        }

        // this means that perimeter backends failed and perimeter didn't process the request
        // we return DB_EXCEPTION for passport to retry and hope it gets to another backend instance
        TLog::Debug("Perimeter request failed: with status '%s' and description '%s'", status.c_str(), msg.c_str());
        throw TBlackboxError(TBlackboxError::EType::DbException)
            << "Internal error: perimeter request failed with status " << status;
    }

    bool TPerimeterWrapper::DbOk(TString& msg) const {
        return PerimeterDb_->IsOk(&msg);
    }
}
