#include "rate_limiter.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/badauth/facade.h>
#include <passport/infra/daemons/blackbox/src/badauth/kolmogor.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NBb {
    TKolmogorBasedRateLimiter::TKolmogorBasedRateLimiter(NDbPool::TDbPool& kolmogor,
                                                         const TRateLimiterSettings& settings)
        : Kolmogor_(kolmogor)
        , Settings_(settings)
    {
    }

    bool TKolmogorBasedRateLimiter::CheckLimit(const TString& key, TRateLimiterSettings::ELimit limit) const {
        Y_ENSURE(limit < TRateLimiterSettings::ELimit::COUNT,
                 "got limit type: " << size_t(limit));

        TString comment;
        if (!Kolmogor_.IsOk(&comment)) {
            TLog::Debug() << "RateLimiter: kolmogor is not ok, so we assume that limit was not reached: "
                          << comment;
            return true;
        }

        try {
            const TString response = NDbPool::NUtils::GetHttpBody(
                Kolmogor_,
                BuildQuery(Settings_, key, Settings_.Limits[size_t(limit)]),
                "rate limiting");

            return ParseResponse(response, Settings_.Keyspace, key);
        } catch (const std::exception& e) {
            TLog::Debug() << "RateLimiter: failed to check rate limit: " << e.what();
        }

        return true;
    }

    NDbPool::TQuery TKolmogorBasedRateLimiter::BuildQuery(const TRateLimiterSettings& settings,
                                                          const TString& key,
                                                          ui64 limit) {
        TKolmogor::TSpaces querySpaces;

        querySpaces.push_back({
            .Name = settings.Keyspace,
            .Keys = {{key}},
            .IncIfLessThan = settings.Period * limit,
        });

        return TKolmogor::CreateIncQuery(TKolmogor::PrepareIncQuery(querySpaces));
    }

    bool TKolmogorBasedRateLimiter::ParseResponse(const TStringBuf response,
                                                  const TString& keyspace,
                                                  const TString& key) {
        rapidjson::Document doc;
        Y_ENSURE(NJson::TReader::DocumentAsObject(response, doc),
                 "invalid json: " << response);

        const rapidjson::Value* jsonSpace = nullptr;
        Y_ENSURE(NJson::TReader::MemberAsObject(doc, keyspace.c_str(), jsonSpace),
                 "failed to get space '" << keyspace << "' from response: " << response);

        const rapidjson::Value* jsonKey = nullptr;
        Y_ENSURE(NJson::TReader::MemberAsObject(*jsonSpace, key.c_str(), jsonKey),
                 "failed to get key '" << key << "' from response: " << response);

        bool wasIncremented = false;
        Y_ENSURE(NJson::TReader::MemberAsBool(*jsonKey, "was_incremented", wasIncremented),
                 "failed to get key 'was_incremented' from response: " << response);

        return wasIncremented;
    }

    bool TNoopRateLimiter::CheckLimit(const TString&, TRateLimiterSettings::ELimit) const {
        return true;
    }
}
