#include "sess_kill_wrapper.h"

#include "utils.h"

#include <passport/infra/daemons/blackbox/src/loggers/tskvlog.h>

#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/datetime/base.h>

namespace NPassport::NBb {
    TSessKillWrapper::TSessKillWrapper(const TTskvLog* tskv,
                                       std::shared_ptr<NDbPool::TDbPool> db)
        : TskvLogger_(tskv)
        , Path_("/1/session/alive/?uids=")
        , SessKillDb_(std::move(db))
    {
    }

    TSessKillWrapper::~TSessKillWrapper() = default;

    NDbPool::TNonBlockingHandle TSessKillWrapper::SendRequest(const TSessKillWrapper::TUidsList& uids, const TString& authid) {
        if (!SessKillDb_) {
            return NDbPool::TNonBlockingHandle();
        }

        TString errMsg;
        if (!SessKillDb_->IsOk(&errMsg)) {
            return NDbPool::TNonBlockingHandle(); // ignore if pool not working
        }

        TString uidsArg;
        uidsArg.reserve(10 * uids.size());

        for (const TString& uid : uids) {
            NUtils::AppendSeparated(uidsArg, ',', uid);
        }

        if (uidsArg.empty() || authid.empty()) {
            return NDbPool::TNonBlockingHandle();
        }

        TString request = NUtils::CreateStr(Path_, uidsArg, "&authid=", authid);

        try {
            NDbPool::TNonBlockingHandle sqlh(*SessKillDb_);
            sqlh.SendQuery(request);
            return sqlh;
        } catch (const std::exception& e) {
            return NDbPool::TNonBlockingHandle();
        }
    }

    bool TSessKillWrapper::IsKilled(NDbPool::TNonBlockingHandle sqlh,
                                    const TUidLoginDeltaMap& deltas,
                                    const TString& authid,
                                    time_t authidTime) {
        std::unique_ptr<NDbPool::TResult> result;
        try {
            result = sqlh.WaitResult();
        } catch (const std::exception& e) {
            return false;
        }

        TString body;
        TString err;
        if (!NDbPool::NUtils::FetchBodyFromHttpResult(std::move(result), Path_, body, err)) {
            TLog::Debug("SessKill request failed: %s", err.c_str());
            return false;
        }

        if (!body.empty()) { // no uids found
            std::vector<TString> entries = NUtils::ToVector(body, '\n');

            // check all logout entries from cassandra
            for (const TString& enrty : entries) {
                size_t colonPos = enrty.find(':');
                if (colonPos == TString::npos) {
                    continue;
                }

                TString uid = enrty.substr(0, colonPos);
                TString strlogout = enrty.substr(colonPos + 1);

                time_t logoutTime = TUtils::ToTime(strlogout);

                TUidLoginDeltaMap::const_iterator loginDelta = deltas.find(uid);
                if (loginDelta == deltas.end()) {
                    // this user is not in cookie anymore, it is safe to skip the check
                    continue;
                }

                // check if user relogined after logout, ignore user if he did
                if (logoutTime < authidTime + loginDelta->second) {
                    continue;
                }

                // Session is killed but we need to check allowed time lag
                time_t delta = time(nullptr) - logoutTime;

                if (TskvLogger_) {
                    TskvLogger_->LogDeadSession(uid, authid, delta);
                }

                if (delta >= 0) { // we have logout entry that is not in future
                    return true;
                }
            }
        }

        return false;
    }
}
