#include "session_utils.h"

#include "exception.h"
#include "strings.h"

#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionparser.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NBb {
    TSessionidProcessor::ESessionStatus
    TSessionUtils::Session2UserStatus(NAuth::TSession::EStatus status) {
        switch (status) {
            case NAuth::TSession::VALID:
                return TSessionidProcessor::ESessionStatus::VALID;
            case NAuth::TSession::NEED_RESET:
                return TSessionidProcessor::ESessionStatus::NEED_RESET;
            case NAuth::TSession::EXPIRED:
                return TSessionidProcessor::ESessionStatus::EXPIRED;
            case NAuth::TSession::NOAUTH:
                return TSessionidProcessor::ESessionStatus::NOAUTH;
            case NAuth::TSession::DISABLED:
                return TSessionidProcessor::ESessionStatus::DISABLED;
            case NAuth::TSession::CANT_CHECK:
            case NAuth::TSession::SIGN_BROKEN:
            case NAuth::TSession::NO_COOKIE:
                return TSessionidProcessor::ESessionStatus::INVALID;
        }
    }

    bool TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus status) {
        return status == TSessionidProcessor::ESessionStatus::VALID || status == TSessionidProcessor::ESessionStatus::NEED_RESET;
    }

    static const TString VALID = "VALID";
    static const TString NEED_RESET = "NEED_RESET";
    static const TString EXPIRED = "EXPIRED";
    static const TString NOAUTH = "NOAUTH";
    static const TString DISABLED = "DISABLED";
    static const TString INVALID = "INVALID";
    static const TString WRONG_GUARD = "WRONG_GUARD";

    const TString& TSessionUtils::StatusName(std::optional<TSessionProcessorBase::ESessionStatus> status) {
        if (!status) {
            return TStrings::UNKNOWN;
        }

        switch (*status) {
            case TSessionidProcessor::ESessionStatus::VALID:
                return VALID;
            case TSessionidProcessor::ESessionStatus::NEED_RESET:
                return NEED_RESET;
            case TSessionidProcessor::ESessionStatus::EXPIRED:
                return EXPIRED;
            case TSessionidProcessor::ESessionStatus::NOAUTH:
                return NOAUTH;
            case TSessionidProcessor::ESessionStatus::DISABLED:
                return DISABLED;
            case TSessionidProcessor::ESessionStatus::INVALID:
                return INVALID;
            case TSessionidProcessor::ESessionStatus::DEPRECATED_1:
            case TSessionidProcessor::ESessionStatus::DEPRECATED_2:
                return TStrings::UNKNOWN;
            case TSessionidProcessor::ESessionStatus::WRONG_GUARD:
                return WRONG_GUARD;
        }
    }

    // delete external user session from sess by idx, return true if user was deleted entirely
    bool TSessionUtils::DropExternalAuth(NAuth::TSession& sess, int idx) {
        if (idx < 0 || idx > (int)sess.UserCount()) {
            return false;
        }

        if (!sess.IsExternalAuth(idx)) {      // no external auth, do nothing
            sess.SetExtGlogouted(false, idx); // but clean up external glogout if it is here
            return false;
        }

        sess.SetExternalAuth(false, idx);
        sess.SetExtGlogouted(false, idx);

        if (!sess.IsInternalAuth(idx)) { // we had only external auth, so drop entire user from session(s)
            TString uid = sess.Uid(idx);

            sess.RemoveUser(uid);

            return true;
        }

        return false;
    }

    TString TSessionUtils::GetLoginId(const NAuth::TSession& sess) {
        if (!sess.LoginId().empty()) {
            return sess.LoginId();
        }

        if (!sess.AuthId().empty()) {
            return NUtils::CreateStr("s:", sess.AuthId());
        }

        return {};
    }
}
