#include "signer.h"

#include "exception.h"
#include "utils.h"

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/random.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/string.h>
#include <util/string/cast.h>

namespace NPassport::NBb {
    static const TString SIGN_KEYSPACE = "signsmth";
    static const TString SIGN_KEYSPACE_LONG = "signsmth_long";
    static const char SIGN_DELIMITER = '.';

    TSigner::TSigner(const NAuth::TSessionSigner& sessSigner,
                     std::unordered_set<TString>&& longSignSpaces)
        : LongSignSpaces_(std::move(longSignSpaces))
    {
        Keyring_ = sessSigner.GetRingByName(SIGN_KEYSPACE);

        if (!Keyring_) {
            throw TBlackboxError(TBlackboxError::EType::NoDataKeyspace) << "No keyspace " << SIGN_KEYSPACE;
        }

        KeyringLong_ = sessSigner.GetRingByName(SIGN_KEYSPACE_LONG);

        if (!KeyringLong_) {
            throw TBlackboxError(TBlackboxError::EType::NoDataKeyspace) << "No keyspace " << SIGN_KEYSPACE_LONG;
        }
    }

    static const TString OK = "OK";
    static const TString EXPIRED = "EXPIRED";
    static const TString NO_KEY = "NO_KEY";
    static const TString INVALID = "INVALID";

    const TString& TSigner::StatusStr(TSigner::EStatus status) {
        switch (status) {
            case EStatus::Ok:
                return OK;
            case EStatus::Expired:
                return EXPIRED;
            case EStatus::NoKey:
                return NO_KEY;
            case EStatus::Invalid:
                return INVALID;
        }
    }

    NAuth::TKeyRing* TSigner::GetRing(const TString& signSpace) const {
        return (LongSignSpaces_.find(signSpace) != LongSignSpaces_.end()) ? KeyringLong_ : Keyring_;
    }

    TString TSigner::Sign(const TString& value, const TString& signSpace, const TString& ttl, time_t now) const {
        TString signedValue;

        time_t delta = TUtils::ToTime(ttl);

        signedValue.assign(IntToString<10>(now + delta)).push_back(SIGN_DELIMITER);

        const NAuth::TKeyWithGamma key = GetRing(signSpace)->GetKeyForSign();

        if (key.Error()) {
            throw TBlackboxError(TBlackboxError::EType::KeyspaceEmpty) << "No key for sign.";
        }

        signedValue.append(key.Id().AsString()).push_back(SIGN_DELIMITER);

        // remember original size and append sign_space
        size_t signedValueSize = signedValue.size();

        signedValue.append(signSpace);

        TString cryptoMsg;
        NUtils::TCrypto::TCiphertext ctext;

        if (!NUtils::TCrypto::EncryptGcm(key.Body(), value, ctext, signedValue, &cryptoMsg)) {
            TLog::Error("Failed to encrypt and sign value: %s", cryptoMsg.c_str());
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "Failed to encrypt and sign value";
        }

        signedValue.resize(signedValueSize);

        // iv
        signedValue.append(NUtils::Bin2base64url(ctext.Iv)).push_back(SIGN_DELIMITER);

        // data
        signedValue.append(NUtils::Bin2base64url(ctext.Text)).push_back(SIGN_DELIMITER);

        // tag
        signedValue.append(NUtils::Bin2base64url(ctext.Tag));

        return signedValue;
    }

    TSigner::TCheckResult TSigner::CheckSign(const TString& signedValue, const TString& signSpace, time_t now) const {
        TStringBuf signedView = signedValue;

        const TStringBuf expireView = signedView.NextTok(SIGN_DELIMITER);
        const TStringBuf keyIdView = signedView.NextTok(SIGN_DELIMITER);

        TString aaData = signedValue.substr(0, signedValue.size() - signedView.size());
        aaData.append(signSpace);

        const TStringBuf ivView = signedView.NextTok(SIGN_DELIMITER);
        const TStringBuf dataView = signedView.NextTok(SIGN_DELIMITER);
        const TStringBuf tagView = signedView.NextTok(SIGN_DELIMITER);

        if (!keyIdView || !ivView || !dataView || !tagView) {
            return TCheckResult(EStatus::Invalid, keyIdView);
        }

        time_t expires;
        if (!TryIntFromString<10>(expireView, expires)) {
            return TCheckResult(EStatus::Invalid, keyIdView);
        }

        if (now > expires) {
            return TCheckResult(EStatus::Expired, keyIdView, expires);
        }

        const NAuth::TKeyWithGamma key = GetRing(signSpace)->GetKeyById(keyIdView);
        if (key.Error()) {
            return TCheckResult(EStatus::NoKey, keyIdView, expires);
        }

        TString cryptoMsg;
        NUtils::TCrypto::TCiphertext ctext;

        ctext.Iv.assign(NUtils::Base64url2bin(ivView));
        ctext.Text.assign(NUtils::Base64url2bin(dataView));
        ctext.Tag.assign(NUtils::Base64url2bin(tagView));

        TString value;
        if (!NUtils::TCrypto::DecryptGcm(key.Body(), ctext, value, aaData, &cryptoMsg)) {
            TLog::Debug("Failed to decrypt and check sign: %s", cryptoMsg.c_str());
            return TCheckResult(EStatus::Invalid, keyIdView, expires);
        }

        return TCheckResult(EStatus::Ok, keyIdView, expires, std::move(value));
    }

}

// for tests
template <>
void Out<NPassport::NBb::TSigner::TCheckResult>(IOutputStream& o,
                                                const NPassport::NBb::TSigner::TCheckResult& val) {
    o << "status==" << val.Status << ";";
    o << "expires==" << val.Expires << ";";
    o << "keyId==" << val.KeyId << ";";
    o << "value==" << val.Value << ";";
}
